/**
 * @file libDynamoMiddlewareARM.h
 * @brief ARM Middleware Libraries.  
 *
 * Dynamo Middleware libraries, provide a easy to use
 * interface for querying system information, utilizing shared memory, and communication
 * framework
 * 
 * Copyright (C) 2004
 * Dynamo Middleware Group
 *
 * version 0.5
 *
 * Authors: Michael Philpott, Shivajit Mohapatra 
 */

#ifndef _DYNMIDDLEWARE_ARM_H_
#define _DYNMIDDLEWARE_ARM_H_

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h> 
#include <string.h> 
#include <stdlib.h>
#include <sys/uio.h>
#include <memory.h>
#include <pthread.h> 
#include <unistd.h>

//
// Structures

/**
 * @struct dynamo_backlight
 * @brief Contains Backlight information.
 */
struct dynamo_backlight 
{
	int level;			///< Level from 0 - 255
	int status;			///< If the backlight is currently on or off 1|0
};

/**
 * @struct dynamo_battery
 * @brief Contains Battery information.
 */
struct dynamo_battery 
{
	int ac_status;		///< If the device is plugged
	int level;			///< The power level of the battery 1 - 100
	int min_remaining;	///< minutes until there is no power
};

/**
 * @struct dynamo_cpu
 * @brief Contains CPU Information
 */
struct dynamo_cpu 
{
	char model[40];		///< The model name of the CPU
	float load;			///< load value of the cpu 0 - 1.0 average of CPU over the past min
	int total_process;	///< total processes active on the device
	int active_process;	///< currently active processes on the server
};

/**
 * @struct dynamo_memory
 * @brief Contains Memory Information.
 */
struct dynamo_memory 
{
	int total_mem;		///< Total memory available to the device
	int free_mem;		///< Free memory on the device
	int total_swap;		///< Total swap available on the device
	int free_swap;		///< Free swap on the device
};

/**
 * @struct dynamo_network
 * @brief Contains Network Information
 */
struct dynamo_network 
{
	int connected;		///< If the device is connected to the network
	int wireless;		///< If the device is connected to a wireless network
	int link;			///< The link strength of the wireless signal
	int signal;			///< The signal strength of the wireless network
};

//
// Status Libraries

/**
 * @brief Gets the backlight level
 * @returns the backlight level 0 - 255 or -1 if an error
 */
extern int get_backlight_level();

/**
 * @brief Sets the backlight to level
 * Allows the caller to set the backlight intensity
 * @param lvl integer 0-255 with 255 being the highest level
 */
extern int set_backlight_level(int lvl);

/**
 * @brief Gets the backlights status [on|off]
 * @return 0 (off) 1 (on) or -1 (error reading backlight)
 */
extern int get_backlight_status();

/**
 * @brief Sets the backlight [on 1|off 0]
 * Allows the caller to set the status of the backlight 1 for on 0 for off
 * @param status 0 or 1
 */
extern int set_backlight_status(int status);

/**
 * @brief Checks if the Device is plugged in
 * @return 0 or 1
 */
extern int get_battery_ac_status();

/**
 * @brief Gets devices battery level
 * @return the systems power level in percentage 0(empty) - 100(full)
 */
extern int get_battery_power_level();

/**
 * @brief Gets the amount of power in minutes remaining
 * @return the amount of minutes that the system will have power for
 * Note: need to implament to return -1 if its pluged in
 */
extern int get_battery_min_remaining();

/**
 * @brief Gets the CPU model string
 * @return a char* of the processors model information this will have to be freed by the caller
 */ 
extern char* get_cpu_model();

/**
 * @brief returns the average load of the device
 * @return the average load of the cpu over the past minute
 */
extern float get_cpu_load();

/**
 * @brief Gets the amount of total processes on the device
 * @return the amount of processes that have been started (active or passive)
 */
extern int get_cpu_total_process();

/**
 * @brief Gets the current number of active processes on the device
 * @return the amount of active tasks on the device
 */
extern int get_cpu_running_process();

/**
 * @brief Gets the total amount of memory
 * @return the full amount of memory (ie your system has 512 megs of ram)
 */
extern int get_mem_total();

/**
 * @brief Gets the amount of free memory
 * @return the amount of free memory available on your system
 */
extern int get_mem_free();

/**
 * @brief Gets the total amount of swap
 * @return the total swap amount avaiable to your system
 */
extern int get_mem_total_swap();

/**
 * @brief Gets the amount of free swap
 * @return the free swap amount of the system
 */
extern int get_mem_free_swap();

/**
 * @brief Gets the current network connection status
 * @return 1 if the device is connected or 0 if no network is present
 */  
extern int get_net_is_connected();

/**
 * @brief Gets the current wireless network connection status
 * @return 1 if the device has a wireless connection or 0 if no wireless connection
 *
 * need to test when the card is in but not connected
 */ 
extern int get_net_is_wireless();

/**
 * @brief Gets the link quality of the wireless network
 * @return the link quality of the wireless network or -1 if nothing is present
 */  
extern int get_net_link_quality();

/**
 * @brief Gets the signal strength of the wireless network
 * @return the signal quality of the wireless network or -1 nothing is present
 */  
extern int get_net_sig_strength();

/**
 * @brief Collects the devices backlight information
 * @see get_backlight_level()
 * @see get_backlight_status()
 */
extern int get_backlight_info(struct dynamo_backlight* backlight);

/**
 * @brief Collects the devices battery information
 * @see get_battery_ac_status()
 * @see get_battery_power_level()
 * @see get_battery_min_remaining()
 */
extern int get_battery_info(struct dynamo_battery* battery);

/**
 * @brief Collects the devices cpu information
 * @see get_cpu_model()
 * @see get_cpu_load()
 * @see get_cpu_total_process()
 * @see get_cpu_running_process()
 */
extern int get_cpu_info(struct dynamo_cpu* cpu);

/**
 * @brief Collects the devices memory information
 * @see get_mem_total()
 * @see get_mem_free()
 * @see get_mem_total_swap()
 * @see get_mem_free_swap()
 */
extern int get_memory_info(struct dynamo_memory* memory);

/** 
 * @brief Collects the devices network information
 * @see get_net_is_connected()
 * @see get_net_is_wireless()
 * @see get_net_link_quality()
 * @see get_net_sig_strength()
 */
extern int get_network_info(struct dynamo_network* network);

//
// Shared Memory Operations

/**
 * @brief creates a chunk of shared memory
 * @param key the Key to the shared memory
 * @param size the size that the shaed memory should be
 * @return The ID to the shared memory
 */
int createSharedMemory(int key, int size);

/**
 * @brief reads from the shared memory into a structuer
 * @param shmid the ID of the shared memory
 * @param *data the structure that will be read
 * @param size the size of the item that will be read
 */
int readSharedMemory(int shmid, void* data, int size);

/**
 * @brief writes data to the shared memory
 * @param shmid the ID of the shared memory to write to
 * @param *data the information to write inot the shared memory
 * @param size the size of the item that will be written
 */
int writeSharedMemory(int shmid, void* data, int size);

/**
 * @brief releases the memory with id of shmid
 * @param shmid the id of the shared memory that is being freed
 */
int freeSharedMemory(int shmid);

//
// Semaphore Operations

/**
 * @brief translate semaphore key to an ID
 * @param key the Key that the semaphore should be created with 
 * @return the ID to the shared memory
 */
int createSemaphore(int key);

/**
 * @brief free specified Semaphore
 * @param sem_id the ID of the semaphore to free
 */
int freeSemaphore(int sem_id);

/**
 * @brief acquire semaphore 
 * @param sid the ID of the semaphore to obtain
 */
void P(int sid);

/**
 * @brief release semaphore
 * @param sid the ID of the semaphore to release
 */
void V(int sid);

/**
 * @brief Sends a Datagram to a remote computer
 * @param *targetIP character array of target computers IP Address
 * @param localPort local port of the UPD socket
 * @param targetPort the remote hosts port
 * @param *buf the information to be sent
 * @param bufsiz the size of the item to be sent
 */
int SendDatagram(char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

/**
 * @brief creates a UPD socket
 * @param *localIP character array of the local IP address
 * @param localPort a port number
 */
int CreateUDPSocket(char *localIP, int localPort);

/**
 * @brief Recives a datagram a remote computer
 * @param sockfd pointer to the UDP socket
 * @param *buf address of the structure to fill
 * @param size size of the structure to fill
 */
int UDP_Recv(int sockfd, void *buf, int size);

/**
 * @brief Sends a Datagram to a remote computer
 * @param sockfd pointer to the upd socket
 * @param *targetIP character array of target computers IP Address
 * @param targetPort the remote hosts port
 * @param localPort local port of the UPD socket
 * @param *buf the information to be sent
 * @param bufsiz the size of the item to be sent
 */
int UDP_Send(int sockfd, char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

#endif

