#ifndef __PAAPI_SCHED_H
#define __PAAPI_SCHED_H

#include "linux/sched.h"
#include "linux/power/paapi-user.h"
#include <linux/time.h>

/*--------------------------------------------------------------------*/
/* Types */
#define GET_PRIORITY(task)              (task->rt_priority)
#define GET_TABLE_INSTANCE_INDEX(task)  (task->instance_index)

/*--------------------------------------------------------------------*/
/* Constants */

#define MAX_TASKS_TYPES     10
#define MAX_TASKS_INSTANCES 60

#define HISTORY_WINDOW_SIZE 10
#define NUMBER_OF_WCET_SLOTS 10

#define PAAPI_TASK_FLAG 0x1
#define PAAPI_TASK_STARTED 0x2
#define PAAPI_TASK_ALREADYSCHED 0x4

#define PAAPI_SCHED_POLICY SCHED_FIFO
#define PAAPI_CURRENT_TIME (jiffies)

#define PAAPI_DVS_DEBUG

#define DIV_CEIL(a,b) ( (a / b) + (a % b > 0 ? 1 : 0) )

#define TRACE_BUFFER_LENGTH    (8*1024)

#define APP_SLEEP           0
#define SCHED_START         1
#define SCHED_PREEMPT       2
#define SCHED_PREEMPT_DONE  3
#define SCHED_PREEMPT_DEAD  4
#define LEAVING             5
#define WAKE_UP             6
#define WAKE_UP_READJUST    7
#define WAKE_UP_DEAD        8
#define CHANGE_FREQ_START   9
#define CHANGE_FREQ_DONE   10 
#define STATIC_FACTOR      11 
#define DYNAMIC_FACTOR     12 
#define CHANGE_PARAMETERS  13

#define OVERHEAD 20

struct trace_info {
  char type;
  unsigned task;
  unsigned time;
  unsigned exec_time;
  unsigned total_time;
  unsigned deadline;
  unsigned next_arrival;
  unsigned speed;
  unsigned factor;
};

/*--------------------------------------------------------------------*/
/* Exported variables */
extern struct thread_type_info_t *type_table[MAX_TASKS_TYPES];
extern unsigned num_types;

/* table of task instances */
extern struct thread_instance_info_t *instance_table[MAX_TASKS_INSTANCES];
extern unsigned num_instances;

extern unsigned dvs_policy;
extern unsigned admitted;

extern struct trace_info tracebuffer[TRACE_BUFFER_LENGTH];
extern unsigned trace_buffer_index;

/*--------------------------------------------------------------------*/
/* Types and data structures */


/* static information about a thread type */
struct thread_type_info_t {

  /* type index */
  unsigned short type_index;

  unsigned ref_counter;

  /* timing information */
  struct thread_type_t info;

  /* pointer to the type related run-time info */
  struct thread_type_rt_info_t *thread_type_rt;
};


/* thread type related runtime info */
struct thread_type_rt_info_t {

  /* history of previous executions */
  unsigned history[HISTORY_WINDOW_SIZE];

  /* number of previous executions */
  unsigned num_prev_executions;

  /* number of previous executions */
  unsigned num_last_prev_executions;

  /* pointers to walk over the history vector */
  unsigned char pos;

  /* wcet slots or probability distribution for the wcet */
  unsigned slotted_wcet[NUMBER_OF_WCET_SLOTS];

  /* number of deadlines missed */
  unsigned num_missed_deadlines; 

  /* number of deadlines missed since the adaptive factor was updated */
  unsigned num_last_missed_deadlines;
};

/* thread instance related info */
struct thread_instance_info_t {

  /* information about the thread type */
  struct thread_type_info_t *type_info;

  /* instance number */
  unsigned instance_number;

  /* thread arrival time. This is absolute time */
  unsigned arrival_time; 

  /* this should be arrival_time + period */
  unsigned next_arrival; 

  /* static real time priority */
  unsigned priority;

  /* last time this task was reescheduled to resume execution. This is 
     absolute time */
  unsigned last_resched;

  /* deadline of the current execution. This is absolute time */
  unsigned deadline;
  
  /* number of time units executed so far for this instance. This represents 
     the number of time units this task has executed without counting the
     time units wherein the task was preempted */
  unsigned total_time_so_far; 

  /* execution time predicted by the task/app */
  unsigned time_remaining; 

  /* time stamp of the time when the prediction was computed by the app. 
     This is absolute time */
  unsigned timestamp_tr;

  /* execution time predicted by the OS */
  unsigned os_time_remaining; 

  /* time stamp of the time when the prediction was by carried out by the 
     OS. This is absolute time */
  unsigned timestamp_os_tr; 

  /* string identifier */
  char *name;

  /* data passed in as parameter */
  void *entry_data;

  /* POSIX thread representation */
  void *thread;

  /* thread attributes */
  void *thread_data;

  /* whether the task has been preempted or not */
  unsigned char preempted;

  /* whether the task has finished its execution */
  unsigned char done;

  /* dynamic scale down factor. Represents the percentage of the 
     maximum voltage to scale the processor to */
  unsigned dscale_factor;

  /* static scale down factor. Represents the percentage of the 
     maximum voltage to scale the processor to */
  unsigned sscale_factor;

  /* this factor is sensitive to the number of deadlines missed in the
     previous HISTORY_WINDOW_SIZE executions */
  unsigned adaptive_factor;

  char missed_deadline;

  unsigned compute_dfactor;
  unsigned dfactor_counter;
};

struct instance_list_t {
  struct thread_instance_info_t *th_info;
  struct instance_list_t *next;
};

/*--------------------------------------------------------------------*/
/* Exported functions */

extern void 
log_trace(char type, unsigned task, unsigned time, unsigned param0,
   unsigned param1);

extern void
print_trace(void);

extern void 
compute_dynamic_factor(struct thread_instance_info_t *);

extern void 
compute_adaptive_factor(struct thread_instance_info_t *);

extern void 
predict_execution_time(struct thread_instance_info_t *);

extern void
wake_up_tasks(void); /* paapi.c */

#endif
