#include <linux/kernel.h>
#include <linux/unistd.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/errno.h>
#include <linux/cpufreq.h>

#include "asm/uaccess.h" 

#include "include/paapi.h"
#include "linux/power/paapi-user.h"

#define MIN(a,b) ((a < b) ? a : b)
#define MAX(a,b) ((a > b) ? a : b)

static DECLARE_WAIT_QUEUE_HEAD(wait_for_admission_control);
static DECLARE_WAIT_QUEUE_HEAD(wait_for_next_period);
static spinlock_t wait_for_adm_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t wait_for_np_lock = SPIN_LOCK_UNLOCKED;

static struct instance_list_t *ordered_instance_list = (void *)0;
static struct thread_instance_info_t **ordered_instances;
static unsigned *exec_times;
static const unsigned max_speed = 206400;
static unsigned new_speed;
static unsigned static_factor = 100;
static unsigned dynamic_factor = 100;
static unsigned overall_factor = 100;

struct speed_info {
  unsigned speed;
  unsigned normalized;
};

#define NUM_SPEEDS 3
struct speed_info speeds[NUM_SPEEDS] = {
  { 206400, 206400*100 / (float)206400 },
  { 191700, 191700*100 / (float)206400 },
  { 176900, 176900*100 / (float)206400 }//,
//  { 162200, 162200*100 / (float)206400 },
//  { 147500, 147500*100 / (float)206400 },
  /*  { 132700, 132700*100 / (float)206400 },*/
//  { 118000, 118000*100 / (float)206400 },
//  { 103200, 103200*100 / (float)206400 },
//  {  88500,  88500*100 / (float)206400 },
//  {  73700,  73700*100 / (float)206400 }, 
//  {  59000,  59000*100 / (float)206400 }  
};

unsigned adjust_factor(unsigned norm_speed, unsigned *speed)
{
  int i;
  for ( i = 0; i < NUM_SPEEDS; i++ ) {
    if ( norm_speed > speeds[i].normalized ) {
      *speed = speeds[i-1].speed;
      return speeds[i-1].normalized;
    }
  }
  *speed = speeds[NUM_SPEEDS-1].speed;
  return speeds[NUM_SPEEDS-1].normalized;
}
   
void
ordered_list_insert(struct thread_instance_info_t *new_task)
{
  struct instance_list_t *it, *last = NULL;
  struct instance_list_t *new_entry = 
    kmalloc(sizeof(struct instance_list_t), GFP_KERNEL);
  
  new_entry->th_info = new_task;
  new_entry->next = (void *)0;
  if ( ordered_instance_list != (void *)0) {
    for ( it = ordered_instance_list; it != (void *)0; it = it->next) {
      //printk("comparing %d with %d\n",
      //	     it->th_info->type_info->info.period,
      //     new_entry->th_info->type_info->info.period);
	     
      if (new_entry->th_info->type_info->info.period <
	  it->th_info->type_info->info.period ) {
	new_entry->next = it;
	//it->next = new_entry;
	if ( it == ordered_instance_list ) {
	  ordered_instance_list = new_entry;
	}
	break;
      }
      last = it;
    }
    if ( it == (void *)0 ) {
      last->next = new_entry;
    }
  }
  else {
    ordered_instance_list = new_entry;
  }
  //printk("%d inserted in the list\n", 
  // new_entry->th_info->type_info->info.period);
}

void 
ordered_list_delete(unsigned index)
{
  struct instance_list_t *it, *prev = (void *)0;
  
  if ( ordered_instance_list != (void *)0) {
    for ( it = ordered_instance_list; it != (void *)0; it = it->next) {
      if ( it->th_info->instance_number == index ) {
	if ( prev == (void *)0 ) {
	  ordered_instance_list = it->next;
	}
	else {
	  prev->next = it->next;
	}
	//printk("%d deleting from the list\n", 
	//       it->th_info->type_info->info.period);

	kfree(it);
	break;
      }
      prev = it;
    }
  }
}

void 
ordered_list_print(void)
{
  struct instance_list_t *it;
  
  //printk("ordered_list_print:\n");
  if ( ordered_instance_list != (void *)0) {
    for ( it = ordered_instance_list; it != (void *)0; it = it->next) {
      //printk("task_%d: %d ", it->th_info->instance_number,
      //     it->th_info->type_info->info.period);
    }
  }
  //printk("\n");
}

// Get the response time given the index of the (void *)0 and the scheduling point.
// The index relates to the vector 'ordered_instances', which is sorted 
// by the periods ascending order
unsigned response(
  int i, 
  int t
  )
{
  int k;
  unsigned  r = 0;

  //printk("response for sp: %d\n", t);
  for ( k = 0; k <= i; k++ ) {
    r += exec_times[k] * 
      DIV_CEIL(t,ordered_instances[k]->type_info->info.period);
      //printk("%d * ( %d / %d )\n", exec_times[k], t, 
	  //ordered_instances[k]->type_info->period);
  }
  //printk("response: %d\n", r);
  return r;
}

// Rate monotonic analysis function. Check schedulability of the (void *)0 set
// and computes the static slow down factors
void rm_schedulability_analysis(
  int max_iterations, 
  int calc_dynamic_factor,
  int change_frequency
  )
{
  unsigned i, j, k, 
             max_index = 0,  // index of the maximum factor among the
                             // minimum ones
             init_index = 0; // index of the inital (void *)0 to be 
                             // analyzed. This index changes after 
                             // each iteration of the 'do while' loop
  unsigned *factors;        // intermediate factors
  unsigned *final_factors;  // final factors
  unsigned n;
  unsigned fixed_load, new_load;
  unsigned num_iterations = 0;
  unsigned average_exec_time;
  unsigned history_size;
  int error = 0, instance_index;
  struct instance_list_t *it;

  //printk("rma: %d\n", num_instances);
  instance_index = 0;
  ordered_instances = (struct thread_instance_info_t **)
    kmalloc(sizeof(struct thread_instance_info_t *)*num_instances, GFP_KERNEL);
  exec_times = (unsigned *)kmalloc(sizeof(unsigned)*num_instances,GFP_KERNEL); 
  factors = (unsigned *)kmalloc(sizeof(unsigned)*num_instances,GFP_KERNEL); 
  final_factors = (unsigned *)kmalloc(sizeof(unsigned)*num_instances,GFP_KERNEL);

  i = 0;
  if ( ordered_instance_list != (void *)0) {
    for ( it = ordered_instance_list; it != (void *)0; it = it->next) {
      //printk("APP period: %d\n", it->th_info->type_info->info.period);
      ordered_instances[i] = it->th_info;
      ++i;
    }
  }

  /* initialize temporary execution time vector */
  for ( i = 0; i < num_instances; i++ ) {
    //printk("period: %d\n", ordered_instances[i]->type_info->info.period);
    if ( !calc_dynamic_factor ) {
      exec_times[i] = ordered_instances[i]->type_info->info.wcet;
    }
    else {
      average_exec_time = 0;
      history_size = 
        ordered_instances[i]->type_info->thread_type_rt->num_prev_executions;
      history_size = (history_size < HISTORY_WINDOW_SIZE) ? history_size :
        HISTORY_WINDOW_SIZE;
      for ( j = 0; j < history_size; j++ ) {
        average_exec_time += 
          ordered_instances[i]->type_info->thread_type_rt->history[j];
      }
      if ( history_size > 0 )
        average_exec_time = (average_exec_time * 100) / history_size;
      else 
        average_exec_time = (ordered_instances[i]->type_info->info.wcet
          *(10000/static_factor)) / 100;

      exec_times[i] = average_exec_time;
    }
  }

  /* initialize factors vector */
  for ( i = 0; i < num_instances; i++ )
    factors[i] = final_factors[i] = 100;

  // do until the scale factor of all (void *)0s get computed or until one
  // finds out that the (void *)0 set is not schedulable
  do {
    n = 0;

    /* scale the execution times according ot the last factors computed */
    for ( i = 0; i < init_index; i++ )
      exec_times[i] = ( ordered_instances[i]->type_info->info.wcet * 
        (10000/final_factors[i]) ) / 100;

    /* inialize factors with a 'big' number so that it's easy to get 
       the minimum */
    for ( i = init_index; i < num_instances; i++ )
      factors[i] = 100000;

    /* for each (void *)0 */
    for ( i = init_index; i < num_instances; i++ ) {
      //printk("Task %d\n", instance_table[i]->instance_number);

      /* for each scheduling point one gets the response time and stores it
         depending whether it is the mininum or not */
      for ( j = 0; j < i + 1; j++ ) {
	k = 1;
	while ( k <= (ordered_instances[i]->type_info->info.period / 
                      ordered_instances[j]->type_info->info.period ) ) {
	  int sp = k * ordered_instances[j]->type_info->info.period;
	  //printk("%d", sp);
	  //printk("(%d) ", response(i, sp));
	  if ( init_index > 0 ) {
	    fixed_load = response(init_index - 1, sp);
	  }
	  else {
	    fixed_load = 0;
	  }
	  new_load = response(i, sp) - fixed_load;
	  //printk("fixed_load: %d, new_load: %d\n", fixed_load, new_load);
	  factors[i] = MIN(factors[i], ((new_load*100) / ( sp - fixed_load )) );
	  //printk("factor for sp[%d]: %d\n", sp,  factors[i]);
	  k++;
	}
      } 
      //printk("\n");
    } 

    /* select the maximum factor computed. If one of them is greater than 1
       the (void *)0 set is not schedulable and variable 'error' is set */
    for ( i = init_index; i < num_instances; i++ ) {
      //printk("%d: %d\n", i, factors[i]);
      if ( factors[i] > 100 ) {
        error = 1;
	break;
      }
      if ( factors[i] >= n ) {
	max_index = i;
      }
      n = MAX(n, factors[i]);
    }

    if (error) {
      //printk("Not schedulable\n");
      n = 100;
      break;
    }

    // update the final factors
    for ( i = init_index; i <= max_index; i++ )
      final_factors[i] = factors[i] = n;

    //printk("Static scale down factor (from task0%d to task0%d) %d\n", 
    //             init_index, max_index, n);

    // static scale down factor up to (void *)0 'max_index + 1' has already been
    // calculated
    num_iterations++;
    init_index = max_index + 1;
    //printk("max index: %d\n", max_index);
  } while (init_index < num_instances && num_iterations < max_iterations);

  for ( i = init_index; i < num_instances; i++ ) {
    final_factors[i] = n;
  }

  // assign the final factor to the respective instances structures. This for
  // asssigns the threads priorities as well according to the RMS criterion
  //printk("Final factors:\n");
  
  if ( !calc_dynamic_factor ) {
    overall_factor = static_factor = adjust_factor(n, &new_speed);
    log_trace(STATIC_FACTOR, 0XFFFF, PAAPI_CURRENT_TIME, static_factor, n);
  }
  else {
    dynamic_factor = n;
    log_trace(DYNAMIC_FACTOR, 0XFFFF, PAAPI_CURRENT_TIME, dynamic_factor, n);
    overall_factor = adjust_factor((static_factor * dynamic_factor) / 100, 
      &new_speed);
  }
  ordered_instances[num_instances-1]->compute_dfactor = 1;
  /* how often to compute the dynamic factor */
  ordered_instances[num_instances-1]->dfactor_counter = (OVERHEAD)*(100) / 
    (ordered_instances[num_instances-1]->type_info->info.period * 2);

  for ( i = 0; i < num_instances; i++ ) {
    if ( !calc_dynamic_factor ) {
      ordered_instances[i]->sscale_factor = final_factors[i];
      ordered_instances[i]->priority = (MAX_PRIORITY - 1) - i;
      //printk("%d: sf: %d, prio: %d\n", ordered_instances[i]->instance_number, 
      //             (unsigned)(ordered_instances[i]->sscale_factor),
      //             (unsigned)(ordered_instances[i]->priority));
    }
    else {
      ordered_instances[i]->dscale_factor = final_factors[i];
    }
  }
  kfree(exec_times);
  kfree(factors);
  kfree(final_factors);
  kfree(ordered_instances);

  if ( change_frequency ) {
    //log_trace(CHANGE_FREQ_START, 0XFFFF, PAAPI_CURRENT_TIME, new_speed, 0);
    cpufreq_set(new_speed);
    //log_trace(CHANGE_FREQ_DONE, 0XFFFF, PAAPI_CURRENT_TIME, new_speed, 0);
  }
}

void
wake_up_tasks()
{
  wait_queue_t *curr;
  struct list_head *tmp;
  unsigned index;

  spin_lock(&wait_for_np_lock);
  if (!list_empty(&(wait_for_next_period.task_list))) { 
    list_for_each(tmp, &(wait_for_next_period.task_list)) {
      curr = list_entry(tmp, wait_queue_t, task_list);
      index = curr->task->instance_index;
        if ( curr->task->paapi_task & PAAPI_TASK_FLAG ) {
 	if ( instance_table[index]->next_arrival == PAAPI_CURRENT_TIME || 
	     instance_table[index]->missed_deadline ) {
        wake_up_process(curr->task);
	if ( instance_table[index]->arrival_time != 0 )
	  instance_table[index]->arrival_time = PAAPI_CURRENT_TIME;
	if ( instance_table[index]->missed_deadline ) {
	  instance_table[index]->missed_deadline = 0;
	  //printk("%d: deadline (%d) was missed, waking up task_%d\n", 
          //  PAAPI_CURRENT_TIME, instance_table[index]->deadline, index);
          log_trace(WAKE_UP_DEAD, current->instance_index,
            PAAPI_CURRENT_TIME,instance_table[index]->deadline,0);
	  if ( PAAPI_CURRENT_TIME > instance_table[index]->deadline ) {
	    //printk("%d: readjusting task_%d, deadline: %d\n", 
            //  PAAPI_CURRENT_TIME, index, instance_table[index]->deadline);
            log_trace(WAKE_UP_READJUST, current->instance_index,
              PAAPI_CURRENT_TIME,0,0);
	    instance_table[index]->next_arrival = PAAPI_CURRENT_TIME;
	    instance_table[index]->deadline = 
              instance_table[index]->next_arrival +
	      instance_table[index]->type_info->info.deadline;
	  }
	}
	//printk("%d: waking up task_%d: %d : %d\n", PAAPI_CURRENT_TIME, index,
        //  instance_table[index]->next_arrival, PAAPI_CURRENT_TIME);
        log_trace(WAKE_UP, current->instance_index, PAAPI_CURRENT_TIME, 0, 0);
      }
      }
    }
  }
  spin_unlock(&wait_for_np_lock);
}

/* system calls */

void
sys_paapi_create_type(struct thread_type_t* th_type_info,
		      struct thread_type_handle_t *type_handle)
{
  int i;
  struct thread_type_handle_t local_handle;

  /* create a new entry in the type table */
  for ( i = 0; i < MAX_TASKS_TYPES; i++ ) {
    if ( type_table[i] == (void *)0 ) {

      /* allocate an entry in the type table within kernel space */
      type_table[i] = (struct thread_type_info_t *)
	kmalloc(sizeof(struct thread_type_info_t), GFP_KERNEL);

      type_table[i]->type_index = i;
      type_table[i]->thread_type_rt = (void *)0;
      type_table[i]->ref_counter = 0;

      /* Copy the data into the kernel space */
      copy_from_user(&type_table[i]->info, th_type_info,
		     sizeof(struct thread_type_t));  
      break;
    }
  }

  //printk("New type table entry (index: %d):\n", i);
  //printk("  wcet: %d\n", type_table[i]->info.wcet);
  //printk("  deadline: %d\n", type_table[i]->info.deadline);
  //printk("  period: %d\n", type_table[i]->info.period);
  //printk("  hardness: %d\n", type_table[i]->info.hard);

  if ( i < MAX_TASKS_TYPES ) {
    /* return the handle to the new type */
    local_handle.type_info_index = i;
    num_types++;
  }
  else {
    local_handle.type_info_index = -1;
  }

  /* copy handle back to user space */
  copy_to_user(type_handle, &local_handle, 
	       sizeof(struct thread_type_handle_t));
} 


void 
sys_paapi_create_instance(struct thread_type_handle_t *type_handle,
			  struct thread_instance_handle_t *instance_handle,
			  int instance_index)
{
  int i;
  int type_index;
  struct thread_instance_handle_t local_handle;

  type_index = type_handle->type_info_index;

  // if ( instance_index == (-1) ) {
  for ( i = 0; i < MAX_TASKS_INSTANCES; i++ ) {
    if ( instance_table[i] == (void *)0 ) {
      instance_table[i] = (struct thread_instance_info_t *)
	kmalloc(sizeof(struct thread_instance_info_t), GFP_KERNEL );
      instance_index = i;
      break;
    }
  }
  //printk("Type table index: %d\n", type_index);
  //printk("New instance table entry index: %d\n", instance_index);
  
  /* if a proper table entry was found, we continue */
  if ( instance_index < MAX_TASKS_INSTANCES ) {
    /* copy information to instance structure. This is used when the 
       thread is killed and needs to be recreated */
    instance_table[instance_index]->instance_number = instance_index;
    instance_table[instance_index]->type_info = type_table[type_index];
    instance_table[instance_index]->type_info->ref_counter++;
    instance_table[instance_index]->missed_deadline = 0;

    /* only allocates space for this info if no instances of this thread
       has been created before */
    if (type_table[type_index]->thread_type_rt == (void *)0) {
      /* allocate space for the run-time type related information */
      type_table[type_index]->thread_type_rt = (struct thread_type_rt_info_t *)
	kmalloc(sizeof(struct thread_type_rt_info_t), GFP_KERNEL);

      /* initialize all the type run-time related info */
      type_table[type_index]->thread_type_rt->num_prev_executions = 0;
      type_table[type_index]->thread_type_rt->pos = 0;

      for ( i = 0; i < HISTORY_WINDOW_SIZE; i++ ) {
	type_table[type_index]->thread_type_rt->history[i] =
	  type_table[type_index]->info.wcet;
      }
      for ( i = 0; i < NUMBER_OF_WCET_SLOTS; i++ ) {
	type_table[type_index]->thread_type_rt->slotted_wcet[i] = 0;
      }
      type_table[type_index]->thread_type_rt->num_missed_deadlines = 0;
      type_table[type_index]->thread_type_rt->num_last_missed_deadlines = 0;
    }

    /* initializes instance information */
    // FIXME: use the start time + deadline to calculate absolute deadline 
    instance_table[instance_index]->total_time_so_far = 0;
    instance_table[instance_index]->deadline = 0xFFFFFFF;
    instance_table[instance_index]->preempted = 0;
    instance_table[instance_index]->done = 0;
    instance_table[instance_index]->arrival_time = 0;
    instance_table[instance_index]->sscale_factor = 100;
    instance_table[instance_index]->dscale_factor = 100;
    instance_table[instance_index]->adaptive_factor = 100;
    instance_table[instance_index]->compute_dfactor = 0;
    instance_table[instance_index]->dfactor_counter = 0;
  
    /* return the handle to the new instance */
    local_handle.type_info_index = type_index;
    local_handle.instance_info_index = instance_index;
    num_instances++;
    ordered_list_insert(instance_table[instance_index]);
    ordered_list_print();
  }
  /* otherwise just assign error codes to the handlers */
  else {
    local_handle.type_info_index = -1;
    local_handle.instance_info_index = -1;
  }

  /* copy handle back to user space */
  copy_to_user(instance_handle, &local_handle,
               sizeof(struct thread_instance_handle_t));
}


void 
sys_paapi_app_status(int status)
{
  unsigned index = current->instance_index;

  if (status == PAAPI_DVS_APP_START) {
  }
  else if (status == PAAPI_DVS_APP_DONE) {
    instance_table[index]->done = 1;

    if ( instance_table[index]->compute_dfactor ) {
      if ( instance_table[index]->dfactor_counter > 0 ) {
        instance_table[index]->dfactor_counter--;
        if ( instance_table[index]->dfactor_counter == 0 ) {
          rm_schedulability_analysis(1, 1, 1);
        }
        //else if ( instance_table[index]->deadline - PAAPI_CURRENT_TIME >
        //  OVERHEAD ) {
        //  rm_schedulability_analysis(1, 1, 1);
        //}
      }
    }
  }	
  else if ( status == PAAPI_DVS_APP_START_PERIOD ) {
    if ( !(current->paapi_task & PAAPI_TASK_STARTED) ) {
      current->paapi_task |= PAAPI_TASK_STARTED;
    }
    instance_table[index]->next_arrival = PAAPI_CURRENT_TIME + 100 +
      instance_table[index]->type_info->info.period;
    instance_table[index]->deadline = instance_table[index]->next_arrival +
      instance_table[index]->type_info->info.deadline;
    
    //printk("%d: APP SYS_START_PERIOD task_%d, nextarrival: %d, deadline: %d\n", 
    //  PAAPI_CURRENT_TIME, index, instance_table[index]->next_arrival,
    //  instance_table[index]->deadline);
    
    sys_paapi_sleep_until_next_period();
  }
}

void 
sys_paapi_associate(int index)
{
  /* associate the instance index with the kernel task structure */
  current->instance_index = index;
  current->paapi_task = PAAPI_TASK_FLAG;
  printk("APP %d: associating task_%d: %d, %d\n", PAAPI_CURRENT_TIME, 
         current->instance_index, current->rt_priority, current->paapi_task);
  spin_lock(&wait_for_adm_lock);
  /* put the thread to sleep */
  sleep_on(&wait_for_admission_control);
  spin_unlock(&wait_for_adm_lock);
}

void 
sys_paapi_sleep_until_next_period(void)
{
  //printk("%d: APP task_%d SLEEP\n", 
  //   PAAPI_CURRENT_TIME, current->instance_index);
  log_trace(APP_SLEEP, current->instance_index, PAAPI_CURRENT_TIME, 0,0);
  spin_lock(&wait_for_np_lock);
  /* put the thread to sleep */
  sleep_on(&wait_for_next_period);
  spin_unlock(&wait_for_np_lock);
}

int 
sys_paapi_control_taskset(int cmd, void *data)
{
  if ( cmd == PAAPI_CTRL_START_TASKSET ) {
    wait_queue_t *curr;
    struct list_head *tmp;
    if ( dvs_policy & PAAPI_DVS_STATIC_SLOWDOWN ) {
      rm_schedulability_analysis(1, 0, 1);

      //printk("List of threads sleeping (update priorities)\n");
      spin_lock(&wait_for_adm_lock);
      list_for_each(tmp, &(wait_for_admission_control.task_list)) {
        curr = list_entry(tmp, wait_queue_t, task_list);
        curr->task->rt_priority = 
          instance_table[curr->task->instance_index]->priority;
        //instance_table[curr->task->instance_index]->arrival_time = 
   	//  PAAPI_CURRENT_TIME;
        //printk("%d:%d\n", curr->task->instance_index, curr->task->rt_priority);
      }
      spin_unlock(&wait_for_adm_lock);

      admitted = 1;
      // wake up all processes
      wake_up(&wait_for_admission_control);
    }
  }
  else if ( cmd == PAAPI_CTRL_GET_NUM_INSTANCES ) {
    copy_to_user((unsigned *)data, &num_instances, sizeof(unsigned));
  }
  else if ( cmd == PAAPI_CTRL_GET_TASKSET_INFO ) {
    unsigned index, vec_index = 0;
    struct instance_t *instance_vector = (struct instance_t *)data;

    // FIXME: use the list of sleeping threads for the loop instead -- more 
    // efficient
    //printk("copying the data to instance_vector\n");
    for ( index = 0; index < MAX_TASKS_INSTANCES; index++ ) {
      if ( instance_table[index] != (void *)0) {
	//printk("instance vector: %d\n", 
	//instance_vector[index].type_info.wcet);
	//printk("instance table: %d\n", 
	//instance_table[index]->type_info->info.wcet);
	
	copy_to_user(&(instance_vector[vec_index].type_info), 
		     &(instance_table[index]->type_info->info),
		     sizeof(struct thread_type_t));
	copy_to_user(&(instance_vector[vec_index].instance_index),
		     &index, sizeof(unsigned));
	vec_index++;
      }
      if ( vec_index == num_instances ) break;
    }
  }
  else if ( cmd == PAAPI_CTRL_SET_TASKSET_INFO ) {
    unsigned index;
    struct instance_t *instance_vector = (struct instance_t *)data;
    wait_queue_t *curr;
    struct list_head *tmp;
    
    for ( index = 0; index < num_instances; index++ ) {
      if ( instance_table[instance_vector[index].instance_index] != (void *)0){
	/* copy the static slowdown factors back to the kernel level instance 
	   table */
	//printk("Updating slowdown factors and priority\n");
	instance_table[instance_vector[index].instance_index]->sscale_factor =
	  instance_vector[index].sfactor;
	instance_table[instance_vector[index].instance_index]->priority =
	  instance_vector[index].rt_priority;
	//printk("index: %d sfactor %d %d\n", 
	//       instance_vector[index].instance_index ,
	//       instance_vector[index].sfactor,
	//       instance_table[instance_vector[index].instance_index]->sscale_factor);
      }
    }
    //printk("List of threads sleeping (update priorities)\n");
    spin_lock(&wait_for_adm_lock);
    list_for_each(tmp, &(wait_for_admission_control.task_list)) {
      curr = list_entry(tmp, wait_queue_t, task_list);
      curr->task->rt_priority = 
        instance_table[curr->task->instance_index]->priority;
      //instance_table[curr->task->instance_index]->arrival_time = 
      //PAAPI_CURRENT_TIME;
      //printk("%d:%d\n", curr->task->instance_index, curr->task->rt_priority);
    }
    spin_unlock(&wait_for_adm_lock);

    admitted = 1;
    // wake up all processes
    wake_up(&wait_for_admission_control);
  }
  else if ( cmd == PAAPI_CTRL_GET_TASK_INFO ) {
    copy_to_user((unsigned *)data, 
      &(instance_table[current->instance_index]->type_info->info.wcet),
		 sizeof(unsigned));
  }
  else if ( cmd == PAAPI_CTRL_SET_POLICY ) {
    copy_from_user(&dvs_policy, (unsigned *)data, sizeof(unsigned));
  }
  else if ( cmd == PAAPI_CTRL_GET_POLICY ) {
    copy_to_user((unsigned *)data, &dvs_policy, sizeof(unsigned));    
  }
  else if ( cmd == PAAPI_CTRL_GET_SLOWDOWN_INFO ) {
    struct slowdown_info_t *si = (struct slowdown_info_t *)data;
    copy_to_user(&(si->dvs_policy), &dvs_policy, sizeof(unsigned));    
    copy_to_user(&(si->sfactor), 
      &(instance_table[current->instance_index]->sscale_factor),
      sizeof(unsigned));    
    copy_to_user(&(si->dfactor), 
      &(instance_table[current->instance_index]->dscale_factor),
      sizeof(unsigned));    
    copy_to_user(&(si->adaptive), 
      &(instance_table[current->instance_index]->adaptive_factor),
      sizeof(unsigned));    
    copy_to_user(&(si->max_speed), &max_speed, sizeof(unsigned));    
  }
  else if ( cmd == PAAPI_CTRL_CHANGE_TYPE ) {
    struct thread_type_t *thread_type = (struct thread_type_t *)data;
    copy_from_user(&(instance_table[current->instance_index]->type_info->info.wcet), &(thread_type->wcet), sizeof(unsigned));
    copy_from_user(&(instance_table[current->instance_index]->type_info->info.deadline), &(thread_type->deadline), sizeof(unsigned));
    copy_from_user(&(instance_table[current->instance_index]->type_info->info.period), &(thread_type->period), sizeof(unsigned));
    copy_from_user(&(instance_table[current->instance_index]->type_info->info.hard), &(thread_type->hard), sizeof(unsigned char));
    log_trace(CHANGE_PARAMETERS, current->instance_index, PAAPI_CURRENT_TIME, 
      instance_table[current->instance_index]->type_info->info.wcet,0);
    //printk("%d: going to change parameters of task_%d: %d %d %d\n", 
    //   PAAPI_CURRENT_TIME, current->instance_index, thread_type->wcet,
    //      thread_type->deadline, thread_type->period);

    ordered_list_delete(current->instance_index);
    ordered_list_insert(instance_table[current->instance_index]);

    if ( dvs_policy & PAAPI_DVS_STATIC_SLOWDOWN ) {
      rm_schedulability_analysis(1, 0, 1);
    }

    //printk("%d: Changed parameters of task_%d: %d %d %d\n", 
    //   PAAPI_CURRENT_TIME, current->instance_index,
    //   instance_table[current->instance_index]->type_info->info.wcet,
    //   instance_table[current->instance_index]->type_info->info.period,
    //   instance_table[current->instance_index]->type_info->info.deadline);
  }
  else if ( cmd == PAAPI_PRINT_LOG ) {
     print_trace();
  }
  return 0;
}

void
sys_paapi_destroy_task(void)
{
  unsigned index = current->instance_index;

  current->paapi_task = 0;

  if ( num_instances > 0 ) {
    num_instances--;
  }
  // FIXME: there is a race condition here...
  instance_table[index]->type_info->ref_counter--;
  ordered_list_delete(index);
  //printk("ordered_list_delete\n");
  if ( instance_table[index]->type_info->ref_counter == 0 ) {
    kfree(instance_table[index]->type_info->thread_type_rt);
    //printk("%d: task_%d deleting type related info as well\n", 
    //	   PAAPI_CURRENT_TIME, index);
  }
  instance_table[index]->type_info->thread_type_rt = (void *)0;
  instance_table[index]->type_info = (void *)0;
  kfree(instance_table[index]);
  instance_table[index] = (void *)0;
  admitted = 0;
  //printk("%d: task_%d leaving the system\n", PAAPI_CURRENT_TIME, index);
  log_trace(LEAVING, current->instance_index, PAAPI_CURRENT_TIME, 0,0);
  /*
  if ( dvs_policy & PAAPI_DVS_STATIC_SLOWDOWN ) {
    rm_schedulability_analysis(1, 0, !(dvs_policy & PAAPI_DVS_DYNAMIC_PREDICTIVE) );
  }
  if ( dvs_policy & PAAPI_DVS_DYNAMIC_PREDICTIVE ) {
    rm_schedulability_analysis(1, 1, 1);
  }
  */
}
