/*
* Driver interface to the ASIC Complasion chip on the iPAQ H3800
*
* Copyright 2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Andrew Christian
*          <Andrew.Christian@compaq.com>
*          October 2001
*
* Restrutured June 2002, September 2002
*/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/mtd/mtd.h>
#include <linux/ctype.h>
#include <linux/spi/spi.h>
#include <linux/delay.h>
#include <linux/serial.h>  /* For bluetooth */

#include <asm/irq.h>
#include <asm/uaccess.h>   /* for copy to/from user space */
#include <asm/arch/hardware.h>
#include <asm/arch/h3600_hal.h>
#include <asm/arch/h3600_asic.h>
#include <asm/arch/serial_h3800.h>

#include "h3600_asic_io.h"
#include "h3600_asic_core.h"
#include "h3600_asic_battery.h"

enum owm_state {
	OWM_STATE_IDLE = 0,
	OWM_STATE_RESET,
	OWM_STATE_WRITE,
	OWM_STATE_READ,
	OWM_STATE_DONE
};


/* Parameters */
int battery_sample_interval = 200;   /* Two seconds  */
int ds2760_sample_interval  = 1500;  /* 15 seconds  */
int sleep_battery_interval  = 5;     /* Five seconds */
int owm_reset_delay         = 100;   /* 100 milliseconds */
int owm_rw_delay            = 100;   /* 100 milliseconds */
int owm_sample_delay        = 200;   /* 20 milliseconds? */
int learn_voltage           = 3700;  /* 3700 millivolts */

MODULE_PARM(battery_sample_interval,"i");
MODULE_PARM_DESC(battery_sample_interval,"Time between battery charge checks (1/100 second)");
MODULE_PARM(ds2760_sample_interval,"i");
MODULE_PARM_DESC(ds2760_sample_interval,"Time between battery monitor reading (1/100 second)");
MODULE_PARM(sleep_battery_interval,"i");
MODULE_PARM_DESC(sleep_battery_interval,"Time betwen battery samples while asleep (seconds)");
MODULE_PARM(learn_voltage,"i");
MODULE_PARM_DESC(learn_voltage,"Voltage that triggers learn mode while charging (millivolts)");

MODULE_PARM(owm_reset_delay,"i");
MODULE_PARM_DESC(owm_reset_delay,"Pause time after OWM reset (milliseconds)");
MODULE_PARM(owm_rw_delay,"i");
MODULE_PARM_DESC(owm_rw_delay,"Pause time after OWM read/write (milliseconds)");
MODULE_PARM(owm_sample_delay,"i");
MODULE_PARM_DESC(owm_sample_delay,"Pause time while waiting for OWM response (milliseconds)");


struct ctl_table h3600_asic_battery_table[] =
{
	{ 1, "battery_sample_interval", &battery_sample_interval, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 2, "ds2760_sample_interval", &ds2760_sample_interval, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 3, "sleep_battery_interval", &sleep_battery_interval, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 4, "owm_reset_delay", &owm_reset_delay, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 5, "owm_rw_delay", &owm_rw_delay, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 6, "owm_sample_delay", &owm_sample_delay, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 7, "learn_voltage", &learn_voltage, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{0}
};

/***********************************************************************************
 *   One wire interface for talking with batteries
 *
 *   Resources used:     OWM interface on ASIC2
 *                       OWM Clock on CLOCK (CX6)
 *                       OWM KPIO interrupt line
 *   Shared resource:    EX1 (24.576 MHz crystal) on CLOCK
 ***********************************************************************************/

unsigned char *owm_state_names[] = {
	"Idle", "Reset", "Write", "Read", "Done"
};

static struct h3600_asic_owmdev {
	struct semaphore  lock;
	wait_queue_head_t waitq;
	enum owm_state    state;
	unsigned long     shared;
	unsigned char     last;
} g_owmdev;

struct owm_net_address {
	char address[8];
};

static int  h3600_asic_owm_sample( struct h3600_asic_owmdev *dev )
{
	int value = H3800_ASIC2_OWM_Interrupt;

	if ( (value & OWM_INT_PD) && dev->state == OWM_STATE_RESET ) {
		wake_up_interruptible(&dev->waitq);
		return 0;
	}
	
	if ( (value & OWM_INT_TBE) && (value & OWM_INT_RBF) && dev->state == OWM_STATE_WRITE ) {
		dev->last = H3800_ASIC2_OWM_Data;
		wake_up_interruptible(&dev->waitq);
		return 0;
	}

	if ( (value & OWM_INT_RBF) ) {
		dev->last = H3800_ASIC2_OWM_Data;
		if ( dev->state == OWM_STATE_READ ) {
			wake_up_interruptible(&dev->waitq);
			return 0;
		}
	}
	return 1;
}

static void h3600_asic_owm_isr( int irq, void *dev_id, struct pt_regs *regs ) 
{
	if ( h3600_asic_owm_sample(&g_owmdev) )
		g_h3600_asic_statistics.owm_invalid_isr[g_owmdev.state]++;
	else {
		g_h3600_asic_statistics.owm_valid_isr[g_owmdev.state]++;
		g_owmdev.state = OWM_STATE_DONE;
	}
}

static int one_wire_wait_for_interrupt( int msec )
{
	wait_queue_t  wait;
	signed long   timeout;
	int           result = 0;

	/* We're basically using interruptible_sleep_on_timeout */
	/* and waiting for the transfer to finish */
	init_waitqueue_entry(&wait,current);
	add_wait_queue(&g_owmdev.waitq, &wait);
	timeout = msec * HZ / 1000;

	while ( timeout > 0 ) {
		set_current_state( TASK_INTERRUPTIBLE );
		if ( g_owmdev.state == OWM_STATE_DONE )
			break;
		if ( signal_pending(current) ) {
			result = -ERESTARTSYS;
			break;
		}
		timeout = schedule_timeout( timeout );
		if ( timeout <= 0 ) {
			if ( h3600_asic_owm_sample(&g_owmdev) ) {
				result = -ETIMEDOUT;       /* is this right? */
				g_h3600_asic_statistics.owm_timeout++;
			}
			else {
				g_h3600_asic_statistics.owm_post_isr[g_owmdev.state]++;
			}
		}
	}
	set_current_state( TASK_RUNNING );
	remove_wait_queue(&g_owmdev.waitq, &wait);

	g_owmdev.state = OWM_STATE_IDLE;
	return result;
}

static int one_wire_reset( void )
{
	int result;

	if (0) printk(__FUNCTION__ "\n");

	g_h3600_asic_statistics.owm_reset++;
	g_owmdev.state = OWM_STATE_RESET;
	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Command |= OWM_CMD_ONE_WIRE_RESET;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) {
		printk(__FUNCTION__ " OWM reset failed %d (0x%04x)\n", result, H3800_ASIC2_OWM_Interrupt);
		return result;
	}
		
	/* No battery? */
	if ( H3800_ASIC2_OWM_Interrupt & OWM_INT_PDR ) { /* 0 indicates success */
		printk(__FUNCTION__ " OWM reset failed: no battery\n");
		return -ERESTARTSYS;
	}

	udelay(owm_reset_delay);
	return 0;
}

static int one_wire_write( unsigned char data )
{
	int result;

	if (0) printk(__FUNCTION__ ": 0x%02x\n", data);
	g_h3600_asic_statistics.owm_written++;
	g_owmdev.state = OWM_STATE_WRITE;

	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Data = data;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) 
		printk("OWM write failed %d (0x%04x)\n", 
		       result, H3800_ASIC2_OWM_Interrupt);

	udelay(owm_rw_delay);
	return result;
}

/* To read, we must clock in data and then read the output buffer */
static int one_wire_read( void )
{
	int result;

	if (0) printk(__FUNCTION__ "\n");
	g_h3600_asic_statistics.owm_read++;
	g_owmdev.state = OWM_STATE_READ;

	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Data = 0xff;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) {
		printk("OWM read failed %d (0x%04x)\n", 
		       result, H3800_ASIC2_OWM_Interrupt);
		return result;
	}

	udelay(owm_rw_delay);
	return g_owmdev.last;
}

/* Higher-level routines */

static int owm_setup( struct owm_net_address *net )
{
	int result, i;

	if ((result = one_wire_reset()) != 0) 
		return result;

	if (net) {
		if ((result = one_wire_write(0x55)) != 0) 
			return result;

		for ( i = 0 ; i < 8 ; i++ )
			if ((result = one_wire_write(net->address[i])) != 0) 
				return result;
	}
	else {
		if ((result = one_wire_write(0xcc)) != 0) 
			return result;
	}
	return 0;
}

static int h3600_asic_owm_read_bytes( struct owm_net_address *net, unsigned char address, 
				      unsigned char *data, unsigned short len )
{
	int result = 0;
	int i;

	if (down_interruptible(&g_owmdev.lock))
		return -ERESTARTSYS;

	if ((result = owm_setup(net)) < 0)          goto owm_read_bytes_fail;
	if ((result = one_wire_write(0x69)) < 0)    goto owm_read_bytes_fail;
	if ((result = one_wire_write(address)) < 0) goto owm_read_bytes_fail;

	for ( i = 0 ; i < len ; i++ ) {
		if ((result = one_wire_read()) < 0) goto owm_read_bytes_fail;
		data[i] = result;
	}
	result = 0;

owm_read_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}


static int h3600_asic_owm_write_bytes( struct owm_net_address *net, unsigned char address, 
				       unsigned char *data, unsigned short len )
{
	int result = 0;
	int i;

	if ( down_interruptible(&g_owmdev.lock) )
		return -ERESTARTSYS;

	if ((result = owm_setup(net)) < 0)           goto owm_write_bytes_fail;
	if ((result = one_wire_write(0x6c)) < 0 )    goto owm_write_bytes_fail;
	if ((result = one_wire_write(address)) < 0 ) goto owm_write_bytes_fail;

	for ( i = 0 ; i < len ; i++ )
		if ((result = one_wire_write(data[i])) < 0) goto owm_write_bytes_fail;
	result = 0;

owm_write_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}

static int h3600_asic_owm_copy_bytes( struct owm_net_address *net, unsigned char address )
{
	int result = 0;

	if ( down_interruptible(&g_owmdev.lock) )
		return -ERESTARTSYS;

	if ((result = owm_setup(net)) < 0)           goto owm_copy_bytes_fail;
	if ((result = one_wire_write(0x48)) < 0 )    goto owm_copy_bytes_fail;
	result = one_wire_write(address);
owm_copy_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}

static int h3600_asic_owm_recall_bytes( struct owm_net_address *net, unsigned char address )
{
	int result = 0;

	if ( down_interruptible(&g_owmdev.lock) )
		return -ERESTARTSYS;

	if ((result = owm_setup(net)) < 0)           goto owm_recall_bytes_fail;
	if ((result = one_wire_write(0xb8)) < 0 )    goto owm_recall_bytes_fail;
	result = one_wire_write(address);
owm_recall_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}

static void h3600_asic_owm_up( struct h3600_asic_owmdev *owm )
{
	owm->state = OWM_STATE_IDLE;

	h3600_asic_shared_add( &owm->shared, ASIC_SHARED_CLOCK_EX1 );
	H3800_ASIC2_CLOCK_Enable         |= ASIC2_CLOCK_OWM;
	H3800_ASIC2_OWM_ClockDivisor      = 2;
	H3800_ASIC2_OWM_InterruptEnable   = OWM_INTEN_ERBF | OWM_INTEN_IAS | OWM_INTEN_EPD;
	H3800_ASIC2_INTR_MaskAndFlag     |= ASIC2_INTMASK_OWM; /* Turn on ASIC interrupts */
}

static void h3600_asic_owm_down( struct h3600_asic_owmdev *owm )
{
	wake_up_interruptible(&owm->waitq);

	H3800_ASIC2_INTR_MaskAndFlag     &= ~ASIC2_INTMASK_OWM; /* Turn off ASIC interrupts */
	H3800_ASIC2_OWM_InterruptEnable   = 0;
	H3800_ASIC2_CLOCK_Enable         &= ~ASIC2_CLOCK_OWM;
	h3600_asic_shared_release( &owm->shared, ASIC_SHARED_CLOCK_EX1 );
}

int h3600_asic_owm_suspend( void )
{
	DEBUG_INIT();
	down(&g_owmdev.lock);
	h3600_asic_owm_down( &g_owmdev );
	disable_irq( IRQ_H3800_OWM );
	return 0;
}

void h3600_asic_owm_resume( void )
{
	DEBUG_INIT();
	enable_irq( IRQ_H3800_OWM );
	h3600_asic_owm_up( &g_owmdev );
	up(&g_owmdev.lock);
}

int h3600_asic_owm_init( void )
{
	int result;
	DEBUG_INIT();

	init_waitqueue_head( &g_owmdev.waitq );
	init_MUTEX(&g_owmdev.lock);

	h3600_asic_owm_up( &g_owmdev );

	result = request_irq(IRQ_H3800_OWM, h3600_asic_owm_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_owm", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab OWM virtual IRQ\n");
	return result;
}

void h3600_asic_owm_cleanup( void )
{
	DEBUG_INIT();
	h3600_asic_owm_down( &g_owmdev );
	free_irq( IRQ_H3800_OWM, NULL );
}


/***********************************************************************************
 *   Battery control
 * 
 *   Interface for talking with batteries and controlling the battery charger.
 *
 *   Resources used:     OWM interface
 *                       AC GPIO line
 *                       ADC converter to read battery charger status
 ***********************************************************************************/

struct hp_ds2760 {
	u8  protect;
	u8  status;
	u8  eeprom;
	u8  special;

	int voltage;
	int cur;
	int acr;    
	int temp;   

	int   cc[5];     /* Charge capacity array - Fuel gauging full point */
	u16   llcc;      /* Last learn cycle count */
	u32   tad;       /* Total accumulated discharge */
	int   cb;        /* Charging breakpoint */
	int   cbt[3];    /* Charging time breakpoint to full */
	int   cet[3];    /* Charging time empty to full */

	u8    chemistry;       /* Cell chemistry */
	u8    manufacturer;    /* Manufacturer */
	u16   rated_cap;       /* Rated cell capacity * 10 */
	char  current_offset;  /* current offset */
	u8    resistor;        /* Sense resistor - 1/4 mOhm resolution */
	u32   date;
	u16   original_cap;
	int   d2[5];
};

struct h3600_asic_batdev {
	struct timer_list    ac_timer;
	struct timer_list    ds2760_timer;
	enum charging_state  state;

	struct hp_ds2760     ds2760;
	u32                  tad;     
	int                  old_acr;
	int                  is_learning;  // -1 = no battery, 0 = not learning, > 1 learning
	int                  dsupdates;

	int                  full;        // Full level at this temperature (mAh)
	int                  derated;
	int                  empty;       // Empty level at this temperature (mAh)
	int                  percentage;  // 0 to 100
	int                  life;        // Minutes to failure
} g_batdev;

int h3600_asic_battery_get_state( void )
{
	return g_batdev.state;
}

void h3600_asic_battery_set_led( enum charging_state state )
{
	struct h3600_asic_batdev *dev = &g_batdev;

	if ( state != dev->state ) {
		switch (state) {
		case CHARGING_STATE_INIT: /* Deliberate fall through */
		case CHARGING_STATE_NO_AC:
			h3600_asic_led_off(YELLOW_LED);
			break;
		case CHARGING_STATE_ACTIVE:
			h3600_asic_led_blink(YELLOW_LED,0,4,2);
			break;
		case CHARGING_STATE_FULL:
			h3600_asic_led_on(YELLOW_LED);
			break;
		}
		dev->state = state;
	}
}

/*
static int voltage_to_percent( int voltage, int *cal )
{
	int i;

	if ( voltage > cal[0] )
		return 100;

	for ( i = 2 ; cal[i] > 0 ; i+=2 ) 
		if ( voltage > cal[i] ) {
			int p1 = cal[i+1];
			int v1 = cal[i];
			int p2 = cal[i-1];
			int v2 = cal[i-2];
			return (p1 * (v2 - voltage) + p2 * (voltage - v1)) / (v2 - v1);
		}

	return 0;
}

static int lipolymer_calibration[] = {
	881, 100, // 100%
	830,  90, //  90%
	816,  80, //  80%
	805,  70, //  70%
	797,  60, //  60%
	789,  50, //  50%
	783,  40, //  40%
	777,  30, //  30%
	770,  20, //  20%
	764,  10, //  10%
	750,   0, //   0% 
	  0
};
*/


/* 
 *  Read all sorts of good information from the battery
*/

#define DS2760_ACR_OFFSET     16
#define DS2760_TAD_OFFSET     39

#define DS2760_EEPROM_0       32
#define DS2760_EEPROM_1       48
#define BATTERY_READ_RAW_SIZE 64

static int h3600_asic_battery_read_raw( unsigned char *buf )
{
	int result;

	result = h3600_asic_owm_read_bytes( NULL, 0, buf, 32 );
	if ( result < 0 )
		return result;

	result = h3600_asic_owm_recall_bytes( NULL, DS2760_EEPROM_0 );
	if ( result < 0 ) return result;
	result = h3600_asic_owm_read_bytes( NULL, DS2760_EEPROM_0, buf + 32, 16 );
	if ( result < 0 ) return result;

	result = h3600_asic_owm_recall_bytes( NULL, DS2760_EEPROM_1 );
	if ( result < 0 ) return result;
	result = h3600_asic_owm_read_bytes( NULL, DS2760_EEPROM_1, buf + 48, 16 );
	if ( result < 0 ) return result;

	return 0;
}

static int h3600_asic_battery_read_all( struct hp_ds2760 *m )
{
	int result;
	unsigned char buf[32];

	result = h3600_asic_owm_read_bytes( NULL, 0, buf, 32 );
	if ( result < 0 )
		return result;

	m->protect = buf[0];
	m->status  = buf[1];
	m->eeprom  = buf[7];
	m->special = buf[8];
	m->voltage = (buf[12] << 3) | (buf[13] >> 5);
	if ( m->voltage > 1023 ) 
		return -EINVAL;    /* Can't have negative voltages */
	m->cur = (buf[14] << 5) | (buf[15] >> 3);
	if ( m->cur > 4095 ) m->cur -= 8192;
	m->acr = (buf[16] << 8) | buf[17];
	if ( m->acr > 32767 ) m->acr -= 65536;
	m->temp = (buf[24] << 3) | (buf[25] >> 5);
	if ( m->temp > 1023 ) m->temp -= 2024;
	
	result = h3600_asic_owm_recall_bytes( NULL, DS2760_EEPROM_0 );
	if ( result < 0 ) return result;
	result = h3600_asic_owm_read_bytes( NULL, DS2760_EEPROM_0, buf, 16 );
	if ( result < 0 ) return result;

	m->cc[0] = (buf[0] << 8) | buf[1];
	m->cc[1] = m->cc[0] + buf[2];
	m->cc[2] = m->cc[1] + buf[3];
	m->cc[3] = m->cc[2] + buf[4];
	m->cc[4] = m->cc[3] + buf[5];
	m->llcc  = buf[6] * 10;
	m->tad   = ((buf[7]<<8) | buf[8]) * 20;
	m->cb    = buf[9];
	m->cbt[0] = buf[10];
	m->cbt[1] = buf[11];
	m->cbt[2] = buf[12];
	m->cet[0] = buf[13];
	m->cet[1] = buf[14];
	m->cet[2] = buf[15];

	result = h3600_asic_owm_recall_bytes( NULL, DS2760_EEPROM_1 );
	if ( result < 0 ) return result;
	result = h3600_asic_owm_read_bytes( NULL, DS2760_EEPROM_1, buf, 16 );
	if ( result < 0 ) return result;

	m->chemistry      = (buf[0] & 0xc0) >> 6;
	m->manufacturer   = (buf[0] & 0x3c) >> 2;
	m->rated_cap      = buf[2] * 10;
	m->current_offset = buf[3];
	m->resistor       = buf[4];
	if ( m->resistor == 0 ) 
		m->resistor = 100;   /* 25 ohms internal */
	m->date           = (buf[8] << 24) | (buf[7] << 16) | (buf[6] << 8) | buf[5];
	m->original_cap   = buf[9] * 10;
	m->d2[4]          = buf[15];
	m->d2[3]          = m->d2[4] + buf[14];
	m->d2[2]          = m->d2[3] + buf[13];
	m->d2[1]          = m->d2[2] + buf[12];
	m->d2[0]          = m->d2[1] + buf[11];

	return 0;
}

int h3600_asic_battery_update_ds2760( struct hp_ds2760 *ds )
{
	unsigned char buf[16];
	int result;

	result = h3600_asic_owm_read_bytes( NULL, 12, buf, 14 );
	if ( result < 0 ) return result;

	ds->voltage = (buf[0] << 3) | (buf[1] >> 5);
	if ( ds->voltage > 1023 ) return -EINVAL;    /* Can't have negative voltages */

	ds->cur = (buf[2] << 5) | (buf[3] >> 3);
	if ( ds->cur > 4095 ) ds->cur -= 8192;
	ds->acr = (buf[4] << 8) | buf[5];
	if ( ds->acr > 32767 ) ds->acr -= 65536;
	ds->temp = (buf[12] << 3) | (buf[13] >> 5);
	if ( ds->temp > 1023 ) ds->temp -= 2024;

	return 0;
}


static int battery_interpolate( int array[], int temp )
{
	int index, dt;

	temp /= 8;
	if ( temp <= 0 )  return array[0];
	if ( temp >= 40 ) return array[4];

	index = temp / 10;
	dt    = temp % 10;

	return (array[index+1] * dt + array[index] * (10 - dt)) / 10;
}

#define ABS(x)  ((x)<0?-(x):(x))

static int set_and_store_acr( struct h3600_asic_batdev *dev, int acr )
{
	unsigned char buf[2];

	if ( ABS(acr - dev->ds2760.acr) <= 2 ) 
		return 0;

	if (0) printk(__FUNCTION__ ": setting ACR to %d\n", acr);

	dev->ds2760.acr = acr;
	dev->old_acr    = acr;

	buf[0] = acr >> 8;
	buf[1] = acr & 0xff;
	return h3600_asic_owm_write_bytes( NULL, DS2760_ACR_OFFSET, buf, 2 );
}

static inline int acr_to_mah( int acr, int resistor )       { return (acr * 25) / resistor; }
static inline int mah_to_acr( int mah, int resistor )       { return (mah * resistor) / 25; }
static inline int active_cur_to_ma( int cur, int resistor ) { return (125 * cur) / (resistor * 2); }
static inline int measured_to_mv( int measured )            { return measured * 5000 / 1024; }

#define BATTERY_DERATE_CYCLES    100
#define BATTERY_DERATE_PERCENT     3
#define LEARNING_THRESHOLD         3

static void battery_calc_levels( struct h3600_asic_batdev *dev )
{
	int delta, derate, acr, cur;

	dev->full    = battery_interpolate( dev->ds2760.cc, dev->ds2760.temp );
	derate       = ((dev->tad / dev->ds2760.original_cap) - dev->ds2760.llcc) / BATTERY_DERATE_CYCLES;
	dev->derated = dev->full - ( derate * BATTERY_DERATE_PERCENT ) / 100;
	dev->empty   = battery_interpolate( dev->ds2760.d2, dev->ds2760.temp );
	
	acr = acr_to_mah( dev->ds2760.acr, dev->ds2760.resistor );
	delta = acr - dev->empty;
	if ( delta < 0 ) delta = 0;

	if ( delta == 0 )
		dev->percentage = 0;
	else if ( acr >= dev->derated )
		dev->percentage = 100;
	else
		dev->percentage = (100*delta) / (dev->derated - dev->empty);

	cur = active_cur_to_ma( dev->ds2760.cur, dev->ds2760.resistor );
	if ( cur >= 0 )
		dev->life = -1;
	else
		dev->life = -(60 * delta) / cur;
}

static int h3600_asic_battery_set_full( struct h3600_asic_batdev *dev )
{
	int result, i;

	if ( dev->is_learning > LEARNING_THRESHOLD ) {
		unsigned char buf[9];
		int delta = dev->full - acr_to_mah(dev->ds2760.acr, dev->ds2760.resistor);

		for ( i = 0 ; i < 5 ; i++ )
			dev->ds2760.cc[i] -= delta;
		dev->ds2760.llcc = dev->tad / dev->ds2760.original_cap;
		dev->ds2760.tad  = dev->tad;
		dev->is_learning = 0;

		printk(__FUNCTION__ ": finished learning cycle, acr=%d llcc=%d delta=%d temp=%d cc=[%d %d %d %d %d]\n",
		       dev->ds2760.acr, dev->ds2760.llcc, delta, dev->ds2760.temp, dev->ds2760.cc[0],
		       dev->ds2760.cc[1], dev->ds2760.cc[2],dev->ds2760.cc[3],dev->ds2760.cc[4]);

		/* Store CC, LLCC, TAD */
		buf[0] =  dev->ds2760.cc[0] >> 8;
		buf[1] =  dev->ds2760.cc[0] & 0xff;
		buf[2] = (dev->ds2760.cc[1] - dev->ds2760.cc[0]) & 0xff;
		buf[3] = (dev->ds2760.cc[2] - dev->ds2760.cc[1]) & 0xff;
		buf[4] = (dev->ds2760.cc[3] - dev->ds2760.cc[2]) & 0xff;
		buf[5] = (dev->ds2760.cc[4] - dev->ds2760.cc[3]) & 0xff;
		buf[6] =  dev->ds2760.llcc / 10;
		buf[7] = (dev->ds2760.tad / 20) >> 8;
		buf[8] = (dev->ds2760.tad / 20) & 0xff;

		if ((result = h3600_asic_owm_recall_bytes(NULL, DS2760_EEPROM_0)) < 0)        return result;
		if ((result = h3600_asic_owm_write_bytes(NULL, DS2760_EEPROM_0, buf, 9)) < 0) return result;
		if ((result = h3600_asic_owm_copy_bytes(NULL, DS2760_EEPROM_0)) < 0)          return result;
		if ((result = h3600_asic_owm_recall_bytes(NULL, DS2760_EEPROM_0)) < 0)        return result;
	}
	else {
		int new_acr = mah_to_acr(dev->derated, dev->ds2760.resistor);

		result = set_and_store_acr( dev, new_acr );
		if ( result ) {
			printk(__FUNCTION__ ": Unable to write ACR\n");
			return result;
		}
	}

	return 0;
}

static int h3600_asic_battery_set_empty( struct h3600_asic_batdev *dev )
{
	if ( dev->is_learning == LEARNING_THRESHOLD ) {
		int new_acr = mah_to_acr(dev->empty, dev->ds2760.resistor);
		int result  = set_and_store_acr( dev, new_acr );

		if ( result ) {
			printk(__FUNCTION__ ": Unable to write ACR\n");
			dev->is_learning = 0;   // Try again in a few cycles
			return result;
		}

		printk(__FUNCTION__ ": starting learning cycle ACR=%d\n", dev->ds2760.acr );
	}
	dev->is_learning++;
	return 0;
}

/* Call this with valid charging state     */
/* Do _not_ call this in interrupt context */

#define MIN_TAD_UPDATE  100
#define TAD_DELTA       280    /* mAh - approximately 20% of a full charge */

static int h3600_asic_battery_store_tad( struct h3600_asic_batdev *dev )
{
	int result;
	unsigned char buf[2];

	buf[0] = (dev->tad / 20) >> 8;
	buf[1] = (dev->tad / 20) & 0xff;
			
	if ((result = h3600_asic_owm_recall_bytes(NULL, DS2760_EEPROM_0)) < 0)        return result;
	if ((result = h3600_asic_owm_write_bytes(NULL, DS2760_TAD_OFFSET, buf, 2))<0) return result;
	if ((result = h3600_asic_owm_copy_bytes(NULL, DS2760_EEPROM_0)) < 0)          return result;
	if ((result = h3600_asic_owm_recall_bytes(NULL, DS2760_EEPROM_0)) < 0)        return result;

	if (0) printk(__FUNCTION__ ": updating TAD from %d to %d\n", dev->ds2760.tad, dev->tad );
	dev->ds2760.tad = dev->tad;
	return 0;
}

static int h3600_asic_battery_update_tad( struct h3600_asic_batdev *dev )
{
	if ( dev->ds2760.acr > dev->old_acr ) {  // We have been charging 
		dev->old_acr = dev->ds2760.acr;
		return 0;
	}
	
	if ( dev->old_acr - dev->ds2760.acr > MIN_TAD_UPDATE ) {
		dev->tad += acr_to_mah(dev->old_acr - dev->ds2760.acr, dev->ds2760.resistor);
		dev->old_acr = dev->ds2760.acr;
	}
		
	if ( dev->tad > dev->ds2760.tad + TAD_DELTA )
		return h3600_asic_battery_store_tad( dev );

	return 0;
}

static int h3600_asic_battery_init_ds2760( struct h3600_asic_batdev *dev )
{
	int result = h3600_asic_battery_read_all(&dev->ds2760);
	if ( result < 0 )
		return result;

	dev->tad         = dev->ds2760.tad;
	dev->old_acr     = dev->ds2760.acr;
	dev->is_learning = 0;
	return 0;
}

static void h3600_asic_battery_update( void *nr )
{
	struct h3600_asic_batdev *dev = (struct h3600_asic_batdev *) nr;
	int result;

	dev->dsupdates++;
	if ( dev->is_learning < 0 )
		result = h3600_asic_battery_init_ds2760(dev);
	else
		result = h3600_asic_battery_update_ds2760( &dev->ds2760 );

	if ( result < 0 )
		return;   // No error messages - that'd just fill the log files

	result = h3600_asic_battery_update_tad( dev );
	if ( result < 0 )
		printk(__FUNCTION__ ": unable to update stored TAD value (%d)\n", result);
	
	battery_calc_levels( dev );

	switch (dev->state) {
	case CHARGING_STATE_INIT:
	case CHARGING_STATE_NO_AC:
		if ( dev->is_learning > LEARNING_THRESHOLD )
			printk(__FUNCTION__ ": canceling learning\n");
		dev->is_learning = 0;
		break;
	case CHARGING_STATE_ACTIVE:
		if ( measured_to_mv(dev->ds2760.voltage) < learn_voltage )
			h3600_asic_battery_set_empty( dev );
		break;
	case CHARGING_STATE_FULL:
		h3600_asic_battery_set_full( dev );
		break;
	}
}

/***********************************************************************************
 *  The OWM interface call doesn't always work ... sometimes the battery
 *  just doesn't respond.  We return an error code, but the APM circuitry
 *  doesn't use error codes (go figure!).  So we also return a structure filled
 *  out with values equal to the last time we called.  The only value that is
 *  affected will be the battery voltage level.
 ***********************************************************************************/

int h3600_asic_battery_read( struct h3600_battery *query )
{
	int voltage    = 783;         /* DUMMY VALUE - good for first initialization */
	int chemistry  = H3600_BATT_CHEM_UNKNOWN;
	int percentage = 0;
	int life       = 0;
        int ac_power   = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;

	h3600_asic_battery_update( &g_batdev );
	if ( g_batdev.is_learning >= 0 ) {
		chemistry  = g_batdev.ds2760.chemistry;
		voltage    = g_batdev.ds2760.voltage;
		percentage = g_batdev.percentage;
		life       = g_batdev.life;
	}

	query->ac_status            = (ac_power ? H3600_AC_STATUS_AC_ONLINE : H3600_AC_STATUS_AC_OFFLINE );
	query->battery_count        = 1;
	query->battery[0].chemistry = ( chemistry == 0 ? H3600_BATT_CHEM_LIPOLY : H3600_BATT_CHEM_UNKNOWN );
	query->battery[0].voltage   = voltage;

	query->battery[0].status    = H3600_BATT_STATUS_UNKNOWN;
	switch ( g_batdev.state ) {
	case CHARGING_STATE_INIT:
	case CHARGING_STATE_NO_AC:
		if ( voltage > 782 )  /* About 3.82 volts */
			query->battery[0].status = H3600_BATT_STATUS_HIGH;
		else if ( voltage > 764 ) /* About 3.73 voltags */
			query->battery[0].status = H3600_BATT_STATUS_LOW;
		else
			query->battery[0].status = H3600_BATT_STATUS_CRITICAL;
		break;
	case CHARGING_STATE_ACTIVE:
		query->battery[0].status = H3600_BATT_STATUS_CHARGING;
		break;
	case CHARGING_STATE_FULL:
		query->battery[0].status = H3600_BATT_STATUS_FULL;
		break;
	}

	query->battery[0].percentage = percentage;
	query->battery[0].life       = life;
	
	/* If a sleeve has been inserted, give it a try */
	if ( !(GPLR & GPIO_H3800_NOPT_IND ) && (H3800_ASIC2_GPIOPIOD & GPIO2_OPT_ON)) {
		unsigned char chem, percent, flag;
		if ( !h3600_asic_spi_read_pcmcia_battery( &chem, &percent, &flag ) ) {
			query->battery_count = 2;
			query->battery[1].chemistry  = chem;
			query->battery[1].voltage    = 0;
			query->battery[1].status     = flag;
			query->battery[1].percentage = percent;
			query->battery[1].life       = 0;
		}
	}

	return 0;
}

int h3600_asic_thermal_sensor( unsigned short *result )
{
	h3600_asic_battery_update(&g_batdev);
	*result = g_batdev.ds2760.temp;
	if ( g_batdev.is_learning < 0 )
		return -EINVAL;
	return 0;
}

/*****************************************************************************/

static struct tq_struct ds2760_task = { 
	routine: h3600_asic_battery_update,
	data:    &g_batdev
};

static void h3600_asic_battery_ds2760_timer_callback( unsigned long nr )
{
	struct h3600_asic_batdev *dev = (struct h3600_asic_batdev *) nr;
	schedule_task(&ds2760_task);
	mod_timer(&dev->ds2760_timer, jiffies + ds2760_sample_interval);
}


static void h3600_asic_battery_probe_task_handler( void *nr )
{
	int result = h3600_asic_adc_read_channel( ASIC2_ADMUX_1_IMIN );

	if ( result < 0 ) {
		printk(__FUNCTION__ " error reading battery channel\n");
	}
	else if ( result < 100 ) {
		h3600_asic_battery_set_led( CHARGING_STATE_FULL );
	}
	else {
		h3600_asic_battery_set_led( CHARGING_STATE_ACTIVE );
	}
}

static struct tq_struct battery_probe_task = { routine: h3600_asic_battery_probe_task_handler };

static void h3600_asic_battery_ac_timer_callback( unsigned long nr )
{
	struct h3600_asic_batdev *dev = (struct h3600_asic_batdev *) nr;
	schedule_task(&battery_probe_task);
	mod_timer(&dev->ac_timer, jiffies + battery_sample_interval);
}

/*****************************************************************************/

static void h3600_asic_battery_up( struct h3600_asic_batdev *dev )
{
        int ac_power = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;

	if ( ac_power ) {
		h3600_asic_battery_set_led( CHARGING_STATE_ACTIVE );
		H3800_ASIC1_GPIO_OUT |= GPIO1_CH_TIMER;
		mod_timer(&dev->ac_timer, jiffies + battery_sample_interval);
	}
	else {
		h3600_asic_battery_set_led( CHARGING_STATE_NO_AC );
		H3800_ASIC1_GPIO_OUT &= ~GPIO1_CH_TIMER;
		del_timer_sync(&dev->ac_timer);
	}

	mod_timer(&dev->ds2760_timer, jiffies + 1);
}

static void h3600_asic_battery_down( struct h3600_asic_batdev *dev )
{
	h3600_asic_battery_set_led( CHARGING_STATE_NO_AC );
	H3800_ASIC1_GPIO_OUT &= ~GPIO1_CH_TIMER;
	del_timer_sync(&dev->ac_timer);
	del_timer_sync(&dev->ds2760_timer);
        flush_scheduled_tasks();
}

void h3600_asic_ac_in_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	h3600_asic_battery_up( &g_batdev );
}

int h3600_asic_battery_suspend( void )
{
	DEBUG_INIT();
	del_timer_sync(&g_batdev.ac_timer);
	del_timer_sync(&g_batdev.ds2760_timer);
        flush_scheduled_tasks();
	return 0;
}

void h3600_asic_battery_resume( void )
{
	DEBUG_INIT();
	h3600_asic_battery_up( &g_batdev );
}

int h3600_asic_battery_init( void )
{
	DEBUG_INIT();

	init_timer(&g_batdev.ac_timer);
	g_batdev.ac_timer.function = h3600_asic_battery_ac_timer_callback;
	g_batdev.ac_timer.data     = (unsigned long) &g_batdev;

	init_timer(&g_batdev.ds2760_timer);
	g_batdev.ds2760_timer.function = h3600_asic_battery_ds2760_timer_callback;
	g_batdev.ds2760_timer.data     = (unsigned long) &g_batdev;

	g_batdev.state          = CHARGING_STATE_INIT;
	g_batdev.is_learning    = -1;
	g_batdev.dsupdates      = 0;

	h3600_asic_battery_up( &g_batdev );
	return 0;
}

void h3600_asic_battery_cleanup( void )
{
	DEBUG_INIT();

	h3600_asic_battery_down( &g_batdev );
}

/***********************************************************************************
 *   Proc filesystem
 ***********************************************************************************/

int h3600_asic_proc_battery(char *page, char **start, off_t off,
			    int count, int *eof, void *data)
{
	char *p = page;
	unsigned char chem, percent, flag;

	MOD_INC_USE_COUNT;    // Necessary because we can sleep

	h3600_asic_battery_update( &g_batdev );

	if ( g_batdev.is_learning < 0 ) {
		p += sprintf(p, "Unable to read DS2760 battery monitor\n");
	}
	else {
		struct hp_ds2760 *ds = &g_batdev.ds2760;
		p += sprintf(p, "Battery Monitor\n");
		p += sprintf(p, "  Voltage                        : %d mV\n", measured_to_mv(ds->voltage));
		p += sprintf(p, "  Active current                 : %d mA\n", active_cur_to_ma(ds->cur,ds->resistor));
		p += sprintf(p, "  Accumulated current            : %d mAh\n", acr_to_mah(ds->acr,ds->resistor));
		p += sprintf(p, "  Temperature                    : %d.%03d deg C\n", 
			     ds->temp / 8, (ds->temp % 8) * 125);

		p += sprintf(p, "Local calculations\n");
		p += sprintf(p, "  Local TAD                      : %d\n", g_batdev.tad );
		p += sprintf(p, "  Last ACR update value          : %d\n", g_batdev.old_acr );
		p += sprintf(p, "  Is learning?                   : %s\n", 
			     (g_batdev.is_learning >= LEARNING_THRESHOLD) ? "yes" : "no" );
		p += sprintf(p, "  DS2760 updates                 : %d\n", g_batdev.dsupdates );
		p += sprintf(p, "  Full level                     : %d mAh\n", g_batdev.full );
		p += sprintf(p, "  Derated full level             : %d mAh\n", g_batdev.derated );
		p += sprintf(p, "  Empty level                    : %d mAh\n", g_batdev.empty );
		p += sprintf(p, "  Percentage                     : %d\n", g_batdev.percentage );
		p += sprintf(p, "  Lifetime                       : %d min\n", g_batdev.life );
	}

	if ( !h3600_asic_spi_read_pcmcia_battery( &chem, &percent, &flag ) ) {
		p += sprintf(p, "Option jacket\n");
		p += sprintf(p, "          chemistry : 0x%02x\n", chem );
		p += sprintf(p, "         percentage : 0x%02x (%d)\n", percent, percent );
		p += sprintf(p, "               flag : 0x%02x\n", flag );
	}

	*eof = 1;
	MOD_DEC_USE_COUNT;
	return (p - page);
}

int h3600_asic_proc_battery_ds2760(char *page, char **start, off_t off,
				   int count, int *eof, void *data)
{
	char *p = page;

	MOD_INC_USE_COUNT;    // Necessary because we can sleep

	h3600_asic_battery_update( &g_batdev );

	if ( g_batdev.is_learning < 0 ) {
		p += sprintf(p, "Unable to read DS2760 battery monitor\n");
	}
	else {
		struct hp_ds2760 *ds = &g_batdev.ds2760;
		p += sprintf(p, "Battery Monitor\n");
		p += sprintf(p, "  Protection                     : 0x%02x\n", ds->protect );
		p += sprintf(p, "  Status                         : 0x%02x\n", ds->status );
		p += sprintf(p, "  EEPROM                         : 0x%02x\n", ds->eeprom );
		p += sprintf(p, "  Special feature                : 0x%02x\n", ds->special );
		p += sprintf(p, "  Voltage                        : %d (%d mV)\n", 
			     ds->voltage, measured_to_mv(ds->voltage));
		p += sprintf(p, "  Active current                 : %d (%d mA)\n", 
			     ds->cur, active_cur_to_ma(ds->cur,ds->resistor));
		p += sprintf(p, "  Accumulated current            : %d (%d mAh)\n", ds->acr, 
			     acr_to_mah(ds->acr,ds->resistor));
		p += sprintf(p, "  Temperature                    : %d (%d.%03d deg C)\n", 
			     ds->temp, ds->temp / 8, (ds->temp % 8) * 125);

		p += sprintf(p, "  Charge capacity by temperature : %d %d %d %d %d (mAh)\n", 
			     ds->cc[0],ds->cc[1],ds->cc[2],ds->cc[3],ds->cc[4]);
		p += sprintf(p, "  Last learn cycle count         : %d (cycles)\n", ds->llcc );
		p += sprintf(p, "  Total accumulated discharge    : %d (mAh)\n", ds->tad );
		p += sprintf(p, "  Charging breakpoint            : %d (mAh)\n", ds->cb );
		p += sprintf(p, "  Charge time breakpoint to full : %d %d %d (min)\n", ds->cbt[0],ds->cbt[1],ds->cbt[2]);
		p += sprintf(p, "  Charge time empty to full      : %d %d %d (min)\n", ds->cet[0],ds->cet[1],ds->cet[2]);

		p += sprintf(p, "  Cell chemistry                 : %d\n", ds->chemistry );
		p += sprintf(p, "  Manufacturer                   : %d\n", ds->manufacturer );
		p += sprintf(p, "  Rated cell capacity            : %d (mAh)\n", ds->rated_cap );
		p += sprintf(p, "  Current offset                 : 0x%02x\n", ds->current_offset );
		p += sprintf(p, "  Sense resistor                 : %d (%d.%02d mOhm)\n", 
			     ds->resistor, ds->resistor / 4, (ds->resistor % 4) * 25);
		p += sprintf(p, "  Manufacture date               : %d (seconds from 1970)\n", ds->date );
		p += sprintf(p, "  Original capacity              : %d (mAh)\n", ds->original_cap );
		p += sprintf(p, "  Active empty level by temp     : %d %d %d %d %d (mAh)\n", 
			     ds->d2[0],ds->d2[1],ds->d2[2],ds->d2[3],ds->d2[4]);
	}

	*eof = 1;
	MOD_DEC_USE_COUNT;
	return (p - page);
}

int h3600_asic_proc_battery_raw(char *page, char **start, off_t off,
				       int count, int *eof, void *data)
{
	char *p = page;
	int result, i;
	unsigned char buf[BATTERY_READ_RAW_SIZE];

	MOD_INC_USE_COUNT;    // Necessary because we can sleep

	result = h3600_asic_battery_read_raw( buf );
	if ( result < 0 )
		p += sprintf(p, "Unable to read battery (%d)\n", result);
	else
		for ( i = 0 ; i < BATTERY_READ_RAW_SIZE ; i++ ) {
			p += sprintf(p, "%02x", buf[i]);
			*p++ = ( (i+1) % 8 ? ' ' : '\n');
		}

	*eof = 1;
	MOD_DEC_USE_COUNT;
	return (p - page);
}


