/*
 * Flash memory access on iPAQ Handhelds (either SA1100 or PXA250 based)
 * 
 * (C) 2000 Nicolas Pitre <nico@cam.org>
 * (C) 2002 Hewlett-Packard Company <jamey.hicks@hp.com>
 * 
 * $Id: ipaq-flash.c,v 1.1 2002/08/23 18:39:42 jamey Exp $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/hardware.h>
#include <asm/io.h>


#ifndef CONFIG_IPAQ_H3XXX
#error This is for iPAQ Handhelds only
#endif


static __u8 ipaq_read8(struct map_info *map, unsigned long ofs)
{
	return readb(map->map_priv_1 + ofs);
}

static __u16 ipaq_read16(struct map_info *map, unsigned long ofs)
{
	return readw(map->map_priv_1 + ofs);
}

static __u32 ipaq_read32(struct map_info *map, unsigned long ofs)
{
	return readl(map->map_priv_1 + ofs);
}

static void ipaq_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy(to, (void *)(map->map_priv_1 + from), len);
}

static void ipaq_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	writeb(d, map->map_priv_1 + adr);
}

static void ipaq_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	writew(d, map->map_priv_1 + adr);
}

static void ipaq_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	writel(d, map->map_priv_1 + adr);
}

static void ipaq_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy((void *)(map->map_priv_1 + to), from, len);
}

#ifdef CONFIG_IPAQ_JORNADA56X

static void jornada56x_set_vpp(struct map_info *map, int vpp)
{
	if (vpp)
		GPSR = GPIO_GPIO26;
	else
		GPCR = GPIO_GPIO26;
	GPDR |= GPIO_GPIO26;
}

#endif

#ifdef CONFIG_IPAQ_JORNADA720

static void jornada720_set_vpp(struct map_info *map, int vpp)
{
	if (vpp)
		PPSR |= 0x80;
	else
		PPSR &= ~0x80;
	PPDR |= 0x80;
}

#endif

static struct map_info ipaq_map = {
	name:		"IPAQ flash",
	read8:		ipaq_read8,
	read16:		ipaq_read16,
	read32:		ipaq_read32,
	copy_from:	ipaq_copy_from,
	write8:		ipaq_write8,
	write16:	ipaq_write16,
	write32:	ipaq_write32,
	copy_to:	ipaq_copy_to,

	map_priv_1:	0,
};


/*
 * Here are partition information for all known IPAQ-based devices.
 * See include/linux/mtd/partitions.h for definition of the mtd_partition
 * structure.
 *
 * The *_max_flash_size is the maximum possible mapped flash size which
 * is not necessarily the actual flash size.  It must be no more than
 * the value specified in the "struct map_desc *_io_desc" mapping
 * definition for the corresponding machine.
 *
 * Please keep these in alphabetical order, and formatted as per existing
 * entries.  Thanks.
 */

#ifdef CONFIG_IPAQ_H3XXX
static unsigned long h3xxx_max_flash_size = 0x02000000;
static struct mtd_partition h3xxx_partitions[] = {
	{
		name:		"H3XXX boot firmware",
		size:		0x00040000,
		offset:		0,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"H3XXX root jffs2",
		size:		MTDPART_SIZ_FULL,
		offset:		0x00040000,
	}
};

static void h3xxx_set_vpp(struct map_info *map, int vpp)
{
	assign_h3600_egpio(IPAQ_EGPIO_VPP_ON, vpp);
}
#endif

#if defined(CONFIG_IPAQ_JORNADA56X) || defined(CONFIG_IPAQ_JORNADA720)
static unsigned long jornada_max_flash_size = 0x02000000;
static struct mtd_partition jornada_partitions[] = {
	{
		name:		"Jornada boot firmware",
		size:		0x00040000,
		offset:		0,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"Jornada root jffs2",
		size:		MTDPART_SIZ_FULL,
		offset:		0x00040000,
	}
};
#endif

extern int parse_redboot_partitions(struct mtd_info *master, struct mtd_partition **pparts);
extern int parse_bootldr_partitions(struct mtd_info *master, struct mtd_partition **pparts);
extern int parse_cmdline_partitions(struct mtd_info *master, struct mtd_partition **pparts, const char *mtd_id);

static struct mtd_partition *parsed_parts;
static struct mtd_info *mymtd;

int __init ipaq_mtd_init(void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	int parsed_nr_parts = 0;
	const char *part_type;

	/* Default flash buswidth */
	// ipaq_map.buswidth = (MSC0 & MSC_RBW) ? 2 : 4;
	ipaq_map.buswidth = 4;

	/*
	 * Static partition definition selection
	 */
	part_type = "static";

#ifdef CONFIG_IPAQ_H3XXX
	if (machine_is_ipaq()) {
		parts = h3xxx_partitions;
		nb_parts = ARRAY_SIZE(h3xxx_partitions);
		ipaq_map.size = h3xxx_max_flash_size;
		ipaq_map.set_vpp = h3xxx_set_vpp;
                ipaq_map.map_priv_1 = (__u32)__ioremap(0x0, 0x04000000, 0);
	}
#endif
#ifdef CONFIG_SA1100_JORNADA56X
	if (machine_is_jornada56x()) {
		parts = jornada_partitions;
		nb_parts = ARRAY_SIZE(jornada_partitions);
		ipaq_map.size = jornada_max_flash_size;
		ipaq_map.set_vpp = jornada56x_set_vpp;
		ipaq_map.map_priv_1 = (__u32)__ioremap(0x0, 0x04000000, 0);
	}
#endif
#ifdef CONFIG_SA1100_JORNADA720
	if (machine_is_jornada720()) {
		parts = jornada_partitions;
		nb_parts = ARRAY_SIZE(jornada_partitions);
		ipaq_map.size = jornada_max_flash_size;
		ipaq_map.set_vpp = jornada720_set_vpp;
		ipaq_map.map_priv_1 = (__u32)__ioremap(0x0, 0x04000000, 0);
	}
#endif

	/*
	 * Now let's probe for the actual flash.  Do it here since
	 * specific machine settings might have been set above.
	 */
	printk(KERN_NOTICE "IPAQ flash: probing %d-bit flash bus, window=%x\n", ipaq_map.buswidth*8, ipaq_map.map_priv_1);
	mymtd = do_map_probe("cfi_probe", &ipaq_map);
	if (!mymtd)
		return -ENXIO;
	mymtd->module = THIS_MODULE;

	/*
	 * Dynamic partition selection stuff (might override the static ones)
	 */
#ifdef CONFIG_MTD_REDBOOT_PARTS
	if (parsed_nr_parts == 0) {
		int ret = parse_redboot_partitions(mymtd, &parsed_parts);

		if (ret > 0) {
			part_type = "RedBoot";
			parsed_nr_parts = ret;
		}
	}
#endif
#ifdef CONFIG_MTD_BOOTLDR_PARTS
	if (parsed_nr_parts == 0) {
		int ret = parse_bootldr_partitions(mymtd, &parsed_parts);
		if (ret > 0) {
			part_type = "Compaq bootldr";
			parsed_nr_parts = ret;
		}
	}
#endif
#ifdef CONFIG_MTD_CMDLINE_PARTS
	if (parsed_nr_parts == 0) {
		int ret = parse_cmdline_partitions(mymtd, &parsed_parts, "ipaq");
		if (ret > 0) {
			part_type = "cmdline";
			parsed_nr_parts = ret;
		}
	}
#endif

	if (parsed_nr_parts > 0) {
		parts = parsed_parts;
		nb_parts = parsed_nr_parts;
	}

	if (nb_parts == 0) {
		printk(KERN_NOTICE "IPAQ flash: no partition info available, registering whole flash at once\n");
		add_mtd_device(mymtd);
	} else {
		printk(KERN_NOTICE "Using %s partition definition\n", part_type);
		add_mtd_partitions(mymtd, parts, nb_parts);
	}
	return 0;
}

static void __exit ipaq_mtd_cleanup(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
		if (parsed_parts)
			kfree(parsed_parts);
	}
}

module_init(ipaq_mtd_init);
module_exit(ipaq_mtd_cleanup);

MODULE_AUTHOR("Jamey Hicks");
MODULE_DESCRIPTION("IPAQ CFI map driver");
MODULE_LICENSE("MIT");
