/*
 * Host AP (software wireless LAN access point) driver for Intersil Prism2
 * Copyright (c) 2001-2002, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com> and/or <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 *
 * FIX:
 * - shutting down pcmcia-cs while wlan# interface is up does not work
 *   (it does not remove the driver and eventually the kernel will panic when
 *   card is removed); when interface is first configured down, pcmcia-cs can
 *   be shut down; update: this seems to work at least with kernel-based
 *   pcmcia and Linux 2.4.13..
 * - SIOCSIWRATE can set TX rate for adhoc and managed modes, but this is not
 *   currently used in host-based TX rate control; it could be used also for
 *   this (i.e., allow AP to be configured to limit/force TX rate)
 * - there is currently no way of associating TX packets to correct wds device
 *   when TX Exc/OK event occurs, so all tx_packets and some
 *   tx_errors/tx_dropped are added to the main netdevice; using sw_support
 *   field in txdesc might be used to fix this (using Alloc event to increment
 *   tx_packets would need some further info in txfid table)
 *
 * Buffer Access Path (BAP) usage:
 *   BAP0 is used for sending data from the host computer to the card and for
 *   RID read/writes; it is protected with local->baplock (spin_lock_bh) and it
 *   must not be used from hardware interrupt context.
 *   BAP1 is used for receiving data from the card to the host computer; it is
 *   used only in hardware interrupt handler and kernel is assumed to not call
 *   the same handler simultaneously for the same interrupt even on SMP
 *   systems (this removes need for spin_lock protecting BAP1 access.
 */


#include <linux/config.h>
#include <linux/version.h>

#include <asm/io.h>
#include <asm/delay.h>
#include <asm/uaccess.h>

#include <linux/slab.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/proc_fs.h>
#include <linux/if_arp.h>
#include <linux/delay.h>
#include <linux/random.h>

#if defined(CONFIG_PRISM2_CS) || defined(CONFIG_PRISM2_CS_MODULE)
#define PRISM2_PCCARD
#endif

#ifdef PRISM2_MONITOR
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
/* net/sock.h (at least in 2.2.17) does not like min()/max() macros from
 * pcmcia-cs's kernel compat header */
#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
#endif /* kernel < 2.4.0 */

#include <net/sock.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#endif /* PRISM2_MONITOR */



#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
/* 2.2 compatibility */
#ifndef spin_lock_bh
#define spin_lock_bh(lock) spin_lock_irq(lock)
#define spin_unlock_bh(lock) spin_unlock_irq(lock)
#endif
#ifndef __constant_cpu_to_le16
#define __constant_cpu_to_le16 __cpu_to_le16
#endif

#ifdef PRISM2_MONITOR
#ifndef rtnl_shlock_nowait 
static struct semaphore rtnl_sem = MUTEX;
#define rtnl_shlock_nowait()    down_trylock(&rtnl_sem)
#endif
#endif /* PRISM2_MONITOR */
#define PRISM2_NETDEV_EXTRA IFNAMSIZ
#define prism2_set_dev_name(dev, pos) (dev)->name = (char *) (pos)
/* end 2.2 compatibility */
#else /* kernel < 2.4.0 */
/* no extra space needed for 2.4.x net_device */
#define PRISM2_NETDEV_EXTRA 0
#define prism2_set_dev_name(dev, pos) do { } while (0)
#endif /* kernel < 2.4.0 */


#include "prism2_ap.h"


/* #define final_version */

#define MAX_PARM_DEVICES 8
#define PARM_MIN_MAX "1-" __MODULE_STRING(MAX_PARM_DEVICES)
#define DEF_INTS -1, -1, -1, -1, -1, -1, -1
#define GET_INT_PARM(var,idx) var[var[idx] < 0 ? 0 : idx]

static int mtu = 1500;
MODULE_PARM(mtu, "i");
MODULE_PARM_DESC(mtu, "Maximum transfer unit");

static int channel[MAX_PARM_DEVICES] = { 3, DEF_INTS };
MODULE_PARM(channel, PARM_MIN_MAX "i");
MODULE_PARM_DESC(channel, "Initial channel");

static char *essid[MAX_PARM_DEVICES] = { "test" };
MODULE_PARM(essid, PARM_MIN_MAX "s");
MODULE_PARM_DESC(essid, "Host AP's ESSID");

static int iw_mode[MAX_PARM_DEVICES] = { IW_MODE_MASTER, DEF_INTS };
MODULE_PARM(iw_mode, PARM_MIN_MAX "i");
MODULE_PARM_DESC(iw_mode, "Initial operation mode");

static int beacon_int[MAX_PARM_DEVICES] = { 100, DEF_INTS };
MODULE_PARM(beacon_int, PARM_MIN_MAX "i");
MODULE_PARM_DESC(beacon_int, "Beacon interval (1 = 1024 usec)");

static int dtim_period[MAX_PARM_DEVICES] = { 1, DEF_INTS };
MODULE_PARM(dtim_period, PARM_MIN_MAX "i");
MODULE_PARM_DESC(dtim_period, "DTIM period");

static int delayed_enable /* = 0 */;
MODULE_PARM(delayed_enable, "i");
MODULE_PARM_DESC(delayed_enable, "Delay MAC port enable until netdevice open");

static int disable_on_close /* = 0 */;
MODULE_PARM(disable_on_close, "i");
MODULE_PARM_DESC(disable_on_close, "Disable MAC port on netdevice close");


/* Ethernet-II snap header */
static unsigned char snap_header[] =
{ 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };

static const long freq_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				  2447, 2452, 2457, 2462, 2467, 2472, 2484 };
#define FREQ_COUNT (sizeof(freq_list) / sizeof(freq_list[0]))



#ifdef final_version
#define EXTRA_EVENTS_WTERR 0
#else
/* check WTERR events (Wait Time-out) in development versions */
#define EXTRA_EVENTS_WTERR HFA384X_EV_WTERR
#endif

#ifdef PRISM2_USE_TX_INTERRUPT
#define EXTRA_EVENTS_TX HFA384X_EV_TX
#define EXTRA_TX_CTRL HFA384X_TX_CTRL_TX_OK
#else
#define EXTRA_EVENTS_TX 0
#define EXTRA_TX_CTRL 0
#endif

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
#define EXTRA_EVENTS_CMD HFA384X_EV_CMD
#else
#define EXTRA_EVENTS_CMD 0
#endif

/* event mask, i.e., events that will result in an interrupt */
#define HFA384X_EVENT_MASK \
	(HFA384X_EV_TXEXC | HFA384X_EV_RX | HFA384X_EV_ALLOC | \
	HFA384X_EV_INFO | EXTRA_EVENTS_WTERR | EXTRA_EVENTS_TX | \
	EXTRA_EVENTS_CMD)

#define HFA384X_TX_CTRL_FLAGS \
	(HFA384X_TX_CTRL_802_11 | HFA384X_TX_CTRL_TX_EX | EXTRA_TX_CTRL)


/* ca. 1 usec */
#define HFA384X_CMD_BUSY_TIMEOUT 1000
#define HFA384X_BAP_BUSY_TIMEOUT 5000

/* ca. 10 usec */
#define HFA384X_INIT_TIMEOUT 50000
#define HFA384X_CMD_COMPL_TIMEOUT 20000
#define HFA384X_ALLOC_COMPL_TIMEOUT 1000


#define TX_TIMEOUT (2 * HZ)
#define PRISM2_MAX_FRAME_SIZE 2304
#define PRISM2_MIN_MTU 256
/* FIX: */
#define PRISM2_MAX_MTU (PRISM2_MAX_FRAME_SIZE - sizeof(snap_header))


/* prism2_wep.c */
void prism2_wep_init(local_info_t *local);
void prism2_wep_encrypt(local_info_t *local, u8 *buf, int len,
			u8 *key_override);
int prism2_wep_decrypt(local_info_t *local, u8 *buf, int len,
		       u8 *key_override);

/* prism2_ioctl.c */
static struct iw_statistics *prism2_get_wireless_stats(struct net_device *dev);
static int prism2_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);

/* prism2_proc.c */
void prism2_init_proc(local_info_t *local);
void prism2_remove_proc(local_info_t *local);
void prism2_cleanup_proc(void);



#ifndef final_version
/* magic value written to SWSUPPORT0 reg. for detecting whether card is still
 * present */
#define HFA384X_MAGIC 0x8A32
#endif


#if defined(PRISM2_PCCARD) || defined(PRISM2_PLX)
#define HFA384X_OUTB(v,a) outb((v), dev->base_addr + (a))
#define HFA384X_INB(a) inb(dev->base_addr + (a))
#define HFA384X_OUTW(v,a) outw((v), dev->base_addr + (a))
#define HFA384X_INW(a) inw(dev->base_addr + (a))
#define HFA384X_INSW(a, buf, wc) insw(dev->base_addr + (a), buf, wc)
#define HFA384X_OUTSW(a, buf, wc) outsw(dev->base_addr + (a), buf, wc)
#endif /* PRISM2_PCCARD || PRISM2_PLX */

#ifdef PRISM2_PCI
#define HFA384X_OUTB(v,a) writeb((v), dev->mem_start + (a))
#define HFA384X_INB(a) (u8) readb(dev->mem_start + (a))
#define HFA384X_OUTW(v,a) writew((v), dev->mem_start + (a))
#define HFA384X_INW(a) (u16) readw(dev->mem_start + (a))
#define HFA384X_OUTW_DATA(v,a) writew(cpu_to_le16(v), dev->mem_start + (a))
#define HFA384X_INW_DATA(a) (u16) le16_to_cpu(readw(dev->mem_start + (a)))
#endif /* PRISM2_PCI */



u16 hfa384x_read_reg(struct net_device *dev, u16 reg)
{
	return HFA384X_INW(reg);
}


int hfa384x_cmd(struct net_device *dev, u16 cmd, u16 param0, u16 *param1,
		u16 *resp0)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int tries, res;
	unsigned long flags;
	u16 reg;

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
	/* Wait for a possible pending transmit command completion without
	 * disabling IRQs */
	tries = 10000;
	for (;;) {
		while (local->last_txfid_idx != PRISM2_DUMMY_FID &&
		       tries > 0) {
			tries--;
			udelay(1);
		}

		if (tries == 0) {
			printk(KERN_DEBUG "hfa384x_cmd: timeout on waiting "
			       "pending TX\n");
			return -ETIMEDOUT;
		}

		spin_lock_irqsave(&local->cmdlock, flags);
		if (local->last_txfid_idx == PRISM2_DUMMY_FID)
			break;

		printk(KERN_DEBUG "hfa384x_cmd: new TX started - retrying\n");
		spin_unlock_irqrestore(&local->cmdlock, flags);
	}
#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */
	spin_lock_irqsave(&local->cmdlock, flags);
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	if (param1 != NULL)
		HFA384X_OUTW(*param1, HFA384X_PARAM1_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	/* wait for command completion */
	tries = HFA384X_CMD_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_EVSTAT_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout2 - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	if (resp0 != NULL)
		*resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
	res = (HFA384X_INW(HFA384X_STATUS_OFF) &
	       (BIT(14) | BIT(13) | BIT(12) | BIT(11) | BIT(10) | BIT(9) |
		BIT(8))) >> 8;
#ifndef final_version
	if (res) {
		u16 resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
		printk("%s: CMD=0x%04x => res=0x%02x, resp0=0x%04x\n",
		       dev->name, cmd, res, resp0);
	}
#endif

	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	spin_unlock_irqrestore(&local->cmdlock, flags);
	return res;
}


static int hfa384x_cmd_no_wait(struct net_device *dev, u16 cmd, u16 param0)
{
	int tries;
	u16 reg;

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	return 0;
}


/* Offset must be even */
int hfa384x_setup_bap(struct net_device *dev, u16 bap, u16 id, int offset)
{
	u16 o_off, tmp;
	int tries, ret = 0;

	if (offset % 2 || bap > 1)
		return -EINVAL;

	if (bap == BAP1)
		o_off = HFA384X_OFFSET1_OFF;
	else {
		o_off = HFA384X_OFFSET0_OFF;
		/* It should be OK to move data concurrently with BAP0 and
		 * BAP1, but setting BAPs concurrently seems to kill the card,
		 * so prevent interrupts when doing BAP0 setup. */
		cli();
	}

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk(KERN_DEBUG "%s: hfa384x_setup_bap - timeout1\n",
		       dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(id,
		     (bap == 1) ? HFA384X_SELECT1_OFF : HFA384X_SELECT0_OFF);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk(KERN_DEBUG "%s: hfa384x_setup_bap - timeout2\n",
		       dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(offset, o_off);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk(KERN_DEBUG "%s: hfa384x_setup_bap - timeout3\n",
		       dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}
#ifndef final_version
	else if (tmp & HFA384X_OFFSET_ERR) {
		printk(KERN_DEBUG "%s: hfa384x_setup_bap - offset error "
		       "(%d,%d,%d,0x%04x)\n",
		       dev->name, bap, id, offset, tmp);
		ret = tmp;
	}
#endif

 out:
	if (bap != BAP1)
		sti();

	return ret;
}


/* FIX: hfa384x_{from,to}_bap could be moved to prism2_{pccard,plx,pci}.o since
 * they differ on different hardware versions */

int hfa384x_from_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
#ifdef PRISM2_PCI
	u16 d_off;
	u16 *pos;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; len > 1; len -= 2)
		*pos++ = HFA384X_INW_DATA(d_off);
	if (len > 0)
		*((char *) pos) = HFA384X_INB(d_off);

	return 0;
#else /* PRISM2_PCI */
	u16 d_off;
	u16 *pos;

#ifdef PRISM2_EXTRA_FROM_BAP_TESTS

#ifndef final_version
	u16 o_off;
	int tries;

	o_off = (bap == 1) ? HFA384X_OFFSET1_OFF : HFA384X_OFFSET0_OFF;

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while ((HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY) && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - timeout\n",
		       dev->name, bap);
	else if (tries < HFA384X_BAP_BUSY_TIMEOUT)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - offset was "
		       "busy\n", dev->name, bap);
#endif

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; len > 1; len -= 2) {
#ifndef final_version
		if (HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY)
			printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - BAP "
			       "busy during read\n", dev->name, bap);
#endif
		*pos++ = __le16_to_cpu(HFA384X_INW(d_off));
	}
	if (len > 0)
		*((u8 *) pos) = HFA384X_INB(d_off);

#else /* PRISM2_EXTRA_FROM_BAP_TESTS */

	int word_count;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;

	word_count = len / 2;
	if (word_count)
		HFA384X_INSW(d_off, buf, word_count);

	if (len & 1)
		*((u8 *)(pos + word_count)) = HFA384X_INB(d_off);

#endif /* PRISM2_EXTRA_FROM_BAP_TESTS */

	return 0;
#endif /* PRISM2_PCI */
}


int hfa384x_to_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
#ifdef PRISM2_PCI
	u16 d_off;
	u16 *pos;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; len > 1; len -= 2)
		HFA384X_OUTW_DATA(*pos++, d_off);
	if (len > 0)
		HFA384X_OUTB(*((char *) pos), d_off);

	return 0;
#else /* PRISM2_PCI */
	u16 d_off;
	u16 *pos;
	int word_count;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	word_count = len / 2;
	if (word_count)
		HFA384X_OUTSW(d_off, buf, word_count);
	if (len & 1)
		HFA384X_OUTB(*((char *) (pos + word_count)), d_off);

	return 0;
#endif /* PRISM2_PCI */
}


int hfa384x_get_rid(struct net_device *dev, u16 rid, void *buf, int len,
		    int exact_len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, rlen = 0;
	struct hfa384x_rid_hdr rec;

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_get_rid: CMDCODE_ACCESS failed (res=%d)\n",
		       dev->name, res);
		return res;
	}

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	rlen = (__le16_to_cpu(rec.len) - 1) * 2;
	if (exact_len && rlen != len) {
		printk("RID len mismatch: rid=0x%04x, len=%d (expected %d)\n",
		       rid, rlen, len);
		res = -ENODATA;
		goto fail;
	}

	res = hfa384x_from_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0(2) failed\n",
		       dev->name);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res) {
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return res;
	}
	return rlen;
}


int hfa384x_set_rid(struct net_device *dev, u16 rid, void *buf, int len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	struct hfa384x_rid_hdr rec;

	rec.rid = __cpu_to_le16(rid);
	/* RID len in words and +1 for rec.rid */
	rec.len = __cpu_to_le16(len / 2 + len % 2 + 1);

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	res = hfa384x_to_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0(2) failed\n", dev->name);
		goto fail;
	}

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS_WRITE, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_set_rid: CMDCODE_ACCESS_WRITE failed "
		       "(res=%d)\n", dev->name, res);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res == -ETIMEDOUT)
		prism2_hw_reset(dev);

	return res;
}


static void hfa384x_disable_interrupts(struct net_device *dev)
{
	/* disable interrupts and clear event status */
	HFA384X_OUTW(0, HFA384X_INTEN_OFF);
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
}


static void hfa384x_enable_interrupts(struct net_device *dev)
{
	/* ack pending events and enable interrupts from selected events */
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
	HFA384X_OUTW(HFA384X_EVENT_MASK, HFA384X_INTEN_OFF);
}


static u16 hfa384x_allocate_fid(struct net_device *dev, int len)
{
	int tries;
	u16 fid;

	if (hfa384x_cmd(dev, HFA384X_CMDCODE_ALLOC, len, NULL, NULL)) {
		printk("%s: cannot allocate fid, len=%d\n", dev->name, len);
		return 0xffff;
	}

	tries = HFA384X_ALLOC_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_ALLOC) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		printk("%s: fid allocate, len=%d - timeout\n", dev->name, len);
		return 0xffff;
	}

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);
	HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);

	return fid;
}


int prism2_reset_port(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;

	if (!local->dev_enabled)
		return 0;

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0,
			  NULL, NULL);
	if (!res)
		res = hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0,
				  NULL, NULL);

	return res;
}


int prism2_set_tim(struct net_device *dev, u16 aid, int set)
{
	u16 tmp;

	if (set) {
		PDEBUG(DEBUG_PS2, "Setting TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(0x8000 | aid);
	} else {
		PDEBUG(DEBUG_PS2, "Clearing TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(aid);
	}
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFTIMCTRL, &tmp, 2)) {
		PDEBUG(DEBUG_PS, "TIM AID %d setting (set=%d) failed\n",
		       aid, set);
		return -1;
	}
	return 0;
}


static void prism2_get_version_info(struct net_device *dev, u16 rid,
				    const char *txt)
{
	struct hfa384x_comp_ident comp;

	if (hfa384x_get_rid(dev, rid, &comp, sizeof(comp), 1) < 0)
		printk("Could not get RID for component %s\n", txt);
	else
		printk("%s: %s: id=0x%02x v%d.%d.%d\n", dev->name, txt,
		       __le16_to_cpu(comp.id), __le16_to_cpu(comp.major),
		       __le16_to_cpu(comp.minor), __le16_to_cpu(comp.variant));
}


/* val is in host byte order */
int prism2_set_word(struct net_device *dev, int rid, u16 val)
{
	u16 tmp = cpu_to_le16(val);
	return hfa384x_set_rid(dev, rid, &tmp, 2);
}


int prism2_set_string(struct net_device *dev, int rid, const char *val)
{
	char buf[MAX_SSID_LEN + 2];
	int len;

	len = strlen(val);
	if (len > MAX_SSID_LEN)
		return -1;
	memset(buf, 0, sizeof(buf));
	buf[0] = len; /* little endian 16 bit word */
	memcpy(buf + 2, val, len);

	return hfa384x_set_rid(dev, rid, &buf, MAX_SSID_LEN + 2);
}


u16 prism2_get_porttype(local_info_t *local)
{
	if (local->iw_mode == IW_MODE_ADHOC && local->pseudo_adhoc)
		return HFA384X_PORTTYPE_PSEUDO_IBSS;
	if (local->iw_mode == IW_MODE_ADHOC)
		return HFA384X_PORTTYPE_IBSS;
	if (local->iw_mode == IW_MODE_INFRA)
		return HFA384X_PORTTYPE_BSS;
	return HFA384X_PORTTYPE_HOSTAP;
}


int prism2_set_encryption(local_info_t *local)
{
	u16 val;
	int i, keylen;

	for (i = 0; i < WEP_KEYS; i++) {
		keylen = local->wep_key_len[i];
		if (keylen > 0 && keylen < 5)
			keylen = 6; /* first 5 octets */
		else  if (keylen > 5)
			keylen = WEP_KEY_LEN + 1; /* first 13 octets */
		if (hfa384x_set_rid(local->dev,
				    HFA384X_RID_CNFDEFAULTKEY0 + i,
				    local->wep_keys[i], keylen)) {
			printk(KERN_DEBUG "Could not set key %d (len=%d)\n",
			       i, keylen);
			goto fail;
		}
	}
	if (prism2_set_word(local->dev, HFA384X_RID_CNFWEPDEFAULTKEYID,
			    local->wep_tx_key)) {
		printk(KERN_DEBUG "Could not set default keyid %d\n",
		       local->wep_tx_key);
		goto fail;
	}

	if (hfa384x_get_rid(local->dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) <
	    0) {
		printk(KERN_DEBUG "Could not read current WEP flags.\n");
		goto fail;
	}
	le16_to_cpus(&val);

	if (local->open_wep)
		val &= ~HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;
	else
		val |= HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;

	if (local->use_wep) {
		val |= HFA384X_WEPFLAGS_PRIVACYINVOKED;
		if (local->iw_mode == IW_MODE_MASTER || local->host_encrypt)
			val |= HFA384X_WEPFLAGS_HOSTENCRYPT;
		else
			val &= ~HFA384X_WEPFLAGS_HOSTENCRYPT;
		if (local->host_decrypt)
			val |= HFA384X_WEPFLAGS_HOSTDECRYPT;
		else
			val &= ~HFA384X_WEPFLAGS_HOSTDECRYPT;
	} else
		val &= ~(HFA384X_WEPFLAGS_PRIVACYINVOKED |
			 HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED);

	if (prism2_set_word(local->dev, HFA384X_RID_CNFWEPFLAGS, val)) {
		printk(KERN_DEBUG "Could not write new WEP flags (0x%x)\n",
		       val);
		goto fail;
	}

	return 0;

 fail:
	printk(KERN_DEBUG "%s: encryption setup failed\n", local->dev->name);
	return -1;
}


static int prism2_setup_rids(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 tmp;
	int ret = 0;

	tmp = prism2_get_porttype(local);
	ret = prism2_set_word(dev, HFA384X_RID_CNFPORTTYPE, tmp);
	if (ret) {
		printk("%s: Port type setting to %d failed\n", dev->name, tmp);
		goto fail;
	}

	/* Setting SSID to empty string seems to kill the card in Host AP mode
	 */
	if (local->iw_mode != IW_MODE_MASTER || local->essid[0] != '\0') {
		ret = prism2_set_string(dev, HFA384X_RID_CNFOWNSSID,
					local->essid);
		if (ret) {
			printk("%s: AP own SSID setting failed\n", dev->name);
			goto fail;
		}
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFMAXDATALEN,
			      PRISM2_DATA_MAXLEN);
	if (ret) {
		printk("%s: MAC data length setting to %d failed\n",
		       dev->name, PRISM2_DATA_MAXLEN);
		goto fail;
	}

	if (hfa384x_get_rid(dev, HFA384X_RID_CHANNELLIST, &tmp, 2, 1) < 0) {
		printk("%s: Channel list read failed\n", dev->name);
		ret = -EINVAL;
		goto fail;
	}
	local->channel_mask = __le16_to_cpu(tmp);

	if (local->channel < 1 || local->channel > FREQ_COUNT ||
	    !(local->channel_mask & (1 << (local->channel - 1)))) {
		printk(KERN_WARNING "%s: Channel setting out of range "
		       "(%d)!\n", dev->name, local->channel);
		ret = -EBUSY;
		goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFOWNCHANNEL, local->channel);
	if (ret) {
		printk("%s: Channel setting to %d failed\n",
		       dev->name, local->channel);
		goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFBEACONINT,
			      local->beacon_int);
	if (ret) {
		printk("%s: Beacon interval setting to %d failed\n",
		       dev->name, local->beacon_int);
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFBASICRATES,
			      HFA384X_RATES_1MBPS | HFA384X_RATES_2MBPS);
	if (ret) {
		printk("%s: Basic rates setting failed\n", dev->name);
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFSUPPORTEDRATES,
			      HFA384X_RATES_1MBPS | HFA384X_RATES_2MBPS |
			      HFA384X_RATES_5MBPS | HFA384X_RATES_11MBPS);
	if (ret) {
		printk("%s: Supported rates setting failed\n", dev->name);
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CNFOWNDTIMPERIOD,
			      local->dtim_period);
	if (ret) {
		printk("%s: DTIM period setting to %d failed\n",
		       dev->name, local->dtim_period);
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = prism2_set_string(dev, HFA384X_RID_CNFDESIREDSSID, local->essid);
	if (ret) {
		printk("%s: Desired SSID setting failed\n", dev->name);
		goto fail;
	}

	/* Setup TXRateControl, defaults to allow use of 1, 2, 5.5, and
	 * 11 Mbps in automatic TX rate fallback */
	if (local->tx_rate_control == 0) {
		local->tx_rate_control =
			HFA384X_RATES_1MBPS |
			HFA384X_RATES_2MBPS |
			HFA384X_RATES_5MBPS |
			HFA384X_RATES_11MBPS;
	}
	ret = prism2_set_word(dev, HFA384X_RID_TXRATECONTROL,
			      local->tx_rate_control);
	if (ret) {
		printk("%s: TXRateControl setting to %d failed\n",
		       dev->name, local->tx_rate_control);
		goto fail;
	}

	ret = prism2_set_word(dev, HFA384X_RID_CREATEIBSS, 1);
	if (ret) {
		printk("%s: Create IBSS setting to 1 failed\n", dev->name);
	}

	if (local->name_set)
		(void) prism2_set_string(dev, HFA384X_RID_CNFOWNNAME,
					 local->name);

	if (prism2_set_encryption(local)) {
		printk(KERN_INFO "%s: could not configure encryption\n",
		       dev->name);
	}

 fail:
	return ret;
}


static int prism2_hw_init(struct net_device *dev, int initial)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret, i, len;

	PDEBUG(DEBUG_FLOW, "prism2_hw_init()\n");

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
	local->last_txfid_idx = PRISM2_DUMMY_FID;
#endif

	/* initialize HFA 384x */
	ret = hfa384x_cmd_no_wait(dev, HFA384X_CMDCODE_INIT, 0);
	if (ret) {
		printk("%s: first command failed - is the card compatible?\n",
		       dev_info);
		goto failed;
	}
	i = HFA384X_INIT_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) && i > 0) {
		i--;
		udelay(10);
	}
	if (i == 0) {
		printk("%s: card initialization timed out\n", dev_info);
		goto failed;
	}
	printk(KERN_DEBUG "prism2_hw_config: initialized in %d iterations\n",
	       HFA384X_INIT_TIMEOUT - i);
	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	hfa384x_disable_interrupts(dev);

#ifndef final_version
	HFA384X_OUTW(HFA384X_MAGIC, HFA384X_SWSUPPORT0_OFF);
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		printk("SWSUPPORT0 write/read failed: %04X != %04X\n",
		       HFA384X_INW(HFA384X_SWSUPPORT0_OFF), HFA384X_MAGIC);
		goto failed;
	}
#endif

	/* allocate TX FIDs */
	len = local->is_symbol ? 1600 : PRISM2_TXFID_LEN;
	for (i = 0; i < PRISM2_TXFID_COUNT; i++) {
		local->txfid[i] = hfa384x_allocate_fid(dev, len);
		if (local->txfid[i] == 0xffff)
			goto failed;
		local->intransmitfid[i] = PRISM2_TXFID_EMPTY;
	}

	if (initial) {
		/* get card version information */
		prism2_get_version_info(dev, HFA384X_RID_NICID, "NIC");
		prism2_get_version_info(dev, HFA384X_RID_PRIID, "PRI");
		prism2_get_version_info(dev, HFA384X_RID_STAID, "STA");

		if (hfa384x_get_rid(dev, HFA384X_RID_CNFOWNMACADDR,
				    &dev->dev_addr, 6, 1) < 0) {
			printk("%s: could not get own MAC address\n",
			       dev->name);
		}
#ifdef PRISM2_HOSTAPD
		memcpy(local->apdev->dev_addr, dev->dev_addr, ETH_ALEN);
#endif /* PRISM2_HOSTAPD */
	}

	prism2_setup_rids(dev);

	/* MAC is now configured, but port 0 is not yet enabled */
	return 0;

 failed:
	printk(KERN_WARNING "%s: Initialization failed\n", dev_info);
	return 1;
}


static int prism2_hw_enable(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0, NULL, NULL)) {
		printk("%s: MAC port 0 enabling failed\n", dev->name);
		return 1;
	}

	/* at least D-Link DWL-650 seems to require additional port reset
	 * before it starts acting as an AP, so reset port automatically
	 * here just in case */
	if (prism2_reset_port(dev)) {
		printk("%s: MAC port 0 reseting failed\n", dev->name);
		return 1;
	}

	local->hw_ready = 1;
	local->hw_reset_tries = 0;
	hfa384x_enable_interrupts(dev);

	return 0;
}


int prism2_hw_config(struct net_device *dev, int initial)
{
	if (prism2_hw_init(dev, initial))
		return 1;

	if (!initial || !delayed_enable) {
		local_info_t *local = (local_info_t *) dev->priv;
		local->dev_enabled = 1;
		return prism2_hw_enable(dev);
	}

	return 0;
}


void prism2_hw_shutdown(struct net_device *dev, int no_disable)
{
	local_info_t *local = (local_info_t *) dev->priv;

	local->hw_ready = 0;
	if (local->func->card_present && !local->func->card_present(local)) {
		printk(KERN_DEBUG "%s: card already removed or not configured "
		       "during shutdown\n", dev->name);
		return;
	}

	hfa384x_disable_interrupts(dev);

	if (!no_disable &&
	    hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0, NULL, NULL))
		printk(KERN_WARNING "%s: Shutdown failed\n", dev_info);
}


void prism2_hw_reset(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

#if 0
	static long last_reset = 0;

	/* do not reset card more than once per second to avoid ending up in a
	 * busy loop reseting the card */
	if (last_reset + HZ >= jiffies)
		return;
	last_reset = jiffies;
#endif

	if (local->hw_resetting) {
		printk(KERN_WARNING "%s: %s: already resetting card - "
		       "ignoring reset request\n", dev_info, dev->name);
		return;
	}

	local->hw_reset_tries++;
	if (local->hw_reset_tries > 10) {
		printk(KERN_WARNING "%s: too many reset tries, skipping\n",
		       dev->name);
		return;
	}

	printk(KERN_WARNING "%s: %s: resetting card\n", dev_info, dev->name);
	local->hw_resetting = 1;
	hfa384x_disable_interrupts(dev);
	if (local->func->cor_sreset)
		local->func->cor_sreset(local);
	prism2_hw_shutdown(dev, 1);
	prism2_hw_config(dev, 0);
	local->hw_resetting = 0;
}


static struct net_device_stats *prism2_get_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	if (local->dev != dev) {
		prism2_wds_info_t *wds = (prism2_wds_info_t *) dev;
		return &wds->stats;
	}
	return &local->stats;
}


/* wake all netif queues in use */
static void prism2_netif_wake_queues(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	prism2_wds_info_t *wds;

	if (local->dev)
		netif_wake_queue(local->dev);

#ifdef PRISM2_HOSTAPD
	if (local->apdev)
		netif_wake_queue(local->apdev);
#endif /* PRISM2_HOSTAPD */

	wds = local->wds;
	while (wds != NULL) {
		netif_wake_queue(&wds->dev);
		wds = wds->next;
	}
}


/* stop all netif queues in use */
static void prism2_netif_stop_queues(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	prism2_wds_info_t *wds;

	if (local->dev)
		netif_stop_queue(local->dev);

#ifdef PRISM2_HOSTAPD
	if (local->apdev)
		netif_stop_queue(local->apdev);
#endif /* PRISM2_HOSTAPD */

	spin_lock(&local->wdslock);
	wds = local->wds;
	while (wds != NULL) {
		netif_stop_queue(&wds->dev);
		wds = wds->next;
	}
	spin_unlock(&local->wdslock);
}


static int prism2_open(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
#ifdef PRISM2_PCCARD
	local->link->open++;
#endif /* PRISM2_PCCARD */

	PDEBUG(DEBUG_FLOW, "%s: prism2_open\n", dev->name);

	MOD_INC_USE_COUNT;

	if (!local->dev_enabled && prism2_hw_enable(dev)) {
		printk(KERN_WARNING "%s: could not enable MAC port\n",
		       dev->name);
		MOD_DEC_USE_COUNT;
		return 1;
	}
	local->dev_enabled = 1;

	netif_device_attach(dev);
	netif_start_queue(dev);

	return 0;
}


static int prism2_close(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	PDEBUG(DEBUG_FLOW, "%s: prism2_close\n", dev->name);

#ifdef PRISM2_PCCARD
	if (!local->link->open) {
		printk(KERN_WARNING "%s: prism2_close(): link not open?!\n",
		       dev->name);
		return 0;
	}

	local->link->open--;
#endif /* PRISM2_PCCARD */

	if (disable_on_close) {
		local->dev_enabled = 0;
		prism2_hw_shutdown(dev, 0);
	}

	if (netif_running(dev)) {
		netif_stop_queue(dev);
		netif_device_detach(dev);
	}
#ifdef PRISM2_PCCARD
	else if (local->link->state & DEV_STALE_CONFIG)
			mod_timer(&local->link->release, jiffies + HZ / 20);
#endif /* PRISM2_PCCARD */

	MOD_DEC_USE_COUNT;

	return 0;
}


static int prism2_change_mtu(struct net_device *dev, int new_mtu)
{
	if (new_mtu < PRISM2_MIN_MTU || new_mtu > PRISM2_MAX_MTU)
		return -EINVAL;

	dev->mtu = new_mtu;
	return 0;
}


#ifdef HAVE_TX_TIMEOUT
static void prism2_tx_timeout(struct net_device *dev)
{
	printk(KERN_WARNING "%s: %s Tx timed out! Resetting card\n",
	       dev_info, dev->name);

	prism2_hw_reset(dev);

	if (netif_queue_stopped(dev)) {
		int i;
		local_info_t *local = (local_info_t *) dev->priv;

		for (i = 0; i < PRISM2_TXFID_COUNT; i++)
			if (local->intransmitfid[i] == PRISM2_TXFID_EMPTY) {
				PDEBUG(DEBUG_EXTRA, "prism2_tx_timeout: "
				       "wake up queue\n");
				netif_wake_queue(dev);
				break;
			}
	}
}
#endif /* HAVE_TX_TIMEOUT */


int prism2_get_txfid_idx(local_info_t *local)
{
	int idx, end;

	spin_lock(&local->txfidlock);
	end = idx = local->next_txfid;
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			local->intransmitfid[idx] = PRISM2_TXFID_RESERVED;
			spin_unlock(&local->txfidlock);
			return idx;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != end);
	spin_unlock(&local->txfidlock);

	PDEBUG(DEBUG_EXTRA2, "prism2_get_txfid_idx: no room in txfid buf: "
	       "packet dropped\n");
	local->stats.tx_dropped++;

	return -1;
}


#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT

/* Called only from software IRQ */
int prism2_transmit(struct net_device *dev, int idx)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	unsigned long flags;

	/* The driver tries to stop netif queue so that there would not be
	 * more than one attempt to transmit frames going on; check that this
	 * is really the case */
	if (local->last_txfid_idx != PRISM2_DUMMY_FID) {
		int tries = 1000;
		while (tries > 0 && local->last_txfid_idx != PRISM2_DUMMY_FID)
		{
			udelay(10);
			tries--;
		}

		if (local->last_txfid_idx != PRISM2_DUMMY_FID) {
			printk(KERN_DEBUG "%s: timeout on waiting pending "
			       "transmit command..\n", dev->name);
			return 1;
		}
		printk(KERN_DEBUG "%s: %d x 10 usec cleared pending TX\n",
		       dev->name, 1000 - tries);
	}

	/* stop the queue for the time that last_txfid_idx is in use */
	prism2_netif_stop_queues(dev);

	local->last_txfid_idx = idx;

	/* transmit packet */
	spin_lock_irqsave(&local->cmdlock, flags);
	res = hfa384x_cmd_no_wait(
		dev,
		HFA384X_CMDCODE_TRANSMIT | HFA384X_CMD_TX_RECLAIM,
		local->txfid[idx]);
	spin_unlock_irqrestore(&local->cmdlock, flags);

	if (res) {
		struct net_device_stats *stats;
		printk(KERN_DEBUG "%s: prism2_transmit: CMDCODE_TRANSMIT "
		       "failed (res=%d)\n", dev->name, res);
		stats = prism2_get_stats(dev);
		stats->tx_dropped++;
		prism2_netif_wake_queues(dev);
		return 1;
	}
	dev->trans_start = jiffies;

	/* Since we did not wait for command completion, the card continues
	 * to process on the background and we will finish handling when
	 * command completion event is handled (prism2_cmd_ev() function) */

	return 0;
}


/* Called only from hardware IRQ */
static void prism2_cmd_ev(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int idx = local->last_txfid_idx;
	u16 resp0;

	if (local->last_txfid_idx == PRISM2_DUMMY_FID) {
		/* no pending transmit command; should not happen since we
		 * wait for command completion (and ACK event) with every other
		 * command */
		printk(KERN_DEBUG "%s: CMD EV - no pending transmit\n",
		       dev->name);
		return;
	}

	local->last_txfid_idx = PRISM2_DUMMY_FID;

	resp0 = HFA384X_INW(HFA384X_RESP0_OFF);

	if (netif_queue_stopped(dev)) {
		/* last_txfid_idx is free for next TX, so wake up queue that
		 * was stopped in prism2_transmit() */
		prism2_netif_wake_queues(dev);
	}

	/* FIX: is some locking needed for txfid data? */

	/* With reclaim, Resp0 contains new txfid for transmit; the old txfid
	 * will be automatically allocated for the next TX frame */
	local->intransmitfid[idx] = resp0;

	PDEBUG(DEBUG_FID, "%s: prism2_cmd_ev: txfid[%d]=0x%04x, resp0=0x%04x, "
	       "transmit_txfid=0x%04x\n", dev->name, idx, local->txfid[idx],
	       resp0, local->intransmitfid[local->next_txfid]);

	idx++;
	if (idx >= PRISM2_TXFID_COUNT)
		idx = 0;
	local->next_txfid = idx;

	/* check if all TX buffers are occupied */
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			return;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);

	/* no empty TX buffers, stop queue */
	prism2_netif_stop_queues(dev);
}

#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */

/* Called only from software IRQ */
int prism2_transmit(struct net_device *dev, int idx)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, end;
	unsigned long flags;
	u16 resp0;

	spin_lock_irqsave(&local->txfidlock, flags);

	/* transmit packet */
	res = hfa384x_cmd(dev,
			  HFA384X_CMDCODE_TRANSMIT | HFA384X_CMD_TX_RECLAIM,
			  local->txfid[idx], NULL, &resp0);
	if (res) {
		struct net_device_stats *stats;
		spin_unlock_irqrestore(&local->txfidlock, flags);
		printk("%s: prism2_transmit: CMDCODE_TRANSMIT failed "
		       "(res=%d, resp0=0x%04x)\n", dev->name, res, resp0);
		stats = prism2_get_stats(dev);
		stats->tx_dropped++;
		return 1;
	}
	dev->trans_start = jiffies;

	/* With reclaim, Resp0 contains new txfid for transmit; the old txfid
	 * will be automatically allocated for the next TX frame */
	local->intransmitfid[idx] = resp0;

	PDEBUG(DEBUG_FID, "prism2_transmit: [%d] txfid=0x%04x, "
	       "transmit_txfid=0x%04x\n", idx, local->txfid[idx],
	       local->intransmitfid[local->next_txfid]);

	idx++;
	if (idx >= PRISM2_TXFID_COUNT)
		idx = 0;
	end = local->next_txfid = idx;

	/* check if all TX buffers are occupied */
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			spin_unlock_irqrestore(&local->txfidlock, flags);
			return 0;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != end);
	spin_unlock_irqrestore(&local->txfidlock, flags);

	/* no empty TX buffers, stop queue */
	prism2_netif_stop_queues(dev);

	return 0;
}

#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */


void prism2_dump_tx_header(const char *name, const struct hfa384x_tx_frame *tx)
{
	u16 fc;

	printk(KERN_DEBUG "%s: TX status=0x%04x retry_count=%d tx_rate=%d "
	       "tx_control=0x%04x; jiffies=%ld\n",
	       name, __le16_to_cpu(tx->status), tx->retry_count, tx->tx_rate,
	       __le16_to_cpu(tx->tx_control), jiffies);

	fc = __le16_to_cpu(tx->frame_control);
	printk(KERN_DEBUG "   FC=0x%04x (type=%d:%d) dur=0x%04x seq=0x%04x "
	       "data_len=%d%s%s\n",
	       fc, WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(tx->duration_id), __le16_to_cpu(tx->seq_ctrl),
	       __le16_to_cpu(tx->data_len),
	       fc & WLAN_FC_TODS ? " [ToDS]" : "",
	       fc & WLAN_FC_FROMDS ? " [FromDS]" : "");

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(tx->addr1), MAC2STR(tx->addr2), MAC2STR(tx->addr3),
	       MAC2STR(tx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(tx->dst_addr), MAC2STR(tx->src_addr),
	       __be16_to_cpu(tx->len));
}

/* Called only from software IRQ */
static int prism2_tx(struct sk_buff *skb, struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, idx = -1, ret = 1, data_len;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta = NULL;
	unsigned long flags;
	u16 fc;
	prism2_wds_info_t *wds = NULL;
	struct net_device_stats *stats;
	u8 *wepbuf = NULL;
	int wepbuf_len = 0;
	int host_encrypt;

	stats = prism2_get_stats(dev);

	if ((local->func->card_present && !local->func->card_present(local)) ||
	    !local->hw_ready) {
		printk(KERN_DEBUG "%s: prism2_tx: hw not ready - skipping\n",
		       dev->name);
		ret = 0;
		goto fail;
	}

	if (skb->len < ETH_HLEN) {
		printk(KERN_DEBUG "%s: prism2_tx: short skb (len=%d)\n",
		       dev->name, skb->len);
		ret = 0;
		goto fail;
	}

	if (local->dev != dev)
		wds = (prism2_wds_info_t *) dev;

	host_encrypt = local->use_wep &&
		(local->iw_mode == IW_MODE_MASTER || wds ||
		 local->host_encrypt);

	/* Incoming skb->data: dst_addr[6], src_addr[6], proto[2], payload
	 * ==>
	 * Prism2 TX frame with 802.11 header:
	 * txdesc (address order depending on used mode; includes dst_addr and
	 * src_addr), snap_header[6], proto[2], payload {, possible addr4[6]}
	 */

	memset(&txdesc, 0, sizeof(txdesc));

	txdesc.tx_control = __cpu_to_le16(local->tx_control);

	/* Length of data after txdesc */
	data_len = skb->len - 2 * ETH_ALEN + sizeof(snap_header);

	fc = (WLAN_FC_TYPE_DATA << 2) | (WLAN_FC_STYPE_DATA << 4);
	if (wds) {
		/* Note! Prism2 station firmware has problems with sending real
		 * 802.11 frames with four addresses; until these problems can
		 * be fixed or worked around, 4-addr frames needed for WDS are
		 * using incompatible format: FromDS flag is not set and the
		 * fourth address is added after the frame payload; it is
		 * assumed, that the receiving station knows how to handle this
		 * frame format */

#ifdef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		fc |= WLAN_FC_FROMDS | WLAN_FC_TODS;
		/* From&To DS: Addr1 = RA, Addr2 = TA, Addr3 = DA, Addr4 = SA
		 */
		memcpy(&txdesc.addr1, wds->remote_addr, ETH_ALEN);
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
		memcpy(&txdesc.addr4, skb->data + ETH_ALEN, ETH_ALEN);
#else /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		/* bogus 4-addr format to workaround Prism2 station f/w bug */
		fc |= WLAN_FC_TODS;
		/* From DS: Addr1 = DA (used as RA),
		 * Addr2 = BSSID (used as TA), Addr3 = SA (used as DA) */
		memcpy(&txdesc.addr1, wds->remote_addr, ETH_ALEN);
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
		/* SA from skb->data + ETH_ALEN will be added after frame
		 * payload */
		data_len += ETH_ALEN;
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		memcpy(&txdesc.dst_addr, &txdesc.addr3, ETH_ALEN);
		memcpy(&txdesc.src_addr, &txdesc.addr2, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_MASTER) {
		fc |= WLAN_FC_FROMDS;
		/* From DS: Addr1 = DA, Addr2 = BSSID, Addr3 = SA */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		/* FIX - addr2 replaced by f/w, so no need to fill it now(?) */
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data + ETH_ALEN, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_INFRA) {
		fc |= WLAN_FC_TODS;
		/* To DS: Addr1 = BSSID, Addr2 = SA, Addr3 = DA;
		 * firmware sets BSSID */
		/* memcpy(&txdesc.addr1, local->bssid, ETH_ALEN); */
		memcpy(&txdesc.addr2, skb->data + ETH_ALEN, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_ADHOC) {
		/* not From/To DS: Addr1 = DA, Addr2 = SA, Addr3 = BSSID */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		memcpy(&txdesc.addr2, skb->data + ETH_ALEN, ETH_ALEN);
		memcpy(&txdesc.addr3, local->bssid, ETH_ALEN);
	}

	if (local->use_wep) {
		fc |= WLAN_FC_ISWEP;
		if (host_encrypt)
			data_len += 8; /* IV + ICV */
	}

	txdesc.frame_control = __cpu_to_le16(fc);
	txdesc.data_len = __cpu_to_le16(data_len);
	txdesc.len = __cpu_to_be16(data_len);
	if (!wds)
		memcpy(&txdesc.dst_addr, skb->data, 2 * ETH_ALEN);

	if (local->iw_mode == IW_MODE_MASTER &&
	    (txdesc.addr1[0] & 0x01) == 0x00) {
		/* unicast packet - check whether destination STA is
		 * associated */
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta = ap_get_sta(local->ap, txdesc.addr1);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (!sta && !wds) {
			/* remove FromDS flag from (pseudo) ad-hoc style
			 * communication between APs */
			txdesc.frame_control &=
				~(__constant_cpu_to_le16(WLAN_FC_FROMDS));

			printk(KERN_DEBUG "AP: packet to non-associated STA "
			       MACSTR "\n", MAC2STR(txdesc.addr1));
		}
	}

	/* Set tx_rate if using host-based TX rate control */
	if (!local->fw_tx_rate_control && sta != NULL) {
		txdesc.tx_rate = sta->tx_rate;
		sta->tx_count[sta->tx_rate_idx]++;
		sta->tx_since_last_failure++;
		if (sta->tx_since_last_failure > WLAN_RATE_UPDATE_COUNT &&
		    sta->tx_rate_idx < sta->tx_max_rate) {
			/* use next higher rate */
			while (sta->tx_rate_idx < sta->tx_max_rate) {
				sta->tx_rate_idx++;
				if (sta->tx_supp_rates &
				    (1 << sta->tx_rate_idx))
					break;
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate raised to"
			       " %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
			sta->tx_since_last_failure = 0;
		}
	}

	if (sta && sta->flags & WLAN_STA_PS) {
		struct sta_buffer_frame *fbuf;
		int set_tim = 0;

		/* STA in PS mode, buffer frame for later delivery */
		if (sta->buffer_count >= STA_MAX_TX_BUFFER) {
			PDEBUG(DEBUG_PS, "No more space in this STA's PS mode "
			       "buffer\n");
			ret = 0; /* drop packet */
			stats->tx_dropped++;
			goto fail;
		}

		fbuf = (struct sta_buffer_frame *)
			kmalloc(sizeof(struct sta_buffer_frame), GFP_ATOMIC);
		if (fbuf == NULL) {
			printk("Malloc failed for STA's PS mode buffer\n");
			goto fail;
		}
		fbuf->next = NULL;
		fbuf->rx_time = jiffies;
		memcpy(&fbuf->txdesc, &txdesc, sizeof(txdesc));
		fbuf->skb = skb;
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta->buffer_count++;
		if (sta->tx_buf_head == NULL) {
			sta->tx_buf_head = sta->tx_buf_tail = fbuf;
			set_tim = 1;
		} else {
			sta->tx_buf_tail->next = fbuf;
			sta->tx_buf_tail = fbuf;
		}
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (set_tim)
			prism2_set_tim(dev, sta->aid, 1);

		atomic_dec(&sta->users);

		/* do not free skb here, it will be freed when the buffered
		 * frame is sent/timed out */
		ret = 0;
		goto tx_exit;
	}

	if (host_encrypt) {
		u8 *pos;
		wepbuf_len = 4 + sizeof(snap_header) +
			skb->len - 2 * ETH_ALEN + 4;
#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		if (wds)
			wepbuf_len += ETH_ALEN; /* Addr4 */
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		wepbuf = (u8 *) kmalloc(wepbuf_len, GFP_ATOMIC);
		if (wepbuf == NULL) {
			printk(KERN_DEBUG "%s: could not allocate TX wepbuf\n",
			       dev->name);
			goto fail;
		}

		pos = wepbuf + 4;
		memcpy(pos, snap_header, sizeof(snap_header));
		pos += sizeof(snap_header);
		memcpy(pos, skb->data + 2 * ETH_ALEN, skb->len - 2 * ETH_ALEN);
#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		if (wds) {
			pos += skb->len - 2 * ETH_ALEN;
			memcpy(pos, skb->data + ETH_ALEN, ETH_ALEN);
		}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

		prism2_wep_encrypt(local, wepbuf, wepbuf_len - 8, NULL);
	}

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		goto fail;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR)
		prism2_dump_tx_header(dev->name, &txdesc);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res && !wepbuf)
		res = hfa384x_to_bap(dev, BAP0, &snap_header,
				     sizeof(snap_header));
#ifdef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (!res && !wepbuf)
		res = hfa384x_to_bap(dev, BAP0, skb->data + 2 * ETH_ALEN,
				     skb->len - 2 * ETH_ALEN);
#else /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
	if (!res && !wepbuf) {
		int wlen = skb->len - 2 * ETH_ALEN;
		int is_odd = wlen & 1;
		if (wds && is_odd)
			wlen--; /* need to avoid using odd offset */
		res = hfa384x_to_bap(dev, BAP0, skb->data + 2 * ETH_ALEN,
				     wlen);

		/* add addr4 (SA) to bogus frame format if WDS is used */
		if (!res && wds && is_odd) {
			char tmpbuf[ETH_ALEN + 1];
			tmpbuf[0] = *(skb->data + skb->len - 1);
			memcpy(tmpbuf + 1, skb->data + ETH_ALEN, ETH_ALEN);
			res = hfa384x_to_bap(dev, BAP0, tmpbuf, ETH_ALEN + 1);
		} else if (!res && wds) {
			res = hfa384x_to_bap(dev, BAP0, skb->data + ETH_ALEN,
					     ETH_ALEN);
		}
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

	if (!res && wepbuf)
		res = hfa384x_to_bap(dev, BAP0, wepbuf, wepbuf_len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);
	if (res) {
		printk("%s: prism2_tx - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		prism2_hw_reset(dev);
		goto fail;
	}

	stats->tx_bytes += data_len + 36;

	if (sta) {
		sta->tx_packets++;
		sta->tx_bytes += data_len + 36;
		sta->last_tx = jiffies;
	}

	ret = 0;

 fail:
	if (wepbuf)
		kfree(wepbuf);

	if (sta)
		atomic_dec(&sta->users);

	if (!ret)
		dev_kfree_skb(skb);

 tx_exit:
	return ret;
}


#ifdef PRISM2_HOSTAPD

/* Called only from software IRQ */
static int prism2_tx_80211(struct sk_buff *skb, struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct hfa384x_tx_frame txdesc;
	int hdr_len, data_len, ret = 1, idx, res;
	u16 fc;

	if ((local->func->card_present && !local->func->card_present(local)) ||
	    !local->hw_ready) {
		printk(KERN_DEBUG "%s: prism2_tx_80211: hw not ready - "
		       "skipping\n", dev->name);
		ret = 0;
		goto tx_exit;
	}

	if (skb->len < 24) {
		printk(KERN_DEBUG "%s: prism2_tx_80211: short skb (len=%d)\n",
		       dev->name, skb->len);
		ret = 0;
		goto tx_exit;
	}

	memset(&txdesc, 0, sizeof(txdesc));
	txdesc.tx_control = __cpu_to_le16(local->tx_control);
	/* txdesc.tx_rate might need to be set if f/w does not select suitable
	 * TX rate */

	/* skb->data starts with txdesc->frame_control */
	hdr_len = 24;
	memcpy(&txdesc.frame_control, skb->data, hdr_len);
 	fc = __le16_to_cpu(txdesc.frame_control);
	if (WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA &&
	    (fc & WLAN_FC_FROMDS) && (fc & WLAN_FC_TODS) && skb->len >= 30) {
		/* Addr4 */
		memcpy(txdesc.addr4, skb->data + hdr_len, ETH_ALEN);
		hdr_len += ETH_ALEN;
	}
	
	data_len = skb->len - hdr_len;
	txdesc.data_len = __cpu_to_le16(data_len);
	txdesc.len = __cpu_to_be16(data_len);

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		goto fail;

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data + hdr_len,
				     skb->len - hdr_len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);
	if (res) {
		printk("%s: prism2_tx_80211 - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		prism2_hw_reset(dev);
		goto fail;
	}

	ret = 0;

 fail:
	if (!ret)
		dev_kfree_skb(skb);

 tx_exit:
	return ret;
}

#endif /* PRISM2_HOSTAPD */


#if defined(PRISM2_MONITOR) || defined(PRISM2_HOSTAPD)

enum {
	PRISM2_RX_MONITOR, PRISM2_RX_MGMT, PRISM2_RX_NON_ASSOC,
	PRISM2_RX_NULLFUNC_ACK
};

/* Called only from hardware IRQ */
static int prism2_rx_80211(struct net_device *dev,
			   struct hfa384x_rx_frame *rxdesc,
			   int type, char *extra, int extra_len)
{
	struct sk_buff *skb;
	int res, hdrlen;
	u16 len, fc;
	unsigned char *pos;

	hdrlen = 24;
	fc = __le16_to_cpu(rxdesc->frame_control);
	if (WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA &&
	    (fc & WLAN_FC_FROMDS) && (fc & WLAN_FC_TODS))
		hdrlen += 6; /* Addr4 */

	if (extra) {
		/* set 802.11 protocol version to 3 to indicate extra data
		 * after the payload */
		fc |= WLAN_FC_PVER;
		rxdesc->frame_control = __cpu_to_le16(fc);
	}

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN || extra_len > 65535) {
		printk(KERN_WARNING "%s: prism2_rx_80211: len(%d) > "
		       "MAX(%d)\n", dev->name, len, PRISM2_DATA_MAXLEN);
		return 0;
	}

	skb = dev_alloc_skb(hdrlen + len + 2 + extra_len + 2);
	if (!skb) {
		printk(KERN_WARNING "%s: prism2_rx_80211 cannot allocate "
		       "buffer\n", dev->name);
		return 0;
	}

	/* align IP on 16b boundary */
	skb_reserve(skb, 2);
	dev->last_rx = jiffies;

	skb_put(skb, hdrlen + len);
	pos = skb->data;
	memcpy(pos, &rxdesc->frame_control, hdrlen);
	pos += hdrlen;
	res = hfa384x_from_bap(dev, BAP1, pos, len);
	pos += len;

	/* TODO: could add 'type' information into the end of the data if it
	 * is needed in the user space daemon */
	if (extra != NULL) {
		u16 *elen;

		skb_put(skb, extra_len + 2);
		memcpy(pos, extra, extra_len);
		pos += extra_len;
		elen = (u16 *) pos;
		*elen = __cpu_to_le16(extra_len);
	}

	if (res) {
		printk(KERN_WARNING "%s: prism2_rx_80211 from_bap failed\n",
		       dev->name);
		dev_kfree_skb_irq(skb);
		return 0;
	}

	skb->dev = dev;
	skb->mac.raw = skb->data;
	skb_pull(skb, hdrlen);
	skb->pkt_type = PACKET_OTHERHOST;
	skb->protocol = htons(ETH_P_802_2);
	netif_rx(skb);

	return (hdrlen + len);
}


static int prism2_80211_header_parse(struct sk_buff *skb, unsigned char *haddr)
{
	memcpy(haddr, skb->mac.raw + 10, ETH_ALEN); /* addr2 */
	return ETH_ALEN;
}
#endif /* defined(PRISM2_MONITOR) || defined(PRISM2_HOSTAPD) */


#ifdef PRISM2_MONITOR

/* Called only from hardware IRQ */
static void monitor_rx_nl(struct net_device *dev,
			  struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb;
	int res;
	u16 len;

	if (local->nl_monitor == NULL) {
		printk(KERN_DEBUG "%s: monitor_rx_nl - netlink not open\n",
		       dev->name);
		return;
	}

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: monitor_rx_nl: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		return;
	}

	skb = dev_alloc_skb(sizeof(struct hfa384x_rx_frame) + len);
	if (!skb) {
		printk(KERN_WARNING "%s: monitor_rx_nl cannot allocate "
		       "buffer\n", dev->name);
		netlink_set_err(local->nl_monitor, 0, PRISM2_MONITOR_GROUP,
				ENOBUFS);
		return;
	}

	memcpy(skb->data, rxdesc, sizeof(struct hfa384x_rx_frame));

	res = hfa384x_from_bap(dev, BAP1, skb->data +
			       sizeof(struct hfa384x_rx_frame), len);

	skb_put(skb, sizeof(struct hfa384x_rx_frame) + len);

	if (res)
		printk(KERN_WARNING "%s: monitor_rx_nl from_bap failed\n",
		       dev->name);

	skb->dev = dev;

	NETLINK_CB(skb).dst_groups = PRISM2_MONITOR_GROUP;
	netlink_broadcast(local->nl_monitor, skb, 0, PRISM2_MONITOR_GROUP,
			  GFP_ATOMIC);
}

/* Called only from hardware IRQ */
static void monitor_rx_dev(struct net_device *dev,
			   struct hfa384x_rx_frame *rxdesc)
{
	struct net_device_stats *stats;
	int len;

	stats = prism2_get_stats(dev);
	len = prism2_rx_80211(dev, rxdesc, PRISM2_RX_MONITOR, NULL, 0);
	stats->rx_packets++;
	stats->rx_bytes += len;
}

/* Called only from hardware IRQ */
static void monitor_rx(struct net_device *dev, struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (le16_to_cpu(rxdesc->status) & HFA384X_RX_STATUS_FCSERR &&
		!local->monitor_allow_fcserr)
		return;

	if (local->monitor_type == PRISM2_MONITOR_NL)
		monitor_rx_nl(dev, rxdesc);
	else
		monitor_rx_dev(dev, rxdesc);
}


/* FIX: monitor_tx() will disappear at some point.. it does not support more
 * than one device and wlan0ap can be used instead of it for packet injection
 */

#ifdef PRISM2_PCCARD
static void monitor_tx_skb(struct net_device *dev, struct sk_buff *skb)
{
	local_info_t *local;
	int idx, res;

	local = (local_info_t *) dev->priv;

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		return;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR &&
	    skb->len >= sizeof(struct hfa384x_tx_frame))
		prism2_dump_tx_header(dev->name,
				      (struct hfa384x_tx_frame *) skb->data);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data, skb->len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);

	if (res) {
		printk("%s: monitor_tx_skb - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
}


static void monitor_tx(struct sock *sk, int len)
{
	struct net_device *dev;
	local_info_t *local;
	struct sk_buff *skb;

	printk("monitor_tx(%p, %d)\n", sk, len);

	if (dev_list == NULL)
		return;

	dev = dev_list->priv;
	local = (local_info_t *) dev->priv;

	if (local == NULL || local->nl_monitor == NULL)
		return;

	do {
		if (rtnl_shlock_nowait())
			return;
		while ((skb = skb_dequeue(&sk->receive_queue)) != NULL) {
			printk("monitor_tx: got skb, len=%d\n", skb->len);
			monitor_tx_skb(dev, skb);
			kfree_skb(skb);
		}
		up(&rtnl_sem);
	} while (local->nl_monitor && local->nl_monitor->receive_queue.qlen);
}
#else /* PRISM2_PCCARD */
static void monitor_tx(struct sock *sk, int len)
{
}
#endif /* PRISM2_PCCARD */
#endif /* PRISM2_MONITOR */

static void prism2_dump_rx_header(const char *name,
				  const struct hfa384x_rx_frame *rx)
{
	u16 status, fc;


	status = __le16_to_cpu(rx->status);

	printk(KERN_DEBUG "%s: RX status=0x%04x (port=%d, type=%d, "
	       "fcserr=%d) silence=%d signal=%d rate=%d rxflow=%d; "
	       "jiffies=%ld\n",
	       name, status, (status >> 8) & 0x07, status >> 13, status & 1,
	       rx->silence, rx->signal, rx->rate, rx->rxflow, jiffies);

	fc = __le16_to_cpu(rx->frame_control);
	printk(KERN_DEBUG "   FC=0x%04x (type=%d:%d) dur=0x%04x seq=0x%04x "
	       "data_len=%d%s%s\n",
	       fc, WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(rx->duration_id), __le16_to_cpu(rx->seq_ctrl),
	       __le16_to_cpu(rx->data_len),
	       fc & WLAN_FC_TODS ? " [ToDS]" : "",
	       fc & WLAN_FC_FROMDS ? " [FromDS]" : "");

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(rx->addr1), MAC2STR(rx->addr2), MAC2STR(rx->addr3),
	       MAC2STR(rx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(rx->dst_addr), MAC2STR(rx->src_addr),
	       __be16_to_cpu(rx->len));
}

/* Called only from software IRQ */
static void handle_bridged_queue(void *data)
{
	local_info_t *local = (local_info_t *) data;
	struct sk_buff *skb;

	while ((skb = skb_dequeue(&local->bridge_list)) != NULL)
		dev_queue_xmit(skb);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	MOD_DEC_USE_COUNT;
#endif
}


/* Called only from hardware IRQ */
static struct prism2_frag_entry *
prism2_frag_cache_find(local_info_t *local, unsigned int seq,
		       unsigned int frag, u8 *src, u8 *dst)
{
	struct prism2_frag_entry *entry;
	int i;

	for (i = 0; i < PRISM2_FRAG_CACHE_LEN; i++) {
		entry = &local->frag_cache[i];
		if (entry->skb != NULL &&
		    entry->first_frag_time + 2 * HZ < jiffies) {
			printk(KERN_DEBUG "%s: expiring fragment cache entry "
			       "seq=%u last_frag=%u\n",
			       local->dev->name, entry->seq, entry->last_frag);
			dev_kfree_skb_irq(entry->skb);
			entry->skb = NULL;
		}

		if (entry->skb != NULL && entry->seq == seq &&
		    (entry->last_frag + 1 == frag || frag == -1) &&
		    memcmp(entry->src_addr, src, ETH_ALEN) == 0 &&
		    memcmp(entry->dst_addr, dst, ETH_ALEN) == 0)
			return entry;
	}

	return NULL;
}


/* Called only from hardware IRQ */
static struct sk_buff *prism2_frag_cache_get(local_info_t *local,
					     struct hfa384x_rx_frame *rxdesc)
{
	struct sk_buff *skb = NULL;
	u16 sc;
	unsigned int frag, seq;
	struct prism2_frag_entry *entry;

	sc = le16_to_cpu(rxdesc->seq_ctrl);
	frag = WLAN_GET_SEQ_FRAG(sc);
	seq = WLAN_GET_SEQ_SEQ(sc);

	if (frag == 0) {
		/* Reserve enough space to fit maximum frame length */
		skb = dev_alloc_skb(local->dev->mtu + ETH_HLEN +
				    2 /* alignment */ +
				    8 /* WEP */ + ETH_ALEN /* WDS */);
		if (skb == NULL)
			return NULL;

		/* align IP on 16b boundary */
		skb_reserve(skb, 2);

		memcpy(skb_put(skb, ETH_HLEN), &rxdesc->dst_addr, ETH_HLEN);

		entry = &local->frag_cache[local->frag_next_idx];
		local->frag_next_idx++;
		if (local->frag_next_idx >= PRISM2_FRAG_CACHE_LEN)
			local->frag_next_idx = 0;

		if (entry->skb != NULL)
			dev_kfree_skb_irq(entry->skb);

		entry->first_frag_time = jiffies;
		entry->seq = seq;
		entry->last_frag = frag;
		entry->skb = skb;
		memcpy(entry->src_addr, rxdesc->addr2, ETH_ALEN);
		memcpy(entry->dst_addr, rxdesc->addr1, ETH_ALEN);
	} else {
		/* received a fragment of a frame for which the head fragment
		 * should have already been received */
		entry = prism2_frag_cache_find(local, seq, frag, rxdesc->addr2,
					       rxdesc->addr1);
		if (entry != NULL) {
			entry->last_frag = frag;
			skb = entry->skb;
		}
	}

	return skb;
}


/* Called only from hardware IRQ */
static int prism2_frag_cache_invalidate(local_info_t *local,
					struct hfa384x_rx_frame *rxdesc)
{
	u16 sc;
	unsigned int seq;
	struct prism2_frag_entry *entry;

	sc = le16_to_cpu(rxdesc->seq_ctrl);
	seq = WLAN_GET_SEQ_SEQ(sc);

	entry = prism2_frag_cache_find(local, seq, -1, rxdesc->addr2,
				       rxdesc->addr1);

	if (entry == NULL) {
		printk(KERN_DEBUG "%s: could not invalidate fragment cache "
		       "entry (seq=%u)\n",
		       local->dev->name, seq);
		return -1;
	}

	entry->skb = NULL;
	return 0;
}


/* Called only from hardware IRQ */
static void prism2_rx(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb, *skb2 = NULL;
	int res, skb_from_frag_cache = 0;
	u16 len = 0;
	u16 rxfid, status, macport, msg_type, fc, type, stype, sc;
	unsigned int frag, seq;
	struct hfa384x_rx_frame rxdesc;
	struct sta_info *sta = NULL;
	prism2_wds_info_t *wds = NULL;
	struct net_device_stats *stats;

	stats = prism2_get_stats(dev);

	rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
#ifndef final_version
	if (rxfid == 0) {
		rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
		printk(KERN_DEBUG "prism2_rx: rxfid=0 (next 0x%04x)\n",
		       rxfid);
		prism2_hw_reset(dev);
		goto rx_dropped;
	}
#endif

	res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &rxdesc, sizeof(rxdesc));
	if (res) {
		printk(KERN_DEBUG "copy from BAP1 failed %d\n", res);
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		goto rx_dropped;
	}

	status = __le16_to_cpu(rxdesc.status);
	macport = (status >> 8) & 0x07;

	if (__le16_to_cpu(rxdesc.data_len) & 0x8000 ||
	    (local->monitor_type == PRISM2_MONITOR_OFF &&
	     (rxdesc.addr1[0] & 0x01) == 0 &&
	     memcmp(rxdesc.addr1, dev->dev_addr, 6) != 0) ||
	    (macport != 0 && macport != 7)) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		/* FIX: this is most probably a concurrency problem (the driver
		 * could be running two operations at the same time (at least
		 * on SMP hosts) and real fix would be to serialize some
		 * operations.. */
		printk(KERN_DEBUG "%s: prism2_rx: re-reading rxdesc to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &rxdesc,
					       sizeof(rxdesc));
		if (res)
			printk(KERN_DEBUG "prism2_rx: could not re-read "
			       "rxdesc\n");
		if (__le16_to_cpu(rxdesc.data_len) & 0x8000) {
			printk(KERN_DEBUG "prism2_rx: re-read did not fix "
			       "rxdesc\n");
			goto rx_dropped;
		}
	}

	status = __le16_to_cpu(rxdesc.status);
	macport = (status >> 8) & 0x07;

	if (local->frame_dump & PRISM2_DUMP_RX_HDR)
		prism2_dump_rx_header(dev->name, &rxdesc);

	if (macport != 0) {
#ifdef PRISM2_MONITOR
		if (macport == 7) {
			monitor_rx(dev, &rxdesc);
		} else {
#else
		{
#endif
			printk(KERN_DEBUG "RX: Unknown MACPort %d\n", macport);
		}
		goto rx_dropped;
	}

	/* FCS errors should not come this far, but let's make sure that frames
	 * with errors will be dropped even in Host AP mode */
	if (status & HFA384X_RX_STATUS_FCSERR) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped FCSErr frame "
		       "(status=%02X)\n", dev->name, status);
		goto rx_dropped;
	}

	msg_type = status >> 13;

	fc = __le16_to_cpu(rxdesc.frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	if (msg_type == HFA384X_RX_MSGTYPE_MGMT) {
#ifdef PRISM2_HOSTAPD
		/* send management frames to the user space daemon for
		 * processing */
		if (type == WLAN_FC_TYPE_MGMT) {
			prism2_rx_80211(local->apdev, &rxdesc, PRISM2_RX_MGMT,
					NULL, 0);
			goto rx_exit;
		}
#endif /* PRISM2_HOSTAPD */

		if (local->iw_mode == IW_MODE_MASTER) {
			/* FIX: should STA's PS flag be also checked here? */
			hostap_rx(dev, &rxdesc, 0);
		} else {
			printk(KERN_DEBUG "%s: prism2_rx: management frame "
			       "received in non-Host AP mode\n", dev->name);
			prism2_dump_rx_header(dev->name, &rxdesc);
			goto rx_dropped;
		}
		goto rx_exit;
	}

	if (msg_type != HFA384X_RX_MSGTYPE_NORMAL &&
	    msg_type != HFA384X_RX_MSGTYPE_RFC1042 &&
	    msg_type != HFA384X_RX_MSGTYPE_BRIDGETUNNEL) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped frame "
		       "(msg_type=%d)\n", dev->name, msg_type);
		goto rx_dropped;
	}

	if (type != WLAN_FC_TYPE_DATA) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped non-data frame "
		       "(type=0x%02x, subtype=0x%02x)\n",
		       dev->name, type, stype);
		prism2_dump_rx_header(dev->name, &rxdesc);
		goto rx_dropped;
	}

	if (local->iw_mode == IW_MODE_MASTER) {
		spin_lock(&local->ap->sta_table_lock);
		sta = ap_get_sta(local->ap, rxdesc.addr2);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock(&local->ap->sta_table_lock);

		if (fc & WLAN_FC_TODS) {
			if (memcmp(rxdesc.addr1, dev->dev_addr, 6) != 0) {
				/* BSSID is not ours; drop */
				goto rx_dropped;
			}

			/* check if the frame came from a registered WDS
			 * connection */
			spin_lock(&local->wdslock);
			wds = local->wds;
			while (wds != NULL &&
			       memcmp(wds->remote_addr, rxdesc.addr2,
				      ETH_ALEN) != 0)
				wds = wds->next;
			spin_unlock(&local->wdslock);
			if (wds) {
				dev = &wds->dev;
				stats = prism2_get_stats(dev);
			}

			if (wds == NULL &&
			    (sta == NULL || !(sta->flags & WLAN_STA_ASSOC))) {
#ifdef PRISM2_HOSTAPD
				prism2_rx_80211(local->apdev, &rxdesc,
						PRISM2_RX_NON_ASSOC, NULL, 0);
#else /* PRISM2_HOSTAPD */
				printk(KERN_DEBUG "%s: dropped received packet"
				       " from non-associated STA " MACSTR
				       " (type=0x%02x, subtype=0x%02x)\n",
				       dev->name,
				       MAC2STR(rxdesc.addr2), type, stype);
				hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
				goto rx_dropped;
			}
		} else if (fc & WLAN_FC_FROMDS) {
			/* FromDS frame - not for us; probably
			 * broadcast/multicast in another BSS - drop */
			if (memcmp(rxdesc.addr1, dev->dev_addr, 6) == 0) {
				printk(KERN_DEBUG "Odd.. FromDS packet "
				       "received with own BSSID\n");
				prism2_dump_rx_header(dev->name, &rxdesc);
			}
			goto rx_dropped;
		} else if (stype == WLAN_FC_STYPE_NULLFUNC && sta == NULL &&
			   memcmp(rxdesc.addr1, dev->dev_addr, 6) == 0) {
#ifdef PRISM2_HOSTAPD
			prism2_rx_80211(local->apdev, &rxdesc,
					PRISM2_RX_NON_ASSOC, NULL, 0);
#else /* PRISM2_HOSTAPD */
			/* At least Lucent f/w seems to send data::nullfunc
			 * frames with no ToDS flag when the current AP returns
			 * after being unavailable for some time. Speed up
			 * re-association by informing the station about it not
			 * being associated. */
			printk(KERN_DEBUG "%s: rejected received nullfunc "
			       "frame without ToDS from not associated STA "
			       MACSTR "\n",
			       dev->name, MAC2STR(rxdesc.addr2));
			hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
			goto rx_dropped;
		} else if (stype == WLAN_FC_STYPE_NULLFUNC) {
			/* At least Lucent cards seem to send periodic nullfunc
			 * frames with ToDS. Let these through to update SQ
			 * stats and PS state. Nullfunc frames do not contain
			 * any data and they will be dropped below. */
		} else {
			printk(KERN_DEBUG "%s: dropped received packet from "
			       MACSTR " with no ToDS flag (type=0x%02x, "
			       "subtype=0x%02x)\n", dev->name,
			       MAC2STR(rxdesc.addr2), type, stype);
			prism2_dump_rx_header(dev->name, &rxdesc);
			goto rx_dropped;
		}
	}

	if (sta) {
		prism2_ap_update_sq(sta, &rxdesc);

		if ((fc & WLAN_FC_PWRMGT) && !(sta->flags & WLAN_STA_PS)) {
			sta->flags |= WLAN_STA_PS;
			PDEBUG(DEBUG_PS2, "STA " MACSTR " changed to use PS "
			       "mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
		} else if (!(fc & WLAN_FC_PWRMGT) &&
			   (sta->flags & WLAN_STA_PS)) {
			sta->flags &= ~WLAN_STA_PS;
			PDEBUG(DEBUG_PS2, "STA " MACSTR " changed to not use "
			       "PS mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
			schedule_packet_send(local, sta);
		}
	}

	if (stype != WLAN_FC_STYPE_DATA &&
	    stype != WLAN_FC_STYPE_DATA_CFACK &&
	    stype != WLAN_FC_STYPE_DATA_CFPOLL &&
	    stype != WLAN_FC_STYPE_DATA_CFACKPOLL) {
		if (local->iw_mode == IW_MODE_MASTER &&
		    local->ap && local->ap->nullfunc_ack &&
		    stype == WLAN_FC_STYPE_NULLFUNC &&
		    fc & WLAN_FC_TODS) {
#ifdef PRISM2_HOSTAPD
			prism2_rx_80211(local->apdev, &rxdesc,
					PRISM2_RX_NULLFUNC_ACK, NULL, 0);
#else /* PRISM2_HOSTAPD */
			/* some STA f/w's seem to require control::ACK frame
			 * for data::nullfunc, but Prism2 f/w 0.8.0 (at least
			 * from Compaq) does not send this.. Try to generate
			 * ACK for these frames from the host driver to make
			 * power saving work with, e.g., Lucent WaveLAN f/w */
			hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
			goto rx_exit;
		}
		if (stype != WLAN_FC_STYPE_NULLFUNC)
			printk(KERN_DEBUG "%s: prism2_rx: dropped data frame "
			       "with no data (type=0x%02x, subtype=0x%02x)\n",
			       dev->name, type, stype);
		goto rx_dropped;
	}

	len = __le16_to_cpu(rxdesc.data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: Rx: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		goto rx_dropped;
	}

#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (wds) {
		if (len < ETH_ALEN) {
			printk(KERN_DEBUG "%s: Rx (WDS): len(%d) < %d\n",
			       dev->name, len, ETH_ALEN);
			goto rx_dropped;
		}
		if (!local->host_decrypt || !(fc & WLAN_FC_ISWEP))
			len -= ETH_ALEN; /* addr4 */
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

	sc = le16_to_cpu(rxdesc.seq_ctrl);
	frag = WLAN_GET_SEQ_FRAG(sc);
	seq = WLAN_GET_SEQ_SEQ(sc);

	if (local->host_decrypt && (fc & WLAN_FC_ISWEP) && len >= 8 &&
	    (frag != 0 || (fc & WLAN_FC_MOREFRAG))) {
		skb = prism2_frag_cache_get(local, &rxdesc);
		if (!skb) {
			printk(KERN_DEBUG "%s: Rx cannot get skb from "
			       "fragment cache (morefrag=%d seq=%u frag=%u)\n",
			       dev->name, (fc & WLAN_FC_MOREFRAG) != 0, seq,
			       frag);
			goto rx_dropped;
		}
		skb_from_frag_cache = 1;
	} else {
		skb = dev_alloc_skb(len + ETH_HLEN + 2);
		if (!skb) {
			printk(KERN_WARNING "%s: Rx cannot allocate buffer\n",
			       dev->name);
			goto rx_dropped;
		}

		/* align IP on 16b boundary */
		skb_reserve(skb, 2);

		/* receive packet (decapsulated Ethernet frame) */
		memcpy(skb_put(skb, ETH_HLEN), &rxdesc.dst_addr, ETH_HLEN);
	}

	dev->last_rx = jiffies;

	if (sta) {
		sta->rx_packets++;
		sta->rx_bytes += len;
		sta->last_rx = jiffies;
	}

	if (local->host_decrypt && (fc & WLAN_FC_ISWEP) && len >= 8) {
		unsigned char *dstart;

		/* TODO: WEP decryption could be performed from a softIRQ
		 * instead of using the CPU from hardIRQ handler. */

		if (skb->tail + len > skb->end) {
			printk(KERN_WARNING "%s: host decrypted and "
			       "reassembled frame did not fit skb\n",
			       dev->name);
			goto rx_dropped_free;
		}

		dstart = skb_put(skb, len);

		/* read in WEP part of the frame: IV (4 bytes), encrypted
		 * payload (including SNAP header), ICV (4 bytes) */
		res = hfa384x_from_bap(dev, BAP1, dstart, len);
		if (res)
			goto copy_failed;

		if (prism2_wep_decrypt(local, dstart, len, NULL)) {
			printk(KERN_DEBUG "%s: WEP decryption failed\n",
			       dev->name);
			local->comm_tallies.rx_discards_wep_undecryptable++;
			goto rx_dropped_free;
		}

		/* frame was successfully decrypted; remove IV and ICV from
		 * length */
		len -= 8;
		skb_trim(skb, skb->len - 8);

#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		if (wds) {
			/* get addr4 from its bogus location */
			memcpy(skb->data + ETH_ALEN,
			       skb->data + ETH_HLEN + len - ETH_ALEN,
			       ETH_ALEN);
			len -= ETH_ALEN;
			skb_trim(skb, skb->len - ETH_ALEN);
		}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

		if (frag == 0 && len >= sizeof(snap_header)) {
			if (memcmp(dstart, snap_header, 3) == 0)
			{
				/* remove Ethernet-II SNAP header */
				len -= 6 + 2;
				memmove(dstart - 2, dstart + 6, len + 2);
				skb_trim(skb, skb->len - 8);
			} else {
				PDEBUG(DEBUG_EXTRA, "%s: no SNAP (host "
				       "decrypt)? (" MACSTR ")\n",
				       dev->name,
				       MAC2STR(skb->data + ETH_HLEN));
			}
		}

		if (fc & WLAN_FC_MOREFRAG) {
			/* more fragments expected - leave the skb in fragment
			 * cache for now; it will delivered to upper layers
			 * after all fragments have been received */
			goto rx_exit;
		}

		/* this was the last fragment and the frame will be
		 * delivered, so remove skb from fragment cache */
		if (skb_from_frag_cache)
			prism2_frag_cache_invalidate(local, &rxdesc);

		goto copy_failed; /* skip plain text copy */
	}

	if (len >= 8) {
		/* try to remove Ethernet-II snap header */
		res = hfa384x_from_bap(dev, BAP1, skb_put(skb, 6), 6);
		if (res)
			goto copy_failed;

		if (memcmp(skb->data + ETH_HLEN, snap_header, 3) == 0) {
			res = hfa384x_from_bap(dev, BAP1, skb->data + 12,
					       len - 6);
			skb_put(skb, len - ETH_HLEN);
		} else {
			PDEBUG(DEBUG_EXTRA, "%s: no SNAP? (" MACSTR ")\n",
			       dev->name, MAC2STR(skb->data + ETH_HLEN));
			res = hfa384x_from_bap(dev, BAP1,
					       skb->data + ETH_HLEN + 6,
					       len - 6);
			skb_put(skb, len - 6);
		}
	} else {
		res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_HLEN, len);
		skb_put(skb, len);
	}

#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (!res && wds) {
		/* get addr4 from its bogus location */
		if (len & 1) {
			/* BAP offset is odd; must arrange it to be even */
			u8 tmpbuf[ETH_ALEN + 1];

			res = hfa384x_setup_bap(dev, BAP1, rxfid,
						sizeof(rxdesc) + len - 1);
			if (!res) {
				res = hfa384x_from_bap(dev, BAP1, tmpbuf,
						       ETH_ALEN + 1);
				memcpy(skb->data + ETH_ALEN, tmpbuf + 1,
				       ETH_ALEN);
			}
		} else {
			res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_ALEN,
					       ETH_ALEN);
		}
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

 copy_failed:
	if (res) {
		printk(KERN_DEBUG "%s: data copy from BAP1 failed %d\n",
		       dev->name, res);
		goto rx_dropped_free;
	}

	skb->dev = dev;

	if (!wds && local->ap->bridge_packets) {
		if (rxdesc.addr3[0] & 0x01) {
			/* copy multicast frame both to the higher layers and
			 * to the wireless media */
			skb2 = skb_clone(skb, GFP_ATOMIC);
			if (skb2 == NULL)
				printk("%s: skb_clone failed for multicast "
				       "frame\n", dev->name);
		} else {
			struct sta_info *dest_sta;
			spin_lock(&local->ap->sta_table_lock);
			dest_sta = ap_get_sta(local->ap, rxdesc.addr3);
			if (dest_sta != NULL &&
			    (dest_sta->flags & WLAN_STA_ASSOC)) {
				/* send frame directly to the associated STA
				 * using wireless media and not passing to
				 * higher layers */
				skb2 = skb;
				skb = NULL;
			}
			spin_unlock(&local->ap->sta_table_lock);
		}
	}

	if (skb2 != NULL) {
		/* send to wireless media (put in a queue and send later) */
		skb2->mac.raw = skb2->data;
		skb2->nh.raw = skb2->data + ETH_HLEN;
		skb_queue_tail(&local->bridge_list, skb2);
/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
		queue_task(&local->bridge_queue, &tq_scheduler);
#else
		MOD_INC_USE_COUNT;
		if (schedule_task(&local->bridge_queue) == 0)
			MOD_DEC_USE_COUNT;
#endif

		if ((rxdesc.addr3[0] & 0x01) == 0x01)
			local->ap->bridged_multicast++;
		else
			local->ap->bridged_unicast++;
	}

	if (skb) {
		skb->protocol = eth_type_trans(skb, dev);
		netif_rx(skb);
	}

	stats->rx_packets++;
	stats->rx_bytes += len;

	/* FIX: add WIRELESS_EXT & SPY handling */

 rx_exit:
	if (sta)
		atomic_dec(&sta->users);
	return;

 rx_dropped_free:
	if (skb_from_frag_cache)
		prism2_frag_cache_invalidate(local, &rxdesc);
	dev_kfree_skb_irq(skb);

 rx_dropped:
	stats->rx_dropped++;
	goto rx_exit;
}


static void prism2_setup_dev(struct net_device *dev, local_info_t *local,
			     int main_dev)
{
	ether_setup(dev);

	/* kernel callbacks */
	dev->get_stats = prism2_get_stats;
#ifdef WIRELESS_EXT
	dev->get_wireless_stats = main_dev ? prism2_get_wireless_stats : NULL;
#endif
	dev->open = prism2_open;
	dev->stop = prism2_close;
	dev->hard_start_xmit = prism2_tx;
#ifdef HAVE_MULTICAST
	/* FIX: to be implemented as soon as Prism2 supports GroupAddresses
	 * and correct documentation is available */

	/* dev->set_multicast_list = prism2_set_multicast_list; */
#endif
#ifdef HAVE_PRIVATE_IOCTL
	dev->do_ioctl = main_dev ? prism2_ioctl : NULL;
#endif
#ifdef HAVE_CHANGE_MTU
	dev->change_mtu = prism2_change_mtu;
#endif
#ifdef HAVE_TX_TIMEOUT
	dev->tx_timeout = prism2_tx_timeout;
	dev->watchdog_timeo = TX_TIMEOUT;
#endif

	dev->mtu = mtu;

	netif_stop_queue(dev);
}


static inline int prism2_wds_special_addr(u8 *addr)
{
	if (addr[0] || addr[1] || addr[2] || addr[3] || addr[4] || addr[5])
		return 0;

	return 1;
}


int prism2_wds_add(local_info_t *local, u8 *remote_addr, int rtnl_locked)
{
	prism2_wds_info_t *wds, *wds2 = NULL;
	unsigned long flags;
	int i, ret;

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL &&
	       memcmp(wds->remote_addr, remote_addr, ETH_ALEN) != 0) {
		if (!wds2 && prism2_wds_special_addr(wds->remote_addr))
			wds2 = wds;
		wds = wds->next;
	}
	if (!wds && wds2) {
		/* take pre-allocated entry into use */
		memcpy(wds2->remote_addr, remote_addr, ETH_ALEN);
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if (wds && !prism2_wds_special_addr(remote_addr))
		return -EEXIST;

	if (!wds && wds2) {
		printk(KERN_DEBUG "%s: using pre-allocated WDS netdevice %s\n",
		       local->dev->name, wds2->dev.name);
		return 0;
	}

	if (local->wds_connections >= local->wds_max_connections)
		return -ENOBUFS;

	/* verify that there is room for wds# postfix in the interface name */
	if (strlen(local->dev->name) > IFNAMSIZ - 5) {
		printk(KERN_DEBUG "'%s' too long base device name\n",
		       local->dev->name);
		return -EINVAL;
	}

	wds = (prism2_wds_info_t *) kmalloc(sizeof(*wds) + PRISM2_NETDEV_EXTRA,
					    GFP_ATOMIC);
	if (wds == NULL)
		return -ENOMEM;

	memset(wds, 0, sizeof(*wds) + PRISM2_NETDEV_EXTRA);
	prism2_set_dev_name(&wds->dev, wds + 1);

	memcpy(wds->remote_addr, remote_addr, ETH_ALEN);

	prism2_setup_dev(&wds->dev, local, 0);

	wds->dev.priv = local;
	memcpy(wds->dev.dev_addr, local->dev->dev_addr, ETH_ALEN);
	wds->dev.base_addr = local->dev->base_addr;
	wds->dev.irq = local->dev->irq;
	wds->dev.mem_start = local->dev->mem_start;
	wds->dev.mem_end = local->dev->mem_end;

	i = 0;
	do {
		sprintf(wds->dev.name, "%swds%d", local->dev->name, i++);
	} while (i < 10000 && dev_get(wds->dev.name));

	if (rtnl_locked)
		ret = register_netdevice(&wds->dev);
	else
		ret = register_netdev(&wds->dev);

	if (ret) {
		printk(KERN_WARNING "%s: registering WDS device '%s' failed\n",
		       local->dev->name, wds->dev.name);
		kfree(wds);
		return -EINVAL;
	}

	spin_lock_irqsave(&local->wdslock, flags);
	local->wds_connections++;
	wds->next = local->wds;
	local->wds = wds;
	spin_unlock_irqrestore(&local->wdslock, flags);

	printk(KERN_DEBUG "%s: registered WDS netdevice %s\n",
	       local->dev->name, wds->dev.name);

	return 0;
}


int prism2_wds_del(local_info_t *local, u8 *remote_addr, int rtnl_locked,
		   int do_not_remove)
{
	prism2_wds_info_t *wds, *prev = NULL;
	unsigned long flags;

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL &&
	       memcmp(wds->remote_addr, remote_addr, ETH_ALEN) != 0) {
		prev = wds;
		wds = wds->next;
	}
	if (wds && !do_not_remove) {
		if (prev != NULL)
			prev->next = wds->next;
		else
			local->wds = wds->next;
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if (!wds)
		return -ENODEV;

	if (do_not_remove) {
		memset(wds->remote_addr, 0, ETH_ALEN);
		return 0;
	}

	if (rtnl_locked)
		unregister_netdevice(&wds->dev);
	else
		unregister_netdev(&wds->dev);
	local->wds_connections--;

	printk(KERN_DEBUG "%s: unregistered WDS netdevice %s\n",
	       local->dev->name, wds->dev.name);

	kfree(wds);

	return 0;
}


/* Called only from hardware IRQ */
static void prism2_alloc_ev(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int idx;
	u16 fid;

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);

	PDEBUG(DEBUG_FID, "FID: interrupt: ALLOC - fid=0x%04x\n", fid);

	idx = local->next_alloc;

	do {
		if (local->txfid[idx] == fid) {
			PDEBUG(DEBUG_FID, "FID: found matching txfid[%d]\n",
			       idx);

#ifndef PRISM2_USE_TX_INTERRUPT
			/* Alloc events are used only for TX frames, so
			 * increase TX packet counter here. This will make
			 * separate TX event handling unnecessary. */
			local->stats.tx_packets++;
#endif /* PRISM2_USE_TX_INTERRUPT */

#ifndef final_version
			if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY)
				printk("Already released txfid found at idx "
				       "%d\n", idx);
			if (local->intransmitfid[idx] == PRISM2_TXFID_RESERVED)
				printk("Already reserved txfid found at idx "
				       "%d\n", idx);
#endif
			local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
			idx++;
			local->next_alloc = idx >= PRISM2_TXFID_COUNT ? 0 :
				idx;

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
			if (local->last_txfid_idx == PRISM2_DUMMY_FID &&
			    netif_queue_stopped(dev))
				prism2_netif_wake_queues(dev);
#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */
			if (netif_queue_stopped(dev))
				prism2_netif_wake_queues(dev);
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */

			goto out;
		}

		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_alloc);

	printk(KERN_WARNING "%s: could not found matching txfid (0x%04x) for "
	       "alloc event\n", dev->name, HFA384X_INW(HFA384X_ALLOCFID_OFF));

 out:
}


/* Called only from hardware IRQ */
static void prism2_txexc(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, status, fc;
	int res;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta;

	local->stats.tx_errors++;
	fid = HFA384X_INW(HFA384X_TXCOMPLFID_OFF);
	PDEBUG(DEBUG_EXTRA, "%s: TXEXC - fid=0x%04x - ", dev->name, fid);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &txdesc, sizeof(txdesc));
	if (res) {
		PDEBUG2(DEBUG_EXTRA, "could not get TXEXC txframe\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
	status = __le16_to_cpu(txdesc.status);
	PDEBUG2(DEBUG_EXTRA, "status=0x%04x (%s%s%s%s) tx_control=%04x\n",
		status,
		status & HFA384X_TX_STATUS_RETRYERR ? "[RetryErr]" : "",
		status & HFA384X_TX_STATUS_AGEDERR ? "[AgedErr]" : "",
		status & HFA384X_TX_STATUS_DISCON ? "[Discon]" : "",
		status & HFA384X_TX_STATUS_FORMERR ? "[FormErr]" : "",
		__le16_to_cpu(txdesc.tx_control));
	fc = __le16_to_cpu(txdesc.frame_control);
	PDEBUG(DEBUG_EXTRA, "   retry_count=%d tx_rate=%d fc=0x%04x "
	       "(%s%s%s::%d)\n",
	       txdesc.retry_count, txdesc.tx_rate, fc,
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_MGMT ? "Mgmt" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_CTRL ? "Ctrl" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA ? "Data" : "",
	       WLAN_FC_GET_STYPE(fc));
	PDEBUG(DEBUG_EXTRA, "   addr1=" MACSTR " addr2=" MACSTR " addr3="
	       MACSTR "\n",
	       MAC2STR(txdesc.addr1), MAC2STR(txdesc.addr2),
	       MAC2STR(txdesc.addr3));

	if (local->iw_mode != IW_MODE_MASTER)
		return;

	spin_lock(&local->ap->sta_table_lock);
	/* FIX: is addr1 correct for all frame types? */
	sta = ap_get_sta(local->ap, txdesc.addr1);
	if (sta) {
		sta->txexc++;
		sta->tx_since_last_failure = 0;
		if (sta->tx_rate_idx > 0 &&
		    txdesc.tx_rate <= sta->tx_rate) {
			int rate = sta->tx_rate_idx;
			/* use next lower rate */
			while (rate > 0) {
				rate--;
				if (sta->tx_supp_rates & (1 << rate)) {
					sta->tx_rate_idx = rate;
					break;
				}
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate lowered "
			       "to %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
		}
	} else {
		PDEBUG(DEBUG_AP, "Could not find STA for this TX error\n");
	}
	spin_unlock(&local->ap->sta_table_lock);
}


static const char* hfa384x_linkstatus_str(u16 linkstatus)
{
	switch (linkstatus) {
	case HFA384X_LINKSTATUS_CONNECTED:
		return "Connected";
	case HFA384X_LINKSTATUS_DISCONNECTED:
		return "Disconnected";
	case HFA384X_LINKSTATUS_AP_CHANGE:
		return "Access point change";
	case HFA384X_LINKSTATUS_AP_OUT_OF_RANGE:
		return "Access point out of range";
	case HFA384X_LINKSTATUS_AP_IN_RANGE:
		return "Access point in range";
	case HFA384X_LINKSTATUS_ASSOC_FAILED:
		return "Association failed";
	default:
		return "Unknown";
	}
}


/* Called only from hardware IRQ */
static void prism2_info(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, val, *pos, len;
	int res, i, left;
	struct hfa384x_info_frame info;
	unsigned char buf[1024];
	struct hfa384x_comm_tallies *tallies;
	struct hfa384x_scan_result *scanres;

	fid = HFA384X_INW(HFA384X_INFOFID_OFF);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
	if (res) {
		printk("Could not get info frame\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}

	if (__le16_to_cpu(info.len) & 0x8000) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		printk(KERN_DEBUG "%s: prism2_info: re-reading header to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, fid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
		if (res)
			printk(KERN_DEBUG "prism2_info: could not re-read "
			       "info header\n");
		if (__le16_to_cpu(info.len) & 0x8000) {
			printk(KERN_DEBUG "prism2_info: re-read did not fix "
			       "header - ignoring this frame\n");
			return;
		}
	}

	le16_to_cpus(&info.len);
	le16_to_cpus(&info.type);

	if (info.type != HFA384X_INFO_COMMTALLIES)
		PDEBUG(DEBUG_EXTRA, "%s: INFO - fid=0x%04x - len=%d "
		       "type=0x%04x\n", dev->name, fid, info.len, info.type);

	switch (info.type) {
	case HFA384X_INFO_COMMTALLIES:
		if (info.len != 22 ||
		    hfa384x_from_bap(dev, BAP1, &buf,
				     sizeof(struct hfa384x_comm_tallies))) {
			printk("  info communication tallies failed\n");
			break;
		}
		tallies = (struct hfa384x_comm_tallies *) buf;
#define ADD_COMM_TALLIES(name) local->comm_tallies.name += tallies->name
		ADD_COMM_TALLIES(tx_unicast_frames);
		ADD_COMM_TALLIES(tx_multicast_frames);
		ADD_COMM_TALLIES(tx_fragments);
		ADD_COMM_TALLIES(tx_unicast_octets);
		ADD_COMM_TALLIES(tx_multicast_octets);
		ADD_COMM_TALLIES(tx_deferred_transmissions);
		ADD_COMM_TALLIES(tx_single_retry_frames);
		ADD_COMM_TALLIES(tx_multiple_retry_frames);
		ADD_COMM_TALLIES(tx_retry_limit_exceeded);
		ADD_COMM_TALLIES(tx_discards);
		ADD_COMM_TALLIES(rx_unicast_frames);
		ADD_COMM_TALLIES(rx_multicast_frames);
		ADD_COMM_TALLIES(rx_fragments);
		ADD_COMM_TALLIES(rx_unicast_octets);
		ADD_COMM_TALLIES(rx_multicast_octets);
		ADD_COMM_TALLIES(rx_fcs_errors);
		ADD_COMM_TALLIES(rx_discards_no_buffer);
		ADD_COMM_TALLIES(tx_discards_wrong_sa);
		ADD_COMM_TALLIES(rx_discards_wep_undecryptable);
		ADD_COMM_TALLIES(rx_message_in_msg_fragments);
		ADD_COMM_TALLIES(rx_message_in_bad_msg_fragments);
		break;

	case HFA384X_INFO_LINKSTATUS:
		if (info.len != 2 || hfa384x_from_bap(dev, BAP1, &val, 2)) {
			printk("  info linkstatus failed\n");
			break;
		}
		le16_to_cpus(&val);
		PDEBUG(DEBUG_EXTRA, "  LinkStatus=%d (%s)\n",
		       val, hfa384x_linkstatus_str(val));
		break;

	case HFA384X_INFO_SCANRESULTS:
		PDEBUG(DEBUG_EXTRA, "  ScanResults:\n");
		if (info.len < 3) {
			printk("    info scanresult failed\n");
			break;
		}
		val = (info.len - 1) * 2;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("    info read failed\n");
			break;
		}

		pos = (u16 *) buf;
		PDEBUG(DEBUG_EXTRA, "    Reserved=0x%04x\n",
		       __le16_to_cpu(*pos));
		pos++;
		PDEBUG(DEBUG_EXTRA, "    ScanReason=%d\n",
		       __le16_to_cpu(*pos));
		pos++;

		left = (info.len - 3) * 2;
		scanres = (struct hfa384x_scan_result *) pos;
		val = 1;
		while (left >= sizeof(struct hfa384x_scan_result)) {
			/* FIX: it might not be that wise to print a lot
			 * from here.. this is called from hw interrupt
			 * handler.. */
			PDEBUG(DEBUG_EXTRA, "    ScanResult %d:\n", val);
			PDEBUG(DEBUG_EXTRA, "      CHID=%d\n",
			       __le16_to_cpu(scanres->chid));
			PDEBUG(DEBUG_EXTRA, "      ANL=%d\n",
			       __le16_to_cpu(scanres->anl));
			PDEBUG(DEBUG_EXTRA, "      SL=%d\n",
			       __le16_to_cpu(scanres->sl));
			PDEBUG(DEBUG_EXTRA, "      bssid=" MACSTR "\n",
			       MAC2STR(scanres->bssid));
			PDEBUG(DEBUG_EXTRA, "      BcnInt=%d\n",
			       __le16_to_cpu(scanres->beacon_interval));
			PDEBUG(DEBUG_EXTRA, "      Capability=0x%04x\n",
			       __le16_to_cpu(scanres->capability));
			PDEBUG(DEBUG_EXTRA, "      SSID='");
			len = __le16_to_cpu(*(u16 *) &scanres->ssid);
			if (len > 32)
				len = 32;
			for (i = 0; i < len; i++) {
				unsigned char c = scanres->ssid[2 + i];
				if (c >= 32 && c < 127)
					PDEBUG2(DEBUG_EXTRA, "%c", c);
				else
					PDEBUG2(DEBUG_EXTRA, "<%02x>", c);
			}
			PDEBUG2(DEBUG_EXTRA, "'\n");
			PDEBUG(DEBUG_EXTRA, "      SupRates=");
			for (i = 0; i < sizeof(scanres->sup_rates); i++)
				PDEBUG2(DEBUG_EXTRA, "%02x ",
					scanres->sup_rates[i]);
			PDEBUG2(DEBUG_EXTRA, "\n");
			PDEBUG(DEBUG_EXTRA, "      Rate=%d\n",
			       __le16_to_cpu(scanres->rate));
			val++;
			left -= sizeof(struct hfa384x_scan_result);
			scanres++;
		}

		if (left > 0) {
			unsigned char *c = (unsigned char *) scanres;
			PDEBUG(DEBUG_EXTRA, "ScanRes extra: left=%d "
			       "(expected %dx):", left,
			       sizeof(struct hfa384x_scan_result));
			for (i = 0; i < left; i++)
				PDEBUG2(DEBUG_EXTRA, " %02x", *c++);
			PDEBUG2(DEBUG_EXTRA, "\n");
		}

		break;

	default:
		val = info.len > 0 ? (info.len - 1) * 2 : 0;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (val > 0 && hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("  info read failed\n");
			break;
		}
		PDEBUG(DEBUG_EXTRA, "Unknown info frame:");
		for (i = 0; i < val; i++)
			PDEBUG2(DEBUG_EXTRA, " %02x", buf[i]);
		PDEBUG2(DEBUG_EXTRA, "\n");
	}
}


/* Called only from hardware IRQ */
static void prism2_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	local_info_t *local = (local_info_t *) dev->priv;
	int events = 0;
	u16 ev;
#ifndef final_version
	static long int last_magic_err = 0;
#endif

	if (local->func->card_present && !local->func->card_present(local)) {
		printk(KERN_DEBUG "%s: Interrupt, but dev not OK\n",
		       dev->name);
		return;
	}

	if (!local->hw_ready || local->hw_resetting || !local->dev_enabled) {
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		if (local->dev_enabled)
			printk(KERN_DEBUG "%s: prism2_interrupt: hw not ready;"
			       " skipping events 0x%04x\n", dev->name, ev);
		HFA384X_OUTW(ev, HFA384X_EVACK_OFF);
		return;
	}

#ifndef final_version
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
		if (jiffies - last_magic_err > 10 * HZ)
			printk("%s: Interrupt, but SWSUPPORT0 does not match: "
			       "%04X != %04X - card removed?\n", dev->name,
			       HFA384X_INW(HFA384X_SWSUPPORT0_OFF),
			       HFA384X_MAGIC);
		last_magic_err = jiffies;
		prism2_hw_reset(dev);
		return;
	}
#endif

	do {
#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
		/* SMP platforms may get command completion events on other
		 * CPUs during hfa384x_cmd() execution, so use spin lock to
		 * prevent problems */
		spin_lock(&local->cmdlock);
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		spin_unlock(&local->cmdlock);
#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */
		if ((ev & HFA384X_EVENT_MASK) == 0)
			break;
		if (ev == 0xffff) {
			HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
			printk(KERN_DEBUG "%s: prism2_interrupt: ev=0xffff\n",
			       dev->name);
			prism2_hw_reset(dev);
			return;
		}

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
		if (ev & HFA384X_EV_CMD) {
			prism2_cmd_ev(dev);
			HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);
		}
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */

		if (ev & HFA384X_EV_ALLOC) {
			prism2_alloc_ev(dev);
			HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);
		}

#ifdef PRISM2_USE_TX_INTERRUPT
		if (ev & HFA384X_EV_TX) {
			local->stats.tx_packets++;
			PDEBUG(DEBUG_FID, "interrupt: TX - fid=0x%04x\n",
			       HFA384X_INW(HFA384X_TXCOMPLFID_OFF));
			HFA384X_OUTW(HFA384X_EV_TX, HFA384X_EVACK_OFF);
		}
#endif /* PRISM2_USE_TX_INTERRUPT */

		if (ev & HFA384X_EV_TXEXC) {
			prism2_txexc(dev);
			HFA384X_OUTW(HFA384X_EV_TXEXC, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_RX) {
			prism2_rx(dev);
			HFA384X_OUTW(HFA384X_EV_RX, HFA384X_EVACK_OFF);
		}

#ifndef final_version
		if (ev & HFA384X_EV_WTERR) {
			PDEBUG(DEBUG_EXTRA, "%s: WTERR event\n", dev->name);
			HFA384X_OUTW(HFA384X_EV_WTERR, HFA384X_EVACK_OFF);
		}
#endif /* final_version */

		if (ev & HFA384X_EV_INFO) {
			prism2_info(dev);
			HFA384X_OUTW(HFA384X_EV_INFO, HFA384X_EVACK_OFF);
		}

		events++;
	} while (events < 20);

	if (events >= 20)
		PDEBUG(DEBUG_EXTRA, "prism2_interrupt: >20 events\n");
}


static void prism2_check_sta_fw_version(local_info_t *local)
{
	struct hfa384x_comp_ident comp;
	int variant, major, minor;

	if (hfa384x_get_rid(local->dev, HFA384X_RID_STAID,
			    &comp, sizeof(comp), 1) < 0)
		return;

	if (__le16_to_cpu(comp.id) != 0x1f)
		return;

	major = __le16_to_cpu(comp.major);
	minor = __le16_to_cpu(comp.minor);
	variant = __le16_to_cpu(comp.variant);

	/* no firmware version specific configuration for the base driver yet,
	 * so just call AP specific parts */
	ap_check_sta_fw_version(local->ap, major, minor, variant);
}


local_info_t *prism2_init_local_data(struct prism2_helper_functions *funcs,
				     int card_idx)
{
	local_info_t *local;
	int len, i;

	if (funcs == NULL)
		return NULL;

	local = kmalloc(sizeof(local_info_t), GFP_KERNEL);
	if (local == NULL)
		return NULL;

	memset(local, 0, sizeof(local_info_t));
	local->ap = kmalloc(sizeof(struct ap_data), GFP_KERNEL);
	if (local->ap == NULL)
		goto fail;

	memset(local->ap, 0, sizeof(struct ap_data));

#ifdef PRISM2_HOSTAPD
	local->apdev = kmalloc(sizeof(struct net_device) + PRISM2_NETDEV_EXTRA,
			       GFP_KERNEL);
	if (local->apdev == NULL)
		goto fail;
	memset(local->apdev, 0, sizeof(struct net_device) +
	       PRISM2_NETDEV_EXTRA);
	prism2_set_dev_name(local->apdev, local->apdev + 1);
#endif /* PRISM2_HOSTAPD */

	local->dev = kmalloc(sizeof(struct net_device) + PRISM2_NETDEV_EXTRA,
			     GFP_KERNEL);
	if (local->dev == NULL)
		goto fail;
	memset(local->dev, 0, sizeof(struct net_device) + PRISM2_NETDEV_EXTRA);
	prism2_set_dev_name(local->dev, local->dev + 1);
	local->dev->priv = local;

	local->func = funcs;

	spin_lock_init(&local->txfidlock);
	spin_lock_init(&local->cmdlock);
	spin_lock_init(&local->baplock);
	spin_lock_init(&local->wdslock);

	if (card_idx < 0 || card_idx >= MAX_PARM_DEVICES)
		card_idx = 0;
	local->card_idx = card_idx;

	i = essid[card_idx] == NULL ? 0 : card_idx;
	len = strlen(essid[i]);
	memcpy(local->essid, essid[i],
	       len > MAX_SSID_LEN ? MAX_SSID_LEN : len);
	local->essid[MAX_SSID_LEN] = '\0';
#ifdef WIRELESS_EXT
	i = GET_INT_PARM(iw_mode, card_idx);
	if (i >= 1 && i <= 3) {
		local->iw_mode = i;
	} else {
		printk(KERN_WARNING "prism2: Unknown iw_mode %d; using "
		       "IW_MODE_MASTER\n", i);
		local->iw_mode = IW_MODE_MASTER;
	}
#endif
	local->channel = GET_INT_PARM(channel, card_idx);
	local->beacon_int = GET_INT_PARM(beacon_int, card_idx);
	local->dtim_period = GET_INT_PARM(dtim_period, card_idx);
	local->wds_max_connections = 16;
	local->tx_control = HFA384X_TX_CTRL_FLAGS;
	local->manual_retry_count = -1;

	skb_queue_head_init(&local->bridge_list);

	/* Initialize task queue structure for bridged packet handling */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	local->bridge_queue.next = NULL;
#else
	INIT_LIST_HEAD(&local->bridge_queue.list);
#endif
	local->bridge_queue.sync = 0;
	local->bridge_queue.routine = handle_bridged_queue;
	local->bridge_queue.data = local;

	prism2_setup_dev(local->dev, local, 1);

#ifdef PRISM2_HOSTAPD
	local->apdev->priv = local;
	prism2_setup_dev(local->apdev, local, 0);
	local->apdev->hard_start_xmit = prism2_tx_80211;
	local->apdev->type = ARPHRD_IEEE80211;
	local->apdev->hard_header_parse = prism2_80211_header_parse;
#endif /* PRISM2_HOSTAPD */

#ifdef PRISM2_MONITOR
	local->saved_eth_header_parse = local->dev->hard_header_parse;
#endif /* PRISM2_MONITOR */

	return local;

 fail:
	if (local->ap != NULL)
		kfree(local->ap);
	if (local->dev != NULL)
		kfree(local->dev);
#ifdef PRISM2_HOSTAPD
	if (local->apdev != NULL)
		kfree(local->apdev);
#endif /* PRISM2_HOSTAPD */
	kfree(local);
	return NULL;
}


int prism2_init_dev(local_info_t *local)
{
	struct net_device *dev = local->dev;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	{
		int i = 0;
		do {
			sprintf(dev->name, "wlan%d", i++);
		} while (dev_get(dev->name));
	}
#else
	memcpy(dev->name, "wlan%d", 7);
#endif
	if (register_netdev(dev)) {
		printk(KERN_WARNING "%s: register_netdev() failed!\n",
		       dev_info);
		return 1;
	}
	printk(KERN_INFO "%s: Registered netdevice %s\n", dev_info, dev->name);

#ifdef PRISM2_HOSTAPD
	local->apdev->base_addr = dev->base_addr;
	local->apdev->irq = dev->irq;
	local->apdev->mem_start = dev->mem_start;
	local->apdev->mem_end = dev->mem_end;
	sprintf(local->apdev->name, "%sap", dev->name);
	if (register_netdev(local->apdev)) {
		printk(KERN_WARNING "%s: register_netdev(AP) failed!\n",
		       dev_info);
		return 1;
	}
	printk(KERN_INFO "%s: Registered netdevice %s for AP management\n",
	       dev_info, local->apdev->name);
#endif /* PRISM2_HOSTAPD */

	prism2_init_proc(local);
	ap_init_data(local);
	prism2_wep_init(local);

	return 0;
}


void prism2_free_local_data(local_info_t *local)
{
	prism2_wds_info_t *wds;
	int i;

	if (local == NULL)
		return;

	if (local->ap != NULL)
		ap_free_data(local->ap);

	prism2_remove_proc(local);

	wds = local->wds;
	local->wds = NULL;
	while (wds != NULL) {
		unregister_netdev(&wds->dev);
		kfree(wds);
		wds = wds->next;
	}

#ifdef PRISM2_HOSTAPD
	if (local->apdev) {
		unregister_netdev(local->apdev);
		printk(KERN_INFO "%s: Netdevice %s unregistered\n",
		       dev_info, local->apdev->name);
		kfree(local->apdev);
	}
#endif /* PRISM2_HOSTAPD */

	if (local->dev) {
		unregister_netdev(local->dev);
		printk(KERN_INFO "%s: Netdevice %s unregistered\n",
		       dev_info, local->dev->name);
		kfree(local->dev);
	}
#ifdef PRISM2_MONITOR
	if (local->nl_monitor != NULL) {
		if (local->nl_monitor->socket == NULL)
			printk("nl_monitor->socket == NULL\n");
		/* this seems to crash the kernel.. */
		/* sock_release(local->nl_monitor->socket); */
	}
#endif /* PRISM2_MONITOR */

	for (i = 0; i < PRISM2_FRAG_CACHE_LEN; i++) {
		if (local->frag_cache[i].skb != NULL)
			dev_kfree_skb(local->frag_cache[i].skb);
	}

	kfree(local->ap);
	kfree(local);
}


/* These might at some point be compiled separately and used as separate
 * kernel modules or linked into one */
#include "prism2_ap.c"
#include "prism2_ioctl.c"
#include "prism2_proc.c"
#include "prism2_wep.c"
