/*
 * Intersil Prism2 driver with Host AP (software access point) support
 * Copyright 2001-2002, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This file is to be included into prism2.c when S/W AP functionality is
 * compiled.
 *
 * AP:  FIX:
 * - if unicast Class 2 (assoc,reassoc,disassoc) frame received from
 *   unauthenticated STA, send deauth. frame (8802.11: 5.5)
 * - if unicast Class 3 (data with to/from DS,deauth,pspoll) frame received
 *   from authenticated, but unassoc STA, send disassoc frame (8802.11: 5.5)
 * - if unicast Class 3 received from unauthenticated STA, send deauth. frame
 *   (8802.11: 5.5)
 *
 * - replace ap_queued_items array with a real queue
 */

static int other_ap_policy[MAX_PARM_DEVICES] = { AP_OTHER_AP_SKIP_ALL,
						 DEF_INTS };
MODULE_PARM(other_ap_policy, PARM_MIN_MAX "i");
MODULE_PARM_DESC(other_ap_policy, "Other AP beacon monitoring policy (0-3)");

static int ap_max_inactivity[MAX_PARM_DEVICES] = { AP_MAX_INACTIVITY / HZ,
						   DEF_INTS };
MODULE_PARM(ap_max_inactivity, PARM_MIN_MAX "i");
MODULE_PARM_DESC(ap_max_inactivity, "AP timeout (in seconds) for station "
		 "inactivity");

static int ap_bridge_packets[MAX_PARM_DEVICES] = { 1, DEF_INTS };
MODULE_PARM(ap_bridge_packets, PARM_MIN_MAX "i");
MODULE_PARM_DESC(ap_bridge_packets, "Bridge packets directly between "
		 "stations");

static int autom_ap_wds[MAX_PARM_DEVICES] = { 0, DEF_INTS };
MODULE_PARM(autom_ap_wds, PARM_MIN_MAX "i");
MODULE_PARM_DESC(autom_ap_wds, "Add WDS connections to other APs "
		 "automatically");


#ifndef PRISM2_HOSTAPD
static void prism2_send_mgmt(struct net_device *dev,
			     int type, int subtype, char *body,
			     int body_len, int txevent, u8 *addr);
#endif /* PRISM2_HOSTAPD */


static int ap_debug_proc_read(char *page, char **start, off_t off,
			      int count, int *eof, void *data)
{
	char *p = page;
	struct ap_data *ap = (struct ap_data *) data;
	int i;
	unsigned long flags;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "BridgedUnicastFrames=%u\n", ap->bridged_unicast);
	p += sprintf(p, "BridgedMulticastFrames=%u\n", ap->bridged_multicast);
	p += sprintf(p, "max_inactivity=%u\n", ap->max_inactivity / HZ);
	p += sprintf(p, "bridge_packets=%u\n", ap->bridge_packets);
	p += sprintf(p, "nullfunc_ack=%u\n", ap->nullfunc_ack);
	p += sprintf(p, "autom_ap_wds=%u\n", ap->autom_ap_wds);
	p += sprintf(p, "auth_algs=%u\n", ap->auth_algs);

	spin_lock_irqsave(&ap->ap_queue_lock, flags);
	for (i = 0; i < AP_QUEUE_LEN; i++) {
		if (ap->ap_queued_items[i] != NULL)
			p += sprintf(p, "AP_QUEUE[%d]\n", i);
	}
	spin_unlock_irqrestore(&ap->ap_queue_lock, flags);

	return (p - page);
}


static void ap_sta_hash_add(struct ap_data *ap, struct sta_info *sta)
{
	sta->hnext = ap->sta_hash[STA_HASH(sta->addr)];
	ap->sta_hash[STA_HASH(sta->addr)] = sta;
}

static void ap_sta_hash_del(struct ap_data *ap, struct sta_info *sta)
{
	struct sta_info *s;

	s = ap->sta_hash[STA_HASH(sta->addr)];
	if (s == NULL) return;
	if (memcmp(s->addr, sta->addr, 6) == 0) {
		ap->sta_hash[STA_HASH(sta->addr)] = s->hnext;
		return;
	}

	while (s->hnext != NULL && memcmp(s->hnext->addr, sta->addr, 6) != 0)
		s = s->hnext;
	if (s->hnext != NULL)
		s->hnext = s->hnext->hnext;
	else
		printk("AP: could not remove STA " MACSTR " from hash table\n",
		       MAC2STR(sta->addr));
}

static void ap_free_sta(struct ap_data *ap, struct sta_info *sta)
{
	if (ap->proc != NULL) {
		char name[20];
		sprintf(name, MACSTR, MAC2STR(sta->addr));
		remove_proc_entry(name, ap->proc);
	}

	if (sta->aid > 0)
		ap->sta_aid[sta->aid - 1] = NULL;

	if (sta->tx_buf_head != NULL) {
		struct sta_buffer_frame *prev, *fbuf =
			sta->tx_buf_head;
		while (fbuf != NULL) {
			prev = fbuf;
			dev_kfree_skb(fbuf->skb);
			fbuf = fbuf->next;
			prev->next = NULL;
			kfree(prev);
			sta->buffer_count--;
		}
		sta->tx_buf_head = sta->tx_buf_tail = NULL;
	}

	ap->num_sta--;
	if (sta->challenge)
		kfree(sta->challenge);
#ifndef PRISM2_HOSTAPD
	del_timer(&sta->timer);
#endif /* PRISM2_HOSTAPD */

	kfree(sta);
}


#ifndef PRISM2_HOSTAPD

static void ap_handle_timer(unsigned long data)
{
	struct sta_info *sta = (struct sta_info *) data;
	local_info_t *local;
	struct ap_data *ap;
	unsigned long flags;
	unsigned long next_time = 0;

	if (sta == NULL || sta->local == NULL || sta->local->ap == NULL) {
		PDEBUG(DEBUG_AP, "ap_handle_timer() called with NULL data\n");
		return;
	}

	local = sta->local;
	ap = local->ap;

	if (atomic_read(&sta->users) != 0)
		next_time = jiffies + HZ;

	if (sta->last_rx + ap->max_inactivity > jiffies) {
		/* station activity detected; reset timeout state */
		sta->timeout_next = STA_NULLFUNC;
		next_time = sta->last_rx + ap->max_inactivity;
	} else if (sta->timeout_next == STA_DISASSOC && sta->txexc == 0) {
		/* data nullfunc frame poll did not produce TX errors; assume
		 * station ACKed it */
		sta->timeout_next = STA_NULLFUNC;
		next_time = jiffies + ap->max_inactivity;
	}

	if (next_time) {
		sta->timer.expires = next_time;
		add_timer(&sta->timer);
		return;
	}

	if (sta->ap)
		sta->timeout_next = STA_DEAUTH;

	if (sta->timeout_next == STA_DEAUTH) {
		spin_lock_irqsave(&ap->sta_table_lock, flags);
		ap_sta_hash_del(ap, sta);
		list_del(&sta->list);
		spin_unlock_irqrestore(&ap->sta_table_lock, flags);
	} else if (sta->timeout_next == STA_DISASSOC)
		sta->flags &= ~WLAN_STA_ASSOC;

	if (sta->timeout_next == STA_DEAUTH && sta->aid > 0 &&
	    sta->tx_buf_head != NULL)
		prism2_set_tim(local->dev, sta->aid, 0);

	if (sta->ap) {
		if (ap->autom_ap_wds) {
			PDEBUG(DEBUG_AP, "%s: removing automatic WDS "
			       "connection to AP " MACSTR "\n",
			       local->dev->name, MAC2STR(sta->addr));
			prism2_wds_del(local, sta->addr, 0, 1);
		}
	} else if (sta->timeout_next == STA_NULLFUNC) {
		/* send data frame to poll STA and check whether this frame
		 * is ACKed */
		sta->txexc = 0;
		/* FIX: WLAN_FC_STYPE_NULLFUNC would be more appropriate, but
		 * it is apparently not retried so TX Exc events are not
		 * received for it */
		prism2_send_mgmt(local->dev, WLAN_FC_TYPE_DATA,
				 WLAN_FC_STYPE_DATA, NULL, 0, 1,
				 sta->addr);
	} else {
		int deauth = sta->timeout_next == STA_DEAUTH;
		u16 resp;
		PDEBUG(DEBUG_AP, "%s: sending %s info to STA " MACSTR
		       "(last=%lu, jiffies=%lu)\n",
		       local->dev->name,
		       deauth ? "deauthentication" : "disassociation",
		       MAC2STR(sta->addr), sta->last_rx, jiffies);

		resp = cpu_to_le16(deauth ? WLAN_REASON_PREV_AUTH_NOT_VALID :
				   WLAN_REASON_DISASSOC_DUE_TO_INACTIVITY);
		prism2_send_mgmt(local->dev, WLAN_FC_TYPE_MGMT,
				 (deauth ? WLAN_FC_STYPE_DEAUTH :
				  WLAN_FC_STYPE_DISASSOC),
				 (char *) &resp, 2, 1, sta->addr);
	}

	if (sta->timeout_next == STA_DEAUTH) {
		ap_free_sta(ap, sta);
		return;
	}

	if (sta->timeout_next == STA_NULLFUNC) {
		sta->timeout_next = STA_DISASSOC;
		sta->timer.expires = jiffies + AP_DISASSOC_DELAY;
	} else {
		sta->timeout_next = STA_DEAUTH;
		sta->timer.expires = jiffies + AP_DEAUTH_DELAY;
	}

	add_timer(&sta->timer);
}


void ap_deauth_all_stas(struct net_device *dev, struct ap_data *ap, int resend)
{
	u8 addr[ETH_ALEN];
	u16 resp;
	int i;

	PDEBUG(DEBUG_AP, "%s: Deauthenticate all stations\n", dev->name);
	memset(addr, 0xff, ETH_ALEN);

	resp = __constant_cpu_to_le16(WLAN_REASON_PREV_AUTH_NOT_VALID);

	/* deauth message sent; try to resend it few times; the message is
	 * broadcast, so it may be delayed until next DTIM; there is not much
	 * else we can do at this point since the driver is going to be shut
	 * down */
	for (i = 0; i < 5; i++) {
		prism2_send_mgmt(dev, WLAN_FC_TYPE_MGMT, WLAN_FC_STYPE_DEAUTH,
				 (char *) &resp, 2, 1, addr);

		if (!resend || ap->num_sta <= 0)
			return;

		mdelay(50);
	}
}


static int ap_control_proc_read(char *page, char **start, off_t off,
				int count, int *eof, void *data)
{
	char *p = page;
	struct ap_data *ap = (struct ap_data *) data;
	char *policy_txt;
	struct list_head *ptr;
	struct mac_entry *entry;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	switch (ap->mac_restrictions.policy) {
	case MAC_POLICY_OPEN:
		policy_txt = "open";
		break;
	case MAC_POLICY_ALLOW:
		policy_txt = "allow";
		break;
	case MAC_POLICY_DENY:
		policy_txt = "deny";
		break;
	default:
		policy_txt = "unknown";
		break;
	};
	p += sprintf(p, "MAC policy: %s\n", policy_txt);
	p += sprintf(p, "MAC entries: %u\n", ap->mac_restrictions.entries);
	p += sprintf(p, "MAC list:\n");
	spin_lock_bh(&ap->mac_restrictions.lock);
	for (ptr = ap->mac_restrictions.mac_list.next;
	     ptr != &ap->mac_restrictions.mac_list; ptr = ptr->next) {
		if (p - page > PAGE_SIZE - 80) {
			p += sprintf(p, "All entries did not fit one page.\n");
			break;
		}

		entry = list_entry(ptr, struct mac_entry, list);
		p += sprintf(p, MACSTR "\n", MAC2STR(entry->addr));
	}
	spin_unlock_bh(&ap->mac_restrictions.lock);

	return (p - page);
}


int ap_control_add_mac(struct mac_restrictions *mac_restrictions, u8 *mac)
{
	struct mac_entry *entry;

	entry = kmalloc(sizeof(struct mac_entry), GFP_KERNEL);
	if (entry == NULL)
		return -1;

	memcpy(entry->addr, mac, 6);

	spin_lock_bh(&mac_restrictions->lock);
	list_add_tail(&entry->list, &mac_restrictions->mac_list);
	mac_restrictions->entries++;
	spin_unlock_bh(&mac_restrictions->lock);

	return 0;
}


int ap_control_del_mac(struct mac_restrictions *mac_restrictions, u8 *mac)
{
	struct list_head *ptr;
	struct mac_entry *entry;

	spin_lock_bh(&mac_restrictions->lock);
	for (ptr = mac_restrictions->mac_list.next;
	     ptr != &mac_restrictions->mac_list; ptr = ptr->next) {
		entry = list_entry(ptr, struct mac_entry, list);

		if (memcmp(entry->addr, mac, 6) == 0) {
			list_del(ptr);
			kfree(entry);
			mac_restrictions->entries--;
			spin_unlock_bh(&mac_restrictions->lock);
			return 0;
		}
	}
	spin_unlock_bh(&mac_restrictions->lock);
	return -1;
}


static int ap_control_mac_deny(struct mac_restrictions *mac_restrictions,
			       u8 *mac)
{
	struct list_head *ptr;
	struct mac_entry *entry;
	int found = 0;

	if (mac_restrictions->policy == MAC_POLICY_OPEN)
		return 0;

	spin_lock_bh(&mac_restrictions->lock);
	for (ptr = mac_restrictions->mac_list.next;
	     ptr != &mac_restrictions->mac_list; ptr = ptr->next) {
		entry = list_entry(ptr, struct mac_entry, list);

		if (memcmp(entry->addr, mac, 6) == 0) {
			found = 1;
			break;
		}
	}
	spin_unlock_bh(&mac_restrictions->lock);

	if (mac_restrictions->policy == MAC_POLICY_ALLOW)
		return !found;
	else
		return found;
}


void ap_control_flush_macs(struct mac_restrictions *mac_restrictions)
{
	struct list_head *ptr, *n;
	struct mac_entry *entry;
  
	spin_lock_bh(&mac_restrictions->lock);
	for (ptr = mac_restrictions->mac_list.next, n = ptr->next;
	     ptr != &mac_restrictions->mac_list;
	     ptr = n, n = ptr->next) {
		entry = list_entry(ptr, struct mac_entry, list);
		list_del(ptr);
		kfree(entry);
	}
	mac_restrictions->entries = 0;
	spin_unlock_bh(&mac_restrictions->lock);
}


void ap_control_kickall(struct ap_data *ap)
{
	struct list_head *ptr, *n;
	struct sta_info *sta;
	unsigned long flags;
  
	spin_lock_irqsave(&ap->sta_table_lock, flags);
	for (ptr = ap->sta_list.next, n = ptr->next; ptr != &ap->sta_list;
	     ptr = n, n = ptr->next) {
		sta = list_entry(ptr, struct sta_info, list);
		ap_sta_hash_del(ap, sta);
		list_del(&sta->list);
		ap_free_sta(ap, sta);
	}
	spin_unlock_irqrestore(&ap->sta_table_lock, flags);
}


int ap_control_kick_mac(struct ap_data *ap, struct net_device *dev, u8 *mac)
{
	struct sta_info *sta;
	unsigned long flags;
	u16 resp;

	spin_lock_irqsave(&ap->sta_table_lock, flags);
	sta = ap_get_sta(ap, mac);
	if (sta) {
		ap_sta_hash_del(ap, sta);
		list_del(&sta->list);
	}
	spin_unlock_irqrestore(&ap->sta_table_lock, flags);

	if (!sta)
		return -EINVAL;

	resp = cpu_to_le16(WLAN_REASON_PREV_AUTH_NOT_VALID);
	prism2_send_mgmt(dev, WLAN_FC_TYPE_MGMT, WLAN_FC_STYPE_DEAUTH,
			 (char *) &resp, 2, 1, sta->addr);

	ap_free_sta(ap, sta);

	return 0;
}

#endif /* PRISM2_HOSTAPD */


void ap_check_sta_fw_version(struct ap_data *ap, int major, int minor,
			     int variant)
{
	if (!ap)
		return;

	if (major == 0 && minor == 8 && variant == 0) {
		PDEBUG(DEBUG_AP, "Using data::nullfunc ACK workaround - "
		       "firmware upgrade recommended\n");
		ap->nullfunc_ack = 1;
	}
}


static void handle_ap_queue(void *data);

void ap_init_data(local_info_t *local)
{
	struct ap_data *ap = local->ap;

	if (ap == NULL) {
		printk(KERN_WARNING "ap_init_data: ap == NULL\n");
		return;
	}
	memset(ap, 0, sizeof(struct ap_data));

	ap->ap_policy = GET_INT_PARM(other_ap_policy, local->card_idx);
	ap->proc = local->proc;
	ap->bridge_packets = GET_INT_PARM(ap_bridge_packets, local->card_idx);
	ap->max_inactivity =
		GET_INT_PARM(ap_max_inactivity, local->card_idx) * HZ;
	ap->auth_algs = PRISM2_AUTH_OPEN | PRISM2_AUTH_SHARED_KEY;
	ap->autom_ap_wds = GET_INT_PARM(autom_ap_wds, local->card_idx);

	spin_lock_init(&ap->ap_queue_lock);
	spin_lock_init(&ap->sta_table_lock);
	INIT_LIST_HEAD(&ap->sta_list);

	if (ap->proc != NULL) {
		create_proc_read_entry("ap_debug", 0, ap->proc,
				       ap_debug_proc_read, ap);
	}

	/* Initialize task queue structure for AP management */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	local->ap->ap_queue.next = NULL;
#else
	INIT_LIST_HEAD(&local->ap->ap_queue.list);
#endif
	local->ap->ap_queue.sync = 0;
	local->ap->ap_queue.routine = handle_ap_queue;
	local->ap->ap_queue.data = local;

#ifndef PRISM2_HOSTAPD
	spin_lock_init(&ap->mac_restrictions.lock);
	INIT_LIST_HEAD(&ap->mac_restrictions.mac_list);
	if (ap->proc != NULL) {
		create_proc_read_entry("ap_control", 0, ap->proc,
				       ap_control_proc_read, ap);
	}
#endif /* PRISM2_HOSTAPD */
}

void ap_free_data(struct ap_data *ap)
{
	struct list_head *ptr;
	int i;

	if (ap == NULL) {
		printk(KERN_WARNING "ap_free_data: ap == NULL\n");
		return;
	}

	for (i = 0; i < AP_QUEUE_LEN; i++) {
		if (ap->ap_queued_items[i] != NULL) {
			kfree(ap->ap_queued_items[i]);
			ap->ap_queued_items[i] = NULL;
		}
	}

	ptr = ap->sta_list.next;
	while (ptr != NULL && ptr != &ap->sta_list) {
		struct sta_info *sta = (struct sta_info *) ptr;
		ptr = ptr->next;
		ap_sta_hash_del(ap, sta);
		list_del(&sta->list);
		ap_free_sta(ap, sta);
	}

	if (ap->proc != NULL) {
		remove_proc_entry("ap_debug", ap->proc);
	}

#ifndef PRISM2_HOSTAPD
	if (ap->proc != NULL)
		remove_proc_entry("ap_control", ap->proc);
	ap_control_flush_macs(&ap->mac_restrictions);
#endif /* PRISM2_HOSTAPD */
}


/* caller should have mutex for AP STA list handling */
struct sta_info* ap_get_sta(struct ap_data *ap, u8 *sta)
{
	struct sta_info *s;

	s = ap->sta_hash[STA_HASH(sta)];
	while (s != NULL && memcmp(s->addr, sta, 6) != 0)
		s = s->hnext;
	return s;
}


#ifndef PRISM2_HOSTAPD

/* Called only from software IRQ */
static void prism2_send_mgmt(struct net_device *dev,
			     int type, int subtype, char *body,
			     int body_len, int txevent, u8 *addr)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct hfa384x_tx_frame txdesc;
	int res, idx;
	u16 fc;
	u8 wepbuf[8];

	memset(&txdesc, 0, sizeof(txdesc));
	/* FIX: set tx_rate if f/w does not know how to do it */
	txdesc.tx_control = cpu_to_le16(txevent ? local->tx_control :
					HFA384X_TX_CTRL_802_11);
	txdesc.data_len = cpu_to_le16(body_len);

	fc = (type << 2) | (subtype << 4);

	memcpy(txdesc.addr1, addr, ETH_ALEN); /* DA / RA */
	if (type == WLAN_FC_TYPE_DATA) {
		fc |= WLAN_FC_FROMDS;
		memcpy(txdesc.addr2, dev->dev_addr, ETH_ALEN); /* BSSID */
		memcpy(txdesc.addr3, dev->dev_addr, ETH_ALEN); /* SA */
	} else if (type == WLAN_FC_TYPE_CTRL) {
		/* control:ACK does not have addr2 or addr3 */
		memset(txdesc.addr2, 0, ETH_ALEN);
		memset(txdesc.addr3, 0, ETH_ALEN);
	} else {
		memcpy(txdesc.addr2, dev->dev_addr, ETH_ALEN); /* SA */
		memcpy(txdesc.addr3, dev->dev_addr, ETH_ALEN); /* BSSID */
	}

	if (local->use_wep && type == WLAN_FC_TYPE_DATA) {
		fc |= WLAN_FC_ISWEP;
		/* FIX: could support >0 len body.. */
		txdesc.data_len = cpu_to_le16(8);
		prism2_wep_encrypt(local, wepbuf, 0, NULL);
	}

	txdesc.frame_control = cpu_to_le16(fc);

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		return;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR)
		prism2_dump_tx_header(dev->name, &txdesc);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res && local->use_wep && type == WLAN_FC_TYPE_DATA)
		res = hfa384x_to_bap(dev, BAP0, wepbuf, 8);
	else if (!res && body != NULL)
		res = hfa384x_to_bap(dev, BAP0, body, body_len);
	spin_unlock_bh(&local->baplock);
	if (!res)
		res = prism2_transmit(dev, idx);

	if (res) {
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		printk(KERN_DEBUG "%s: prism2_send_mgmt - to BAP0 failed\n",
		       dev->name);
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
	}
}
#endif /* PRISM2_HOSTAPD */


static int prism2_sta_proc_read(char *page, char **start, off_t off,
				int count, int *eof, void *data)
{
	char *p = page;
	struct sta_info *sta = (struct sta_info *) data;
	int i;

	/* FIX: possible race condition.. the STA data could have just expired,
	 * but proc entry was still here so that the read could have started;
	 * some locking should be done here.. */

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "%s=" MACSTR "\nusers=%d\naid=%d\n"
		     "flags=0x%04x%s%s%s\n"
		     "capability=0x%02x\nlisten_interval=%d\nsupported_rates=",
		     sta->ap ? "AP" : "STA",
		     MAC2STR(sta->addr), atomic_read(&sta->users), sta->aid,
		     sta->flags,
		     sta->flags & WLAN_STA_AUTH ? " AUTH" : "",
		     sta->flags & WLAN_STA_ASSOC ? " ASSOC" : "",
		     sta->flags & WLAN_STA_PS ? " PS" : "",
		     sta->capability, sta->listen_interval);
	/* supported_rates: 500 kbit/s units with msb ignored */
	for (i = 0; i < sizeof(sta->supported_rates); i++)
		if (sta->supported_rates[i] != 0)
			p += sprintf(p, "%d%sMbps ",
				     (sta->supported_rates[i] & 0x7f) / 2,
				     sta->supported_rates[i] & 1 ? ".5" : "");
	p += sprintf(p, "\njiffies=%lu\nlast_auth=%lu\nlast_assoc=%lu\n"
		     "last_rx=%lu\nlast_tx=%lu\nrx_packets=%lu\n"
		     "tx_packets=%lu\n"
		     "rx_bytes=%lu\ntx_bytes=%lu\nbuffer_count=%d\n"
		     "last_rx: silence=%d signal=%d rate=%d flow=%d\n"
		     "tx_rate=%d\ntx[1M]=%d\ntx[2M]=%d\ntx[5.5M]=%d\n"
		     "tx[11M]=%d\n"
		     "rx[1M]=%d\nrx[2M]=%d\nrx[5.5M]=%d\nrx[11M]=%d\n",
		     jiffies, sta->last_auth, sta->last_assoc, sta->last_rx,
		     sta->last_tx,
		     sta->rx_packets, sta->tx_packets, sta->rx_bytes,
		     sta->tx_bytes, sta->buffer_count, sta->last_rx_silence,
		     sta->last_rx_signal, sta->last_rx_rate,
		     sta->last_rx_flow,
		     sta->tx_rate, sta->tx_count[0], sta->tx_count[1],
		     sta->tx_count[2], sta->tx_count[3],  sta->rx_count[0],
		     sta->rx_count[1], sta->rx_count[2], sta->rx_count[3]);

	return (p - page);
}


/* Called only from software IRQ */
static void add_sta_proc(struct proc_dir_entry *proc_dir, struct sta_info *sta)
{
	char name[20];

	if (proc_dir == NULL)
		return;

	sprintf(name, MACSTR, MAC2STR(sta->addr));
	sta->proc = create_proc_read_entry(name, 0, proc_dir,
					   prism2_sta_proc_read, sta);
}


#ifndef PRISM2_HOSTAPD

/* Called only from software IRQ */
static void handle_authen(local_info_t *local, struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct ap_data *ap = local->ap;
	char body[8 + WLAN_AUTH_CHALLENGE_LEN], *challenge = NULL;
	int len;
	u16 auth_alg, auth_transaction, status_code, *pos;
	u16 resp = WLAN_STATUS_SUCCESS, fc;
	struct sta_info *sta = NULL;
	unsigned long flags;

	len = __le16_to_cpu(rxdesc->data_len);

	fc = le16_to_cpu(rxdesc->frame_control);

	if (local->use_wep && local->host_decrypt && (fc & WLAN_FC_ISWEP) &&
	    len >= 8) {
		if (prism2_wep_decrypt(local, (u8 *) (rxdesc + 1), len, NULL))
		{
			PDEBUG(DEBUG_AP, "%s: handle_authen: auth frame from "
			       "STA " MACSTR " could not be decrypted\n",
			       dev->name, MAC2STR(rxdesc->addr2));
			return;
		}
		len -= 8;
	}

	if (len < 6) {
		PDEBUG(DEBUG_AP, "%s: handle_authen - too short payload "
		       "(len=%d)\n", dev->name, len);
		return;
	}

	pos = (u16 *) (rxdesc + 1);
	auth_alg = __le16_to_cpu(*pos);
	pos++;
	auth_transaction = __le16_to_cpu(*pos);
	pos++;
	status_code = __le16_to_cpu(*pos);
	pos++;

	PDEBUG(DEBUG_AP, "%s: authentication: " MACSTR " len=%d, auth_alg=%d, "
	       "auth_transaction=%d, status_code=%d, fc=0x%04x\n", dev->name,
	       MAC2STR(rxdesc->addr2), len, auth_alg, auth_transaction,
	       status_code, fc);

	if (ap_control_mac_deny(&ap->mac_restrictions, rxdesc->addr2)) {
		PDEBUG(DEBUG_AP, "%s: STA " MACSTR " authentication denied\n",
		       dev->name, MAC2STR(rxdesc->addr2));
		resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
		goto fail;
	}

	if (((ap->auth_algs & PRISM2_AUTH_OPEN) &&
	     auth_alg == WLAN_AUTH_OPEN) ||
	    ((ap->auth_algs & PRISM2_AUTH_SHARED_KEY) &&
	     local->use_wep && auth_alg == WLAN_AUTH_SHARED_KEY)) {
	} else {
		PDEBUG(DEBUG_AP, "%s: Unsupported authentication algorithm "
		       "(%d)\n",
		       dev->name, auth_alg);
		resp = WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG;
		goto fail;
	}

	if (len >= 8) {
		u8 *u = (u8 *) pos;
		if (*u == WLAN_EID_CHALLENGE) {
			if (*(u + 1) != WLAN_AUTH_CHALLENGE_LEN) {
				PDEBUG(DEBUG_AP, "%s: invalid challenge len "
				       "%d\n", dev->name, *(u + 1));
				resp = WLAN_STATUS_CHALLENGE_FAIL;
				goto fail;
			}
			if (len - 8 < WLAN_AUTH_CHALLENGE_LEN) {
				PDEBUG(DEBUG_AP, "%s: challenge underflow\n",
				       dev->name);
				resp = WLAN_STATUS_CHALLENGE_FAIL;
				goto fail;
			}
			challenge = (char *) (u + 2);
		}
	}

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);


	if ((auth_alg == WLAN_AUTH_OPEN && auth_transaction == 1) ||
	    (auth_alg == WLAN_AUTH_SHARED_KEY &&
	     (auth_transaction == 1 ||
	      (auth_transaction == 3 && sta != NULL &&
	       sta->challenge != NULL)))) {
	} else {
		PDEBUG(DEBUG_AP, "%s: Unknown authentication transaction "
		       "number (%d)\n", dev->name, auth_transaction);
		resp = WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION;
		goto fail;
	}

	if (sta == NULL) {
		PDEBUG(DEBUG_AP, "%s: new STA " MACSTR "\n", dev->name,
		       MAC2STR(rxdesc->addr2));

		if (local->ap->num_sta >= MAX_STA_COUNT) {
			/* FIX: might try to remove some old STAs first? */
			PDEBUG(DEBUG_AP, "%s: no more room for new STAs "
			       "(%d/%d)\n", dev->name,
			       local->ap->num_sta, MAX_STA_COUNT);
			resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
			goto fail;
		}

		sta = (struct sta_info *)
			kmalloc(sizeof(struct sta_info), GFP_ATOMIC);
		if (sta == NULL) {
			PDEBUG(DEBUG_AP, "AP: kmalloc failed\n");
			resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
			goto fail;
		}
		memset(sta, 0, sizeof(struct sta_info));

		/* initialize STA info data */
#ifndef PRISM2_HOSTAPD
		sta->local = local;
		init_timer(&sta->timer);
		sta->timer.expires = jiffies + local->ap->max_inactivity;
		sta->timer.data = (unsigned long) sta;
		sta->timer.function = ap_handle_timer;
		add_timer(&sta->timer);
#endif /* PRISM2_HOSTAPD */
		memcpy(&sta->addr, &rxdesc->addr2, 6);

		atomic_inc(&sta->users);
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		list_add(&sta->list, &local->ap->sta_list);
		local->ap->num_sta++;
		ap_sta_hash_add(local->ap, sta);
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		add_sta_proc(local->ap->proc, sta);
	}

	prism2_ap_update_sq(sta, rxdesc);

	switch (auth_alg) {
	case WLAN_AUTH_OPEN:
		sta->flags |= WLAN_STA_AUTH;
		break;

	case WLAN_AUTH_SHARED_KEY:
		if (auth_transaction == 1) {
			if (sta->challenge == NULL) {
				u8 key[WEP_KEY_LEN];
				sta->challenge = (char *)
					kmalloc(WLAN_AUTH_CHALLENGE_LEN,
						GFP_ATOMIC);
				if (sta->challenge == NULL) {
					PDEBUG(DEBUG_AP, "AP: kmalloc failed "
					       "for challenge\n");
					resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
					goto fail;
				}
				memset(body, 0, sizeof(body));
				get_random_bytes(key, WEP_KEY_LEN);
				prism2_wep_encrypt(local, body,
						   WLAN_AUTH_CHALLENGE_LEN,
						   key);
				memcpy(sta->challenge, body + 4,
				       WLAN_AUTH_CHALLENGE_LEN);
			}
		} else {
			if (sta->challenge == NULL || challenge == NULL ||
			    memcmp(sta->challenge, challenge,
				   WLAN_AUTH_CHALLENGE_LEN) != 0 ||
				!(fc & WLAN_FC_ISWEP)) {
				PDEBUG(DEBUG_AP, "%s: challenge response "
				       "incorrect\n", dev->name);
				resp = WLAN_STATUS_CHALLENGE_FAIL;
				goto fail;
			}

			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " authenticated\n",
			       dev->name, MAC2STR(sta->addr));
			sta->flags |= WLAN_STA_AUTH;
			kfree(sta->challenge);
			sta->challenge = NULL;
		}
		break;
	}

 fail:
	pos = (u16 *) body;
	*pos = cpu_to_le16(auth_alg);
	pos++;
	*pos = cpu_to_le16(auth_transaction + 1);
	pos++;
	*pos = cpu_to_le16(resp); /* status_code */
	pos++;
	len = 6;

	if (resp == WLAN_STATUS_SUCCESS && sta != NULL &&
	    sta->challenge != NULL &&
	    auth_alg == WLAN_AUTH_SHARED_KEY && auth_transaction == 1) {
		u8 *tmp = (u8 *) pos;
		*tmp++ = WLAN_EID_CHALLENGE;
		*tmp++ = WLAN_AUTH_CHALLENGE_LEN;
		pos++;
		memcpy(pos, sta->challenge, WLAN_AUTH_CHALLENGE_LEN);
		len += 2 + WLAN_AUTH_CHALLENGE_LEN;
	}

	prism2_send_mgmt(dev, WLAN_FC_TYPE_MGMT, WLAN_FC_STYPE_AUTH,
			 body, len, 1, rxdesc->addr2);

	if (sta) {
		sta->last_auth = sta->last_rx = jiffies;
		atomic_dec(&sta->users);
	}
}


static void prism2_check_tx_rates(struct sta_info *sta)
{
	int i;

	sta->tx_supp_rates = 0;
	for (i = 0; i < sizeof(sta->supported_rates); i++) {
		if ((sta->supported_rates[i] & 0x7f) == 2)
			sta->tx_supp_rates |= WLAN_RATE_1M;
		if ((sta->supported_rates[i] & 0x7f) == 4)
			sta->tx_supp_rates |= WLAN_RATE_2M;
		if ((sta->supported_rates[i] & 0x7f) == 11)
			sta->tx_supp_rates |= WLAN_RATE_5M5;
		if ((sta->supported_rates[i] & 0x7f) == 22)
			sta->tx_supp_rates |= WLAN_RATE_11M;
	}
	sta->tx_max_rate = sta->tx_rate = sta->tx_rate_idx = 0;
	if (sta->tx_supp_rates & WLAN_RATE_1M) {
		sta->tx_rate = 10;
		sta->tx_max_rate = sta->tx_rate_idx = 0;
	}
	if (sta->tx_supp_rates & WLAN_RATE_2M) {
		sta->tx_rate = 20;
		sta->tx_max_rate = sta->tx_rate_idx = 1;
	}
	if (sta->tx_supp_rates & WLAN_RATE_5M5) {
		sta->tx_rate = 55;
		sta->tx_max_rate = sta->tx_rate_idx = 2;
	}
	if (sta->tx_supp_rates & WLAN_RATE_11M) {
		sta->tx_rate = 110;
		sta->tx_max_rate = sta->tx_rate_idx = 3;
	}
}


/* Called only from software IRQ */
static void handle_assoc(local_info_t *local, struct hfa384x_rx_frame *rxdesc,
			 int reassoc)
{
	struct net_device *dev = local->dev;
	char body[12], *p;
	int len, left;
	u16 *pos;
	u16 resp = WLAN_STATUS_SUCCESS;
	struct sta_info *sta = NULL;
	int send_deauth = 0;
	unsigned long flags;

	left = len = __le16_to_cpu(rxdesc->data_len);

	if (len < (reassoc ? 10 : 4)) {
		printk("handle_assoc - too short payload "
		       "(len=%d, reassoc=%d)\n", len, reassoc);
		return;
	}

	PDEBUG(DEBUG_AP, "%s: %sassociation request: from " MACSTR
	       " to " MACSTR " len=%d\n",
	       dev->name, reassoc ? "re" : "", MAC2STR(rxdesc->addr2),
	       MAC2STR(rxdesc->addr1), len);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta == NULL || (sta->flags & WLAN_STA_AUTH) == 0) {
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
		printk("AP: STA " MACSTR " trying to associate before "
		       "authentication\n", MAC2STR(rxdesc->addr2));
		if (sta) {
			PDEBUG(DEBUG_AP, "sta: addr=" MACSTR
			       " aid=%d flags=0x%04x\n",
			       MAC2STR(sta->addr), sta->aid, sta->flags);
		}
		send_deauth = 1;
		resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
		goto fail;
	}
	atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	prism2_ap_update_sq(sta, rxdesc);

	pos = (u16 *) (rxdesc + 1);
	sta->capability = __le16_to_cpu(*pos);
	pos++; left -= 2;
	sta->listen_interval = __le16_to_cpu(*pos);
	pos++; left -= 2;
	PDEBUG(DEBUG_AP, "  capability=0x%04x, listen_interval=%d",
	       sta->capability, sta->listen_interval);

	if (reassoc) {
		PDEBUG2(DEBUG_AP, ", current AP: " MACSTR,
			MAC2STR((u8 *) pos));
		pos++; pos++; pos++; left -= 6;
	}

	if (left >= 2) {
		unsigned int ileft;
		unsigned char *u = (unsigned char *) pos;

		if (*u == WLAN_EID_SSID) {
			u++; left--;
			ileft = *u;
			u++; left--;

			if (ileft > left || ileft > MAX_SSID_LEN) {
				PDEBUG2(DEBUG_AP, " - SSID overflow\n");
				atomic_dec(&sta->users);
				return;
			}

			if (ileft != strlen(local->essid) ||
			    memcmp(local->essid, u, ileft) != 0) {
				PDEBUG2(DEBUG_AP, " - not our SSID\n");
				atomic_dec(&sta->users);
				return;
			}

			u += ileft;
			left -= ileft;
		}

		if (left >= 2 && *u == WLAN_EID_SUPP_RATES) {
			u++; left--;
			ileft = *u;
			u++; left--;
			
			if (ileft > left || ileft == 0 || ileft > 8) {
				PDEBUG2(DEBUG_AP, " - SUPP_RATES len error\n");
				atomic_dec(&sta->users);
				return;
			}

			memset(sta->supported_rates, 0,
			       sizeof(sta->supported_rates));
			memcpy(sta->supported_rates, u, ileft);
			prism2_check_tx_rates(sta);

			u += ileft;
			left -= ileft;
		}

		if (left > 0) {
			PDEBUG2(DEBUG_AP, "\n  extra data (%d bytes) [", left);
			while (left > 0) {
				PDEBUG2(DEBUG_AP, "<%02x>", *u);
				u++; left--;
			}
			PDEBUG2(DEBUG_AP, "]\n");
		}
	} else {
		PDEBUG2(DEBUG_AP, " - frame underflow (left=%d)\n",
			left);
		atomic_dec(&sta->users);
		return;
	}

	/* get a unique AID */
	if (sta->aid > 0)
		PDEBUG2(DEBUG_AP, " - old AID %d\n", sta->aid);
	else {
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		for (sta->aid = 1; sta->aid <= MAX_AID_TABLE_SIZE; sta->aid++)
			if (local->ap->sta_aid[sta->aid - 1] == NULL)
				break;
		if (sta->aid > MAX_AID_TABLE_SIZE) {
			sta->aid = 0;
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			resp = WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA;
			PDEBUG2(DEBUG_AP, " - no room for more AIDs\n");
		} else {
			local->ap->sta_aid[sta->aid - 1] = sta;
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			PDEBUG2(DEBUG_AP, " - new AID %d\n", sta->aid);
		}
	}

 fail:
	pos = (u16 *) body;

	if (send_deauth) {
		*pos = __constant_cpu_to_le16(
			WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH);
		pos++;
	} else {
		/* FIX: CF-Pollable and CF-PollReq should be set to match the
		 * values in beacons/probe responses */
		/* FIX: how about privacy and WEP? */
		/* capability */
		*pos = __constant_cpu_to_le16(WLAN_CAPABILITY_ESS);
		pos++;

		/* status_code */
		*pos = __cpu_to_le16(resp);
		pos++;

		*pos = __cpu_to_le16((sta && sta->aid > 0 ? sta->aid : 0) |
				     BIT(14) | BIT(15)); /* AID */
		pos++;

		/* Supported rates (Information element) */
		p = (char *) pos;
		*p++ = WLAN_EID_SUPP_RATES;
		*p++ = 4; /* len */
		*p++ = 0x82; /* 1 Mbps, base set */
		*p++ = 0x84; /* 2 Mbps, base set */
		*p++ = 0x0b; /* 5.5 Mbps */
		*p++ = 0x16; /* 11 Mbps */
	}

	prism2_send_mgmt(dev, WLAN_FC_TYPE_MGMT,
			 (send_deauth ? WLAN_FC_STYPE_DEAUTH :
			  (reassoc ? WLAN_FC_STYPE_REASSOC_RESP :
			   WLAN_FC_STYPE_ASSOC_RESP)),
			 body, send_deauth ? 2 : sizeof(body), 1,
			 rxdesc->addr2);

	if (sta) {
		if (resp == WLAN_STATUS_SUCCESS) {
			sta->flags |= WLAN_STA_ASSOC;
			sta->last_assoc = sta->last_rx = jiffies;
		}
		atomic_dec(&sta->users);
	}
}


/* Called only from software IRQ */
static void handle_deauth(local_info_t *local, struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	char *body = (char *) (rxdesc + 1);
	int len;
	u16 reason_code, *pos;
	struct sta_info *sta = NULL;
	unsigned long flags;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 2) {
		printk("handle_deauth - too short payload (len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	reason_code = __le16_to_cpu(*pos);

	PDEBUG(DEBUG_AP, "%s: deauthentication: " MACSTR " len=%d, "
	       "reason_code=%d\n", dev->name, MAC2STR(rxdesc->addr2), len,
	       reason_code);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta != NULL) {
		sta->flags &= ~(WLAN_STA_AUTH | WLAN_STA_ASSOC);
		prism2_ap_update_sq(sta, rxdesc);
	}
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
	if (sta == NULL) {
		printk("%s: deauthentication from " MACSTR ", "
	       "reason_code=%d, but STA not authenticated\n", dev->name,
		       MAC2STR(rxdesc->addr2), reason_code);
	}
}


/* Called only from software IRQ */
static void handle_disassoc(local_info_t *local,
			    struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	char *body = (char *) (rxdesc + 1);
	int len;
	u16 reason_code, *pos;
	struct sta_info *sta = NULL;
	unsigned long flags;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 2) {
		printk("handle_disassoc - too short payload (len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	reason_code = __le16_to_cpu(*pos);

	PDEBUG(DEBUG_AP, "%s: disassociation: " MACSTR " len=%d, "
	       "reason_code=%d\n", dev->name, MAC2STR(rxdesc->addr2), len,
	       reason_code);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta != NULL) {
		sta->flags &= ~WLAN_STA_ASSOC;
		prism2_ap_update_sq(sta, rxdesc);
	}
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
	if (sta == NULL) {
		printk("%s: disassociation from " MACSTR ", "
	       "reason_code=%d, but STA not authenticated\n", dev->name,
		       MAC2STR(rxdesc->addr2), reason_code);
	}
}


/* Called only from software IRQ */
static void ap_handle_data_nullfunc(local_info_t *local,
				    struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;

	/* some STA f/w's seem to require control::ACK frame for
	 * data::nullfunc, but at least Prism2 station f/w version 0.8.0 does
	 * not send this..
	 * send control::ACK for the data::nullfunc */

	printk(KERN_DEBUG "Sending control::ACK for data::nullfunc\n");
	prism2_send_mgmt(dev, WLAN_FC_TYPE_CTRL, WLAN_FC_STYPE_ACK,
			 NULL, 0, 0, rxdesc->addr2);
}


/* Called only from software IRQ */
static void ap_handle_dropped_data(local_info_t *local,
				   struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct sta_info *sta;
	u16 reason;
	unsigned long flags;

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	if (sta != NULL && (sta->flags & WLAN_STA_ASSOC)) {
		PDEBUG(DEBUG_AP, "ap_handle_dropped_data: STA is now okay?\n");
		atomic_dec(&sta->users);
		return;
	}

	reason = __constant_cpu_to_le16(
		WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA);
	prism2_send_mgmt(dev, WLAN_FC_TYPE_MGMT,
			 ((sta == NULL || !(sta->flags & WLAN_STA_ASSOC)) ?
			  WLAN_FC_STYPE_DEAUTH : WLAN_FC_STYPE_DISASSOC),
			 (char *) &reason, sizeof(reason), 1,
			 rxdesc->addr2);

	if (sta)
		atomic_dec(&sta->users);
}

#endif /* PRISM2_HOSTAPD */


/* Called only from software IRQ */
static void handle_pspoll(local_info_t *local,
			  struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct sta_info *sta;
	u16 aid;
	struct sta_buffer_frame *fbuf;
	unsigned long flags;

	PDEBUG(DEBUG_PS2, "handle_pspoll: BSSID=" MACSTR ", TA=" MACSTR "\n",
	       MAC2STR(rxdesc->addr1), MAC2STR(rxdesc->addr2));

	if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_pspoll - addr1(BSSID)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr1));
		return;
	}

	aid = __le16_to_cpu(rxdesc->duration_id);
	if ((aid & (BIT(15) | BIT(14))) != (BIT(15) | BIT(14))) {
		PDEBUG(DEBUG_PS, "   PSPOLL and AID[15:14] not set\n");
		return;
	}
	aid &= ~BIT(15) & ~BIT(14);
	if (aid == 0 || aid > MAX_AID_TABLE_SIZE) {
		PDEBUG(DEBUG_PS, "   invalid aid=%d\n", aid);
		return;
	}
	PDEBUG(DEBUG_PS2, "   aid=%d\n", aid);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	if (sta == NULL) {
		PDEBUG(DEBUG_PS, "   STA not found\n");
		return;
	}
	prism2_ap_update_sq(sta, rxdesc);
	if (sta->aid != aid) {
		PDEBUG(DEBUG_PS, "   received aid=%i does not match with "
		       "assoc.aid=%d\n", aid, sta->aid);
		return;
	}

	/* FIX: todo:
	 * - add timeout for buffering (clear aid in TIM vector if buffer timed
	 *   out (expiry time must be longer than ListenInterval for
	 *   the corresponding STA; "8802-11: 11.2.1.9 AP aging function"
	 * - what to do, if buffered, pspolled, and sent frame is not ACKed by
	 *   sta; store buffer for later use and leave TIM aid bit set? use
	 *   TX event to check whether frame was ACKed?
	 */

	for (;;) {
		int len, res, idx;

		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		fbuf = sta->tx_buf_head;
		if (fbuf != NULL) {
			sta->tx_buf_head = fbuf->next;
			if (fbuf->next == NULL)
				sta->tx_buf_tail = NULL;
			sta->buffer_count--;
		}
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (fbuf == NULL) {
			if (sta->buffer_count != 0) {
				printk("STA frame buffer empty, but "
				       "buffer_count=%d\n", sta->buffer_count);
			}
			break;
		}

		/* send buffered frame .. */
		PDEBUG(DEBUG_PS2, "Sending buffered frame to STA after PS POLL"
		       " (buffer_count=%d)\n", sta->buffer_count);

		len = fbuf->skb->len;

		idx = prism2_get_txfid_idx(local);
		if (idx < 0)
			goto fail;

		if (sta->buffer_count > 0) {
			/* indicate to STA that more frames follow */
			fbuf->txdesc.frame_control |=
				__constant_cpu_to_le16(WLAN_FC_MOREDATA);
		}

		if (local->frame_dump & PRISM2_DUMP_TX_HDR)
			prism2_dump_tx_header(dev->name, &fbuf->txdesc);

		spin_lock_bh(&local->baplock);
		res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, &fbuf->txdesc,
					     sizeof(fbuf->txdesc));
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, &snap_header,
					     sizeof(snap_header));
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, fbuf->skb->data + 12,
					     fbuf->skb->len - 12);
		spin_unlock_bh(&local->baplock);
		if (!res)
			res = prism2_transmit(dev, idx);

		if (res) {
			local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
			printk("%s: handle_pspoll - to BAP0 failed\n",
			       dev->name);
			if (res == -ETIMEDOUT)
				prism2_hw_reset(dev);
			goto fail;
		}

		local->stats.tx_bytes += len;

		sta->tx_packets++;
		sta->tx_bytes += len;
		sta->last_tx = jiffies;

	fail:
		dev_kfree_skb(fbuf->skb);
		kfree(fbuf);

		if (sta->flags & WLAN_STA_PS) {
			/* send only one buffered packet per PS Poll */
			/* FIX: should ignore further PS Polls until the
			 * buffered packet that was just sent is acknowledged
			 * (Tx or TxExc event) */
			break;
		}
	}


	if (sta->buffer_count == 0) {
		/* try to clear aid from TIM */
		prism2_set_tim(dev, aid, 0);
	}

	atomic_dec(&sta->users);
}


void prism2_ap_update_sq(struct sta_info *sta, struct hfa384x_rx_frame *rxdesc)
{
	sta->last_rx_silence = rxdesc->silence;
	sta->last_rx_signal = rxdesc->signal;
	sta->last_rx_rate = rxdesc->rate;
	sta->last_rx_flow = rxdesc->rxflow;
	sta->last_rx_updated = 7;
	if (rxdesc->rate == 10)
		sta->rx_count[0]++;
	else if (rxdesc->rate == 20)
		sta->rx_count[1]++;
	else if (rxdesc->rate == 55)
		sta->rx_count[2]++;
	else if (rxdesc->rate == 110)
		sta->rx_count[3]++;
}


#ifndef PRISM2_HOSTAPD

/* Called only from software IRQ */
static void handle_beacon(local_info_t *local, struct hfa384x_rx_frame *rxdesc)
{
	char *body = (char *) (rxdesc + 1);
	int len, left;
	u16 *pos, beacon_int, capability;
	char *ssid = NULL;
	unsigned char *supp_rates = NULL;
	int ssid_len = 0, supp_rates_len = 0;
	unsigned long flags;
	struct sta_info *sta = NULL;
	int new_sta = 0;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 8 + 2 + 2) {
		printk(KERN_DEBUG "handle_beacon - too short payload "
		       "(len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	left = len;

	/* Timestamp (8 octets) */
	pos += 4; left -= 8;
	/* Beacon interval (2 octets) */
	beacon_int = __le16_to_cpu(*pos);
	pos++; left -= 2;
	/* Capability information (2 octets) */
	capability = __le16_to_cpu(*pos);
	pos++; left -= 2;

	if (local->ap->ap_policy != AP_OTHER_AP_EVEN_IBSS &&
	    capability & WLAN_CAPABILITY_IBSS)
		return;

	if (left >= 2) {
		unsigned int ileft;
		unsigned char *u = (unsigned char *) pos;

		if (*u == WLAN_EID_SSID) {
			u++; left--;
			ileft = *u;
			u++; left--;

			if (ileft > left || ileft > MAX_SSID_LEN) {
				PDEBUG(DEBUG_AP, "SSID: overflow\n");
				return;
			}

			if (local->ap->ap_policy == AP_OTHER_AP_SAME_SSID &&
			    (ileft != strlen(local->essid) ||
			     memcmp(local->essid, u, ileft) != 0)) {
				/* not our SSID */
				return;
			}

			ssid = u;
			ssid_len = ileft;

			u += ileft;
			left -= ileft;
		}

		if (*u == WLAN_EID_SUPP_RATES) {
			u++; left--;
			ileft = *u;
			u++; left--;
			
			if (ileft > left || ileft == 0 || ileft > 8) {
				PDEBUG(DEBUG_AP, " - SUPP_RATES len error\n");
				return;
			}

			supp_rates = u;
			supp_rates_len = ileft;

			u += ileft;
			left -= ileft;
		}
	}

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta != NULL)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	if (sta == NULL) {
		/* add new AP */
		new_sta = 1;
		sta = (struct sta_info *)
			kmalloc(sizeof(struct sta_info), GFP_ATOMIC);
		if (sta == NULL) {
			printk(KERN_INFO "prism2: kmalloc failed for AP "
			       "data structure\n");
			return;
		}
		memset(sta, 0, sizeof(struct sta_info));
		list_add(&sta->list, &local->ap->sta_list);
		local->ap->num_sta++;
		atomic_inc(&sta->users);

		/* initialize STA info data */
		memcpy(&sta->addr, &rxdesc->addr2, 6);
		/* mark APs authentication and associated for pseudo ad-hoc
		 * style communication */
		sta->flags = WLAN_STA_AUTH | WLAN_STA_ASSOC;

		if (local->ap->autom_ap_wds) {
			PDEBUG(DEBUG_AP, "%s: adding automatic WDS connection "
			       "to AP " MACSTR "\n",
			       local->dev->name, MAC2STR(sta->addr));
			prism2_wds_add(local, sta->addr, 0);
		}
	}

	sta->ap = 1;
	sta->rx_packets++;
	sta->rx_bytes += len;
	sta->last_rx = jiffies;
	sta->capability = capability;
	sta->listen_interval = beacon_int;
	prism2_ap_update_sq(sta, rxdesc);

	atomic_dec(&sta->users);

	if (new_sta) {
		memset(sta->supported_rates, 0, sizeof(sta->supported_rates));
		memcpy(sta->supported_rates, supp_rates, supp_rates_len);
		prism2_check_tx_rates(sta);

		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		ap_sta_hash_add(local->ap, sta);
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		add_sta_proc(local->ap->proc, sta);
	}
}

#endif /* PRISM2_HOSTAPD */


/* Called only from software IRQ */
static void handle_ap_item(local_info_t *local,
			   struct hfa384x_rx_frame *rxdesc)
{
#ifndef PRISM2_HOSTAPD
	struct net_device *dev = local->dev;
#endif /* PRISM2_HOSTAPD */
	u16 fc, type, stype;

	fc = __le16_to_cpu(rxdesc->frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

#ifndef PRISM2_HOSTAPD
	if (type == WLAN_FC_TYPE_DATA) {
		PDEBUG(DEBUG_AP, "handle_ap_item - data frame\n");

		if (!(fc & WLAN_FC_TODS) || (fc & WLAN_FC_FROMDS)) {
			if (stype == WLAN_FC_STYPE_NULLFUNC) {
				/* no ToDS nullfunc seems to be used to check
				 * AP association; so send reject message to
				 * speed up re-association */
				ap_handle_dropped_data(local, rxdesc);
				goto done;
			}
			PDEBUG(DEBUG_AP, "   not ToDS frame (fc=0x%04x)\n",
			       fc);
			goto done;
		}

		if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
			PDEBUG(DEBUG_AP, "handle_ap_item - addr1(BSSID)="
			       MACSTR " not own MAC\n",
			       MAC2STR(rxdesc->addr1));
			goto done;
		}

		if (local->ap->nullfunc_ack && stype == WLAN_FC_STYPE_NULLFUNC)
			ap_handle_data_nullfunc(local, rxdesc);
		else
			ap_handle_dropped_data(local, rxdesc);
		goto done;
	}
#endif /* PRISM2_HOSTAPD */

	if (type == WLAN_FC_TYPE_CTRL &&
	    stype == WLAN_FC_STYPE_PSPOLL) {
		handle_pspoll(local, rxdesc);
		goto done;
	}

#ifdef PRISM2_HOSTAPD
	PDEBUG(DEBUG_AP, "Unknown frame in AP queue: type=0x%02x "
	       "subtype=0x%02x\n", type, stype);

#else /* PRISM2_HOSTAPD */
	if (type != WLAN_FC_TYPE_MGMT) {
		PDEBUG(DEBUG_AP, "handle_ap_item - not a management frame?\n");
		goto done;
	}

	if (stype == WLAN_FC_STYPE_BEACON) {
		handle_beacon(local, rxdesc);
		goto done;
	}

	if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_ap_item - addr1(DA)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr1));
		goto done;
	}

	if (memcmp(rxdesc->addr3, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_ap_item - addr3(BSSID)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr3));
		goto done;
	}

	switch (stype) {
	case WLAN_FC_STYPE_ASSOC_REQ:
		handle_assoc(local, rxdesc, 0);
		break;
	case WLAN_FC_STYPE_ASSOC_RESP:
		PDEBUG(DEBUG_AP, "==> ASSOC RESP (ignored)\n");
		break;
	case WLAN_FC_STYPE_REASSOC_REQ:
		handle_assoc(local, rxdesc, 1);
		break;
	case WLAN_FC_STYPE_REASSOC_RESP:
		PDEBUG(DEBUG_AP, "==> REASSOC RESP (ignored)\n");
		break;
	case WLAN_FC_STYPE_ATIM:
		PDEBUG(DEBUG_AP, "==> ATIM (ignored)\n");
		break;
	case WLAN_FC_STYPE_DISASSOC:
		handle_disassoc(local, rxdesc);
		break;
	case WLAN_FC_STYPE_AUTH:
		handle_authen(local, rxdesc);
		break;
	case WLAN_FC_STYPE_DEAUTH:
		handle_deauth(local, rxdesc);
		break;
	default:
		PDEBUG(DEBUG_AP, "Unknown mgmt frame subtype 0x%02x\n", stype);
		break;
	}
#endif /* PRISM2_HOSTAPD */

 done:
	kfree(rxdesc);
}


/* Called only from software IRQ */
static void handle_ap_queue(void *data)
{
	local_info_t *local = (local_info_t *) data;
	struct hfa384x_rx_frame *rxdesc;
	int i;

	if (local->func->card_present && !local->func->card_present(local)) {
		printk(KERN_DEBUG "prism2: handle_ap_queue, but dev not OK\n");
		goto out;
	}

	for (;;) {
		int type = 0;

		rxdesc = NULL;
		spin_lock_irq(&local->ap->ap_queue_lock);
		for (i = 0; i < AP_QUEUE_LEN; i++) {
			if (local->ap->ap_queued_items[i] != NULL) {
				rxdesc = (struct hfa384x_rx_frame *)
					local->ap->ap_queued_items[i];
				type = local->ap->ap_queued_type[i];
				local->ap->ap_queued_items[i] = NULL;
			}
		}
		spin_unlock_irq(&local->ap->ap_queue_lock);

		if (rxdesc == NULL)
			break;

		if (type == AP_QUEUED_RXDESC)
			handle_ap_item(local, rxdesc);
		else
			printk(KERN_DEBUG "handle_ap_queue: unknown type %d\n",
			       type);
	}

 out:
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	MOD_DEC_USE_COUNT;
#endif
}


/* Called only from hardware IRQ */
void hostap_rx(struct net_device *dev, struct hfa384x_rx_frame *rxdesc,
	       int force)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fc, type, stype, len;
	char *buf;
	int res, i;

	fc = __le16_to_cpu(rxdesc->frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	/* skip data frames, unless forced to accept them */
	if (!force &&
	    type != WLAN_FC_TYPE_MGMT && type != WLAN_FC_TYPE_CTRL) {
#ifndef final_version
	  PDEBUG(DEBUG_AP, "%s: hostap_rx - hostap ignored frame "
		 "(type=0x%02x, subtype=0x%02x)\n", dev->name, type, stype);
#endif
		return;
	}

	if (local->ap->ap_policy == AP_OTHER_AP_SKIP_ALL &&
	    type == WLAN_FC_TYPE_MGMT && stype == WLAN_FC_STYPE_BEACON)
		return;

	len = __le16_to_cpu(rxdesc->data_len);

	buf = (char *) kmalloc(sizeof(*rxdesc) + len, GFP_ATOMIC);
	if (buf == NULL) {
		printk(KERN_DEBUG "%s: hostap_rx - could not allocate memory "
		       "for queued item\n", dev->name);
		return;
	}
	memcpy(buf, rxdesc, sizeof(*rxdesc));
	res = hfa384x_from_bap(dev, BAP1, buf + sizeof(*rxdesc), len);
	if (res) {
		printk(KERN_DEBUG "%s: data copy from BAP1 failed %d\n",
		       dev->name, res);
		kfree(buf);
		return;
	}

	local->stats.rx_packets++;

	spin_lock(&local->ap->ap_queue_lock);
	for (i = 0; i < AP_QUEUE_LEN && local->ap->ap_queued_items[i] != NULL;
	     i++) {}
	if (i < AP_QUEUE_LEN) {
		local->ap->ap_queued_items[i] = buf;
		local->ap->ap_queued_type[i] = AP_QUEUED_RXDESC;
	}
	spin_unlock(&local->ap->ap_queue_lock);
	if (i >= AP_QUEUE_LEN) {
		printk(KERN_DEBUG "%s: AP queue full - dropping new message\n",
		       dev->name);
		kfree(buf);
		return;
	}

/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	queue_task(&local->ap->ap_queue, &tq_scheduler);
#else
	MOD_INC_USE_COUNT;
	if (schedule_task(&local->ap->ap_queue) == 0)
		MOD_DEC_USE_COUNT;
#endif
}


/* Called only from hardware IRQ */
void schedule_packet_send(local_info_t *local, struct sta_info *sta)
{
	int i;
	struct hfa384x_rx_frame *rxdesc;

	if (sta->buffer_count == 0)
		return;

	rxdesc = (struct hfa384x_rx_frame *)
		kmalloc(sizeof(*rxdesc), GFP_ATOMIC);
	if (rxdesc == NULL) {
		printk(KERN_WARNING "schedule_packet_send: kmalloc failed\n");
		return;
	}

	/* Generate a fake pspoll frame to start packet delivery */
	memset(rxdesc, 0, sizeof(*rxdesc));
	rxdesc->frame_control = __constant_cpu_to_le16(
		(WLAN_FC_TYPE_CTRL << 2) | (WLAN_FC_STYPE_PSPOLL << 4));
	memcpy(rxdesc->addr1, local->dev->dev_addr, 6);
	memcpy(rxdesc->addr2, sta->addr, 6);
	rxdesc->duration_id = __cpu_to_le16(sta->aid | BIT(15) | BIT(14));

	spin_lock(&local->ap->ap_queue_lock);
	for (i = 0; i < AP_QUEUE_LEN && local->ap->ap_queued_items[i] != NULL;
	     i++) {}
	if (i < AP_QUEUE_LEN)
		local->ap->ap_queued_items[i] = rxdesc;
	spin_unlock(&local->ap->ap_queue_lock);
	if (i >= AP_QUEUE_LEN) {
		printk(KERN_INFO "schedule_packet_send: AP queue full\n");
		kfree(rxdesc);
		return;
	}

	PDEBUG(DEBUG_PS2, "Scheduling buffered packet delivery for STA\n");

/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	queue_task(&local->ap->ap_queue, &tq_scheduler);
#else
	MOD_INC_USE_COUNT;
	if (schedule_task(&local->ap->ap_queue) == 0)
		MOD_DEC_USE_COUNT;
#endif
}


#ifdef WIRELESS_EXT
int prism2_ap_get_sta_qual(local_info_t *local, struct sockaddr addr[],
			   struct iw_quality qual[], int buf_size,
			   int aplist)
{
	struct ap_data *ap = local->ap;
	struct list_head *ptr;
	int count = 0;
	unsigned long flags;

	spin_lock_irqsave(&ap->sta_table_lock, flags);

	for (ptr = ap->sta_list.next; ptr != NULL && ptr != &ap->sta_list;
	     ptr = ptr->next) {
		struct sta_info *sta = (struct sta_info *) ptr;

		if (aplist && !sta->ap)
			continue;
		addr[count].sa_family = ARPHRD_ETHER;
		memcpy(addr[count].sa_data, sta->addr, ETH_ALEN);
		if (sta->last_rx_silence == 0)
			qual[count].qual = sta->last_rx_signal < 27 ?
				0 : (sta->last_rx_signal - 27) * 92 / 127;
		else
			qual[count].qual = sta->last_rx_signal -
				sta->last_rx_silence - 35;
		qual[count].level = HFA384X_LEVEL_TO_dBm(sta->last_rx_signal);
		qual[count].noise = HFA384X_LEVEL_TO_dBm(sta->last_rx_silence);
		qual[count].updated = sta->last_rx_updated;

		sta->last_rx_updated = 0;

		count++;
		if (count >= buf_size)
			break;
	}
	spin_unlock_irqrestore(&ap->sta_table_lock, flags);

	return count;
}
#endif /* WIRELESS_EXT */
