/*
 *      Home-agent functionality
 *
 *      Authors:
 *      Sami Kivisaari           <skivisaa@cc.hut.fi>
 *      Henrik Petander          <lpetande@cc.hut.fi>
 *
 *      $Id: ha.c,v 1.49 2002/08/02 12:31:56 ville Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *   
 *      Changes: Venkata Jagana,
 *               Krishna Kumar     : Statistics fix
 *     
 */

#include <linux/autoconf.h>
#include <linux/net.h>
#include <linux/skbuff.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/in6.h>
#include <linux/init.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv6.h>
#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif

#include <net/neighbour.h>
#include <net/ipv6.h>
#include <net/ip6_fib.h>
#include <net/ip6_route.h>
#include <net/ndisc.h>
#include <net/addrconf.h>
#include <net/neighbour.h>
#include <net/ipv6_tunnel.h>
#include <net/mipv6.h>

#include "bcache.h"
#include "tunnel.h"
#include "stats.h"
#include "ha.h"
#include "debug.h"
#include "access.h"
#include "dhaad.h"

static int mipv6_ha_tunnel_sitelocal = 1;

#ifdef CONFIG_SYSCTL
#include "sysctl.h"

static spinlock_t proxy_nd_lock = SPIN_LOCK_UNLOCKED;

static struct ctl_table_header *mipv6_ha_sysctl_header;

static struct mipv6_ha_sysctl_table
{
	struct ctl_table_header *sysctl_header;
	ctl_table mipv6_vars[3];
	ctl_table mipv6_mobility_table[2];
	ctl_table mipv6_proto_table[2];
	ctl_table mipv6_root_table[2];
} mipv6_ha_sysctl = {
	NULL,

        {{NET_IPV6_MOBILITY_TUNNEL_SITELOCAL, "tunnel_sitelocal",
	  &mipv6_ha_tunnel_sitelocal, sizeof(int), 0644, NULL, 
	  &proc_dointvec},
	 {0}},

	{{NET_IPV6_MOBILITY, "mobility", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_vars}, {0}},
	{{NET_IPV6, "ipv6", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_mobility_table}, {0}},
	{{CTL_NET, "net", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_proto_table}, {0}}
};

#endif /* CONFIG_SYSCTL */

/*  this should be in some header file but it isn't  */
extern void ndisc_send_na(
	struct net_device *dev, struct neighbour *neigh,
	struct in6_addr *daddr, struct in6_addr *solicited_addr,
	int router, int solicited, int override, int inc_opt);

/*  this is defined in kernel IPv6 module (sockglue.c)  */
extern struct packet_type ipv6_packet_type;

/**
 * mipv6_lifetime_check - check maximum lifetime is not exceeded
 * @lifetime: lifetime to check
 *
 * Checks @lifetime does not exceed %MAX_LIFETIME.  Returns @lifetime
 * if not exceeded, otherwise returns %MAX_LIFETIME.
 **/
int mipv6_lifetime_check(int lifetime)
{
	return (lifetime > MAX_LIFETIME) ? MAX_LIFETIME : lifetime;
}

/*
 * Extern functions.
 */
extern int mipv6_execute_list(struct prefix_info *plist, 
		struct in6_addr *home_addr, int count,
		int (*func)(struct in6_addr *, void *), void *arg);
extern int mipv6_get_prefix_entries(struct in6_addr *home_addr,
		struct prefix_info **plist, int single,
		int ifindex);

/**
 * mipv6_proxy_nd_rem - stop acting as a proxy for @home_address
 * @home_addr: address to remove
 * @ha_addr: home agent's address on home link
 * @prefix_length: prefix length in bits
 * @single: single bit
 *
 * When Home Agent acts as a proxy for an address it must leave the
 * solicited node multicast group for that address and stop responding 
 * to neighbour solicitations.  
 **/
int mipv6_proxy_nd_rem(
	struct in6_addr *home_addr,
	int ifindex,
	int prefix_length,
	int single)
{
        /* When MN returns home HA leaves the solicited mcast groups
         * for MNs home addresses 
	 */
	int err = -1;
	struct net_device *dev;
	
	DEBUG_FUNC();
	
	spin_lock_bh(&proxy_nd_lock);
	
        if ((dev = dev_get_by_index(ifindex)) == NULL) {
		DEBUG((DBG_ERROR, "couldn't get dev"));
		goto out;
	}
	
	/* Remove link-local entry */
	if (!single) {
		struct in6_addr ll_addr;
		mipv6_generate_ll_addr(&ll_addr, home_addr);
		if ((err = pneigh_delete(&nd_tbl, &ll_addr, dev)) < 0) {
			DEBUG((DBG_INFO,
			       "peigh_delete failed for "
			       "%x:%x:%x:%x:%x:%x:%x:%x",
			       NIPV6ADDR(&ll_addr)));	
		}
	}
	/* Remove global (or site-local) entry */
	if ((err = pneigh_delete(&nd_tbl, home_addr, dev)) < 0) {
		DEBUG((DBG_INFO,
		       "peigh_delete failed for " 
		       "%x:%x:%x:%x:%x:%x:%x:%x",
		       NIPV6ADDR(home_addr)));
	}
	dev_put(dev);
out:
	spin_unlock_bh(&proxy_nd_lock);
	
	return err;
}

/**
 * mipv6_proxy_nd - join multicast group for this address
 * @home_addr: address to defend
 * @ha_addr: home agent's address on home link
 * @prefix_length: prefix length in bits
 * @single: single bit
 *
 * While Mobile Node is away from home, Home Agent acts as a proxy for
 * @home_address. HA responds to neighbour solicitations for  @home_address 
 * thus getting all packets destined to home address of MN. 
 **/
int mipv6_proxy_nd(struct in6_addr *home_addr, 
		   int ifindex,
		   int prefix_length,
		   int single)
{  
	/* The HA sends a proxy ndisc_na message to all hosts on MN's
	 * home subnet by sending a neighbor advertisement with the
	 * home address or all addresses of the mobile node if the
	 * prefix is not 0. The addresses are formed by combining the
	 * suffix or the host part of the address with each subnet
	 * prefix that exists in the home subnet 
	 */
	
        /* Since no previous entry for MN exists a proxy_nd advertisement
	 * is sent to all nodes link local multicast address
	 */	
	int err = -1;

	struct net_device *dev;
	struct in6_addr ll_addr;
	struct pneigh_entry *ll_pneigh;
	struct in6_addr mcdest;
	int send_ll_na = 0;
	int inc_opt = 1;
	int solicited = 0;
	int override = 1;
	
	DEBUG_FUNC();
	
	spin_lock_bh(&proxy_nd_lock);
	if ((dev = dev_get_by_index(ifindex)) == NULL) {
		DEBUG((DBG_ERROR, "couldn't get dev"));
		goto out;
	}
	
	if (!pneigh_lookup(&nd_tbl, home_addr, dev, 1)) {
		DEBUG((DBG_INFO,
		       "peigh_lookup failed for "
		       "%x:%x:%x:%x:%x:%x:%x:%x",
		       NIPV6ADDR(home_addr)));
		goto free_dev;
	}
	if (!single) {
		mipv6_generate_ll_addr(&ll_addr, home_addr);
		
		if ((ll_pneigh = pneigh_lookup(&nd_tbl, &ll_addr, dev, 
					       0)) != NULL) {
			ll_pneigh = pneigh_clone(ll_pneigh);
		} else if ((ll_pneigh = pneigh_lookup(&nd_tbl, &ll_addr, 
						      dev, 1)) == NULL) {
			DEBUG((DBG_INFO,
			       "peigh_lookup failed for "
			       "%x:%x:%x:%x:%x:%x:%x:%x",
			       NIPV6ADDR(&ll_addr)));
			pneigh_delete(&nd_tbl, home_addr, dev);
			goto free_dev;
		} else {
			send_ll_na = 1;
		}
	} else {
		ll_pneigh = NULL;
	}
	
	/* Proxy neighbor advertisement of MN's home address 
	 * to all nodes solicited multicast address 
	 */
	
	ipv6_addr_all_nodes(&mcdest); 
	ndisc_send_na(dev, NULL, &mcdest, home_addr, 0, 
		      solicited, override, inc_opt);
	if (send_ll_na) {
		ndisc_send_na(dev, NULL, &mcdest, &ll_addr, 0, 
			      solicited, override, inc_opt);
	}
	err = 0;
free_dev:
	dev_put(dev);
out:
	spin_unlock_bh(&proxy_nd_lock);
	return err;
	
}

#if 0
/**
 * get_protocol_data - Get pointer to packet payload
 * @skb: pointer to packet buffer
 * @nexthdr: store payload type here
 *
 * Returns a pointer to protocol payload of IPv6 packet.  Protocol
 * code is stored in @nexthdr.
 **/
static __inline__ __u8 *get_protocol_data(struct sk_buff *skb, __u8 *nexthdr)
{
	int offset = (__u8 *)(skb->nh.ipv6h + 1) - skb->data;
	int len = skb->len - offset;

	*nexthdr = skb->nh.ipv6h->nexthdr;

	offset = ipv6_skip_exthdr(skb, offset, nexthdr, len);

	return skb->data + offset;
}

#endif

extern int mipv6_ra_rcv_ptr(struct sk_buff *skb, struct icmp6hdr *msg);

/**
 * mipv6_intercept - Netfilter hook to intercept packets
 * @hooknum: which hook we came from
 * @p_skb: pointer to *skb
 * @in: interface we came in
 * @out: outgoing interface
 * @okfn: next handler
 **/

static unsigned int mipv6_intercept(
        unsigned int hooknum,
	struct sk_buff **p_skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
{
	struct sk_buff *skb = (p_skb) ? *p_skb : NULL;
	struct ipv6hdr *ipv6h;
	struct in6_addr *daddr, *saddr;
	__u8 nexthdr;
	int nhoff;	

	if(skb == NULL) return NF_ACCEPT;
	
	ipv6h = skb->nh.ipv6h;
	daddr = &ipv6h->daddr;
	saddr = &ipv6h->saddr;
		
	nexthdr = ipv6h->nexthdr;
	nhoff = sizeof(*ipv6h);
	
	
	if (ipv6_ext_hdr(nexthdr)) 
		nhoff = ipv6_skip_exthdr(skb, nhoff, &nexthdr, 
					 skb->len - sizeof(*ipv6h));

	/*
	 * Possible ICMP packets are checked to ensure that all neighbor 
	 * solicitations to MNs home address are handled by the HA.
	 */

	if (nexthdr == IPPROTO_ICMPV6) {			
		struct icmp6hdr *icmp6h;
		int dest_type;

		if (nhoff < 0 || 
		    !pskb_may_pull(skb, nhoff + sizeof(struct icmp6hdr)))
			return NF_DROP;

		dest_type = ipv6_addr_type(daddr);
		icmp6h = (struct icmp6hdr *)&skb->nh.raw[nhoff];

		/* HA has to capture all unicast neighbour solicitations in 
		   order to check if it is acting as a proxy for the target 
		   address. */
		
		if ((dest_type & IPV6_ADDR_UNICAST) && 
		    icmp6h->icmp6_type == NDISC_NEIGHBOUR_SOLICITATION) {
			ip6_input(skb);
			return NF_STOLEN;
		} else if ((dest_type & IPV6_ADDR_MULTICAST) && 
			   icmp6h->icmp6_type == NDISC_ROUTER_ADVERTISEMENT) {
			mipv6_ra_rcv_ptr(skb, icmp6h);
		}
	}
	return NF_ACCEPT;
}

#if 0
/**
 * ha_nsol_rcv - Neighbor solicitation handler
 * @skb: packet buffer
 * @saddr: source address
 * @daddr: destination address
 * @dest_type: destination address type
 *
 * Handles incoming Neighbor Solicitation.  Used in Proxy Neighbor
 * Discovery.  Home Agents defends registered MN addresses when DAD is
 * in process.
 **/
static int ha_nsol_rcv(
	struct sk_buff *skb, 
	struct in6_addr *saddr, 
	struct in6_addr *daddr, 
	int dest_type)
{
	__u8 nexthdr;
	struct nd_msg *msg;
	struct neighbour *neigh;
	struct net_device *dev = skb->dev;
	struct mipv6_bcache_entry bc_entry;
	int inc, s_type = ipv6_addr_type(saddr);
	int override;

	msg = (struct nd_msg *) get_protocol_data(skb, &nexthdr);
	DEBUG((DBG_INFO, "Neighbor solicitation received, to "
	       "%x:%x:%x:%x:%x:%x:%x:%x of type %d",
	       NIPV6ADDR(daddr), (dest_type & IPV6_ADDR_MULTICAST)));
	if (!msg)
	  return -1;

	if (msg->opt.opt_len == 0) return 1;
	if (msg->opt.opt_type == ND_OPT_SOURCE_LL_ADDR) {
		neigh = __neigh_lookup(&nd_tbl, saddr, skb->dev, 1);
		if (neigh) {
			neigh_update(neigh, msg->opt.link_addr, NUD_VALID, 1, 1);
			neigh_release(neigh);
			DEBUG((DBG_INFO, "Got SOURCE_LL_ADDR. nd_tbl updated."));
		}
	}
	else
		DEBUG((DBG_ERROR, " Unexpected option in neighbour solicitation: %x", msg->opt.opt_type));

	if (ipv6_addr_type(&msg->target) & 
				(IPV6_ADDR_UNICAST|IPV6_ADDR_ANYCAST)) {
		override = 0;
	} else {
		override = 1;
	}

	if (s_type & IPV6_ADDR_UNICAST) {
		inc = dest_type & IPV6_ADDR_MULTICAST;

                /* TODO: Obviously, a check should be done right about here
		 * whether the address is an on-link address for mobile node
		 */ 
		if (inc) {
			DEBUG((DBG_INFO,
			    "nsol was multicast to %x:%x:%x:%x:%x:%x:%x:%x",
			    NIPV6ADDR(&msg->target)));
		}
		if (mipv6_bcache_get(&msg->target, &bc_entry) >= 0) {
			if (inc) {
				DEBUG((DBG_INFO, 
				    "multicast neighbour lookup succeeded"));
			} else {
				DEBUG((DBG_INFO, 
				    "unicast neighbour lookup succeeded"));
			}
			ndisc_send_na(dev, NULL, saddr, &msg->target,
					bc_entry.router, 1, override, 1);
			return 0;
		}
		if (inc) {
			DEBUG((DBG_INFO, "Multicast neighbor unknown"));
		} else {
			DEBUG((DBG_INFO,"Unicast neighbor unknown"));
		}
		return 1;
	}

	if (s_type == IPV6_ADDR_ANY && mipv6_bcache_get(&msg->target,
				&bc_entry) >= 0) {
		struct in6_addr ret_addr;

		ipv6_addr_all_nodes(&ret_addr);
		if (dest_type & IPV6_ADDR_ANYCAST) {
			DEBUG((DBG_INFO, "neighbor solicitation for MN with "
			       "unspecified source and AC destination"));
		} else {
			DEBUG((DBG_INFO, "neighbor solicitation for MN with "
			       "unspecified source"));
		}
		ndisc_send_na(dev, NULL, &ret_addr, &msg->target,
				bc_entry.router, 0, override, 1);
		return 0;
	}

	return 1;
}
#endif

/*
 * Netfilter hook for packet interception
 */

static struct nf_hook_ops intercept_hook_ops = {
        {NULL, NULL},     // List head, no predecessor, no successor
        mipv6_intercept,
        PF_INET6,
        NF_IP6_PRE_ROUTING,
        NF_IP6_PRI_LAST
};

/**
 * mipv6_ha_tnl_xmit_drop_local_hook - drop local packets 
 * @skb: outgoing skb
 * @flags: flags set by tunnel device driver
 *
 *
 * Return:
 * %IPV6_TNL_ACCEPT if packet can be sent through tunnel,
 * %IPV6_TNL_DROP if packet is invalid,
 **/

static int
mipv6_ha_tnl_xmit_drop_local_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	int dest_type;

	DEBUG_FUNC();

       	/* If this isn't a tunnel used for Mobile IPv6 return */
	if (!is_mipv6_tnl(t))
		return IPV6_TNL_ACCEPT;

	dest_type = ipv6_addr_type(&skb->nh.ipv6h->daddr);

	if ((dest_type & IPV6_ADDR_LINKLOCAL) ||
	    ((dest_type & IPV6_ADDR_SITELOCAL) && 
	     !mipv6_ha_tunnel_sitelocal)) 
		return IPV6_TNL_DROP;
	
	return IPV6_TNL_ACCEPT;
}
	
static struct ipv6_tnl_hook_ops mipv6_ha_tnl_xmit_drop_local_ops = {
	{NULL, NULL}, 
	IPV6_TNL_PRE_ENCAP,
	IPV6_TNL_PRI_FIRST,
	mipv6_ha_tnl_xmit_drop_local_hook
};

static int
mipv6_ha_tnl_xmit_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_encapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_ha_tnl_xmit_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_ENCAP,
	IPV6_TNL_PRI_LAST,
	mipv6_ha_tnl_xmit_stats_hook
};

static int
mipv6_ha_tnl_rcv_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_decapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_ha_tnl_rcv_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_DECAP,
	IPV6_TNL_PRI_LAST,
	mipv6_ha_tnl_rcv_stats_hook
};

int __init mipv6_initialize_ha(void)
{
	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	mipv6_ha_sysctl_header = 
		register_sysctl_table(mipv6_ha_sysctl.mipv6_root_table, 0);
#endif
	if (mipv6_initialize_access() < 0)
		DEBUG((DBG_INFO, "mipv6_initialize_access() failed"));

	/*  register packet interception hooks  */
	nf_register_hook(&intercept_hook_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_xmit_drop_local_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_xmit_stats_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_rcv_stats_ops);
	return 0;
}

void __exit mipv6_shutdown_ha(void)
{
	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	unregister_sysctl_table(mipv6_ha_sysctl_header);
#endif

	/*  remove packet interception hooks  */
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_rcv_stats_ops);
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_xmit_stats_ops);
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_xmit_drop_local_ops);
	nf_unregister_hook(&intercept_hook_ops);
	mipv6_destroy_access();
}

