/**
 * Prefix solicitation and advertisement
 *
 * Authors:
 * Jaakko Laine <medved@iki.fi>
 *
 * $Id: prefix.c,v 1.15 2002/08/16 08:25:09 jola Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/icmpv6.h>
#include <linux/net.h>
#include <linux/spinlock.h>
#include <linux/timer.h>
#include <linux/netdevice.h>
#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/ip6_route.h>
#include <net/mipv6.h>

#include "mipv6_icmp.h"
#include "debug.h"
#include "sortedlist.h"
#include "util.h"
#include "ha.h"
#include "mn.h"
#include "bcache.h"
#include "mdetect.h"

#define INFINITY 0xffffffff

static struct timer_list pfx_timer;

struct pfx_list_entry {
	struct in6_addr daddr;
	struct in6_addr saddr;
	int retries;
	int ifindex;
};

static struct list_head pfx_list;
static rwlock_t pfx_list_lock = RW_LOCK_UNLOCKED;

/**
 * send_pxf_sol - send a prefix solicitation to home agent
 * @saddr: source address of the solicitation
 * @daddr: destination address of the solicitation
 * @ifindex: interface index
 * @initial: is this initial prefix solicitation
 */
static void inline send_pfx_sol(struct in6_addr *daddr,
				struct in6_addr *saddr, int ifindex)
{
	DEBUG_FUNC();

	mipv6_icmpv6_send(daddr, saddr, MIPV6_PREFIX_SOLICIT, 0, NULL,
			  NULL, 0);
}

/**
 * send_pfx_adv - send prefix advertisement to MN
 * @daddr: destination address
 * @saddr: source address
 * @id: packet identifier
 * @iif: interface index
 */
static void send_pfx_adv(struct in6_addr *daddr, struct in6_addr *saddr,
			 __u16 id, int iif, int send_br)
{
	int count;
	struct prefix_info *plist;

	DEBUG_FUNC();

	if ((count = ipv6_get_prefix_entries(&plist, iif, 0)) > 0) {
		mipv6_icmpv6_send(daddr, saddr, MIPV6_PREFIX_ADV, 0, &id,
				  plist, count * sizeof(struct prefix_info));
		kfree(plist);
	}
}

static int compare_pfx_list_entry(const void *data1, const void *data2,
				  int datalen)
{
	struct pfx_list_entry *e1 = (struct pfx_list_entry *) data1;
	struct pfx_list_entry *e2 = (struct pfx_list_entry *) data2;

	return ((ipv6_addr_cmp(&e1->daddr, &e2->daddr) == 0)
		&& (e2->ifindex == -1 || e1->ifindex == e2->ifindex));
}

/**
 * mipv6_pfx_cancel_send - cancel pending items to daddr from saddr
 * @daddr: Destination address
 * @ifindex: pending items on this interface will be canceled
 *
 * if ifindex == -1, all items to daddr will be removed
 */
void mipv6_pfx_cancel_send(struct in6_addr *daddr, int ifindex)
{
	unsigned long flags, tmp;
	struct pfx_list_entry entry;

	DEBUG_FUNC();

	/* We'll just be comparing these parts... */
	memcpy(&entry.daddr, daddr, sizeof(struct in6_addr));
	entry.ifindex = ifindex;

	write_lock_irqsave(&pfx_list_lock, flags);

	while (mipv6_slist_del_item(&pfx_list, &entry,
				    compare_pfx_list_entry) == 0)
		;

	if ((tmp = mipv6_slist_get_first_key(&pfx_list)))
		mod_timer(&pfx_timer, tmp);

	write_unlock_irqrestore(&pfx_list_lock, flags);
}

/**
 * mipv6_pfx_add_ha - add a new HA to send prefix solicitations to
 * @daddr: address of HA
 * @saddr: our address to use as source address
 * @ifindex: interface index
 */
void mipv6_pfx_add_ha(struct in6_addr *daddr, struct in6_addr *saddr,
		      int ifindex)
{
	unsigned long flags, tmp;
	struct pfx_list_entry entry;

	DEBUG_FUNC();

	memcpy(&entry.daddr, daddr, sizeof(struct in6_addr));
	memcpy(&entry.saddr, saddr, sizeof(struct in6_addr));
	entry.retries = 0;
	entry.ifindex = ifindex;

	write_lock_irqsave(&pfx_list_lock, flags);
	if (mipv6_slist_modify(&pfx_list, &entry, sizeof(struct pfx_list_entry),
			       jiffies + INITIAL_SOLICIT_TIMER * HZ,
			       compare_pfx_list_entry))
		DEBUG((DBG_WARNING, "Cannot add new HA to pfx list"));

	if ((tmp = mipv6_slist_get_first_key(&pfx_list)))
		mod_timer(&pfx_timer, tmp);
	write_unlock_irqrestore(&pfx_list_lock, flags);
}

/**
 * pfx_adv_iterator - modify pfx_list entries according to new prefix info
 * @data: MN's home registration bcache_entry
 * @args: new prefix info
 * @sortkey: ignored
 */
static int pfx_adv_iterator(void *data, void *args, unsigned long sortkey)
{
	struct mipv6_bcache_entry *bc_entry =
		(struct mipv6_bcache_entry *) data;
	struct prefix_info *pinfo = (struct prefix_info *) args;

	if (mipv6_prefix_compare(&bc_entry->coa, &pinfo->prefix,
				 pinfo->prefix_len) == 0) {
		struct pfx_list_entry pfx_entry;

		memcpy(&pfx_entry.daddr, &bc_entry->coa,
		       sizeof(struct in6_addr));
		memcpy(&pfx_entry.daddr, &bc_entry->our_addr,
		       sizeof(struct in6_addr));
		pfx_entry.retries = 0;
		pfx_entry.ifindex = bc_entry->ifindex;

		mipv6_slist_modify(&pfx_list, &pfx_entry,
				   sizeof(struct pfx_list_entry),
				   jiffies +
				   net_random() % (MAX_PFX_ADV_DELAY * HZ),
				   compare_pfx_list_entry);
	}

	return 0;
}

/**
 * mipv6_prefix_added - prefix was added to interface, act accordingly
 * @pinfo: prefix_info that was added
 * @ifindex: interface index
 */
void mipv6_pfxs_modified(struct prefix_info *pinfo, int ifindex)
{
	int count;
	unsigned long flags, tmp;
	struct list_head home_regs;

	DEBUG_FUNC();

	INIT_LIST_HEAD(&home_regs);

	if (!(count = mipv6_bcache_get_homeregs(&home_regs)))
		return;

	write_lock_irqsave(&pfx_list_lock, flags);
	mipv6_slist_for_each(&home_regs, pinfo, pfx_adv_iterator);
	if ((tmp = mipv6_slist_get_first_key(&pfx_list)))
		mod_timer(&pfx_timer, tmp);
	write_unlock_irqrestore(&pfx_list_lock, flags);
}

/**
 * mipv6_handle_pfx_icmpv6 - handle prefix advertisements and solicitations
 * @skb: sk_buff including the icmp6 message
 */
int mipv6_handle_pfx_icmpv6(struct sk_buff *skb)
{
	struct icmp6hdr *hdr = (struct icmp6hdr *) skb->h.raw;
	struct in6_addr *saddr = &skb->nh.ipv6h->saddr;
	struct in6_addr *daddr = &skb->nh.ipv6h->daddr;
	__u16 identifier = ntohs(hdr->icmp6_identifier);
	__u8 *opt = (__u8 *) (hdr + 1);
	int optlen = (skb->tail - opt);
	unsigned long min_expire = INFINITY;
	struct inet6_skb_parm *parm = (struct inet6_skb_parm *) skb->cb;

	DEBUG_FUNC();

	if (hdr->icmp6_type == MIPV6_PREFIX_SOLICIT && mipv6_is_ha) {
		struct inet6_ifaddr *ifp;

		if (!(ifp = ipv6_get_ifaddr(daddr, NULL)))
			return -1;

		send_pfx_adv(saddr, daddr, identifier,
			     ifp->idev->dev->ifindex, 0);
		in6_ifa_put(ifp);
		mipv6_pfx_cancel_send(saddr, -1);
	} else if (hdr->icmp6_type == MIPV6_PREFIX_ADV && mipv6_is_mn) {
		unsigned long flags, tmp;

		while (optlen > 0) {
			int len = opt[1] << 3;
			if (len == 0)
				goto set_timer;

			if (opt[0] == ND_OPT_PREFIX_INFO) {
				int ifindex;
				unsigned long expire;
				struct prefix_info *pinfo =
					(struct prefix_info *) opt;
				struct in6_addr home_addr;
				struct net_device *dev;

				ifindex = mipv6_get_home_dev_by_ha(saddr);
				
				if (!(dev = dev_get_by_index(ifindex))) {
					DEBUG((DBG_WARNING, "Cannot find device by index %d", parm->iif));
					goto nextopt;
				}
				
				expire = ntohl(pinfo->valid);
				expire = expire == 0 ? INFINITY : expire;
				
				min_expire = expire < min_expire ? expire : min_expire;
				
				if (addrconf_pfx_adv_rcv(dev, pinfo, &home_addr) > 0)
					mipv6_mn_add_info(&home_addr, 
							  pinfo->prefix_len, 
							  MN_NOT_AT_HOME,
							  ntohl(pinfo->valid),
							  saddr, 0, 0, 0);

				dev_put(dev);
			}

		nextopt:
			optlen -= len;
			opt += len;
		}

	set_timer:
		write_lock_irqsave(&pfx_list_lock, flags);

		if (min_expire != INFINITY) {
			unsigned long expire;
			struct pfx_list_entry entry;
		
			memcpy(&entry.daddr, saddr, sizeof(struct in6_addr));
			memcpy(&entry.saddr, daddr, sizeof(struct in6_addr));
			entry.retries = 0;
			entry.ifindex = parm->iif;

			/* This is against the draft, but we need to set
			 * a minimum interval for a prefix solicitation.
			 * Otherwise a prefix solicitation storm will
			 * result if valid lifetime of the prefix is
			 * smaller than MAX_PFX_ADV_DELAY
			 */
			min_expire -= MAX_PFX_ADV_DELAY;
			min_expire = min_expire < MIN_PFX_SOL_DELAY ? MIN_PFX_SOL_DELAY : min_expire;

			expire = jiffies + min_expire * HZ;

			if (mipv6_slist_modify(&pfx_list, &entry,
					       sizeof(struct pfx_list_entry),
					       expire,
					       compare_pfx_list_entry) != 0)
				DEBUG((DBG_WARNING, "Cannot add new entry to pfx_list"));
		}

		if ((tmp = mipv6_slist_get_first_key(&pfx_list)))
			mod_timer(&pfx_timer, tmp);

		write_unlock_irqrestore(&pfx_list_lock, flags);
	} else
		return -1;

	return 0;
}

/**
 * set_ha_pfx_list - manipulate pfx_list for HA when timer goes off
 * @entry: pfx_list_entry that is due
 */
static inline void set_ha_pfx_list(struct pfx_list_entry *entry)
{
	send_pfx_adv(&entry->daddr, &entry->saddr, -1, entry->ifindex,
		     entry->retries == 0 ? 1 : 0);

	if (++entry->retries <= PREFIX_ADV_RETRIES)
		mipv6_slist_push_first(&pfx_list, jiffies +
				       (PREFIX_ADV_TIMEOUT << entry->retries) * HZ);
	else
		kfree(mipv6_slist_del_first(&pfx_list));
}

/**
 * set_mn_pfx_list - manipulate pfx_list for MN when timer goes off
 * @entry: pfx_list_entry that is due
 */
static inline void set_mn_pfx_list(struct pfx_list_entry *entry)
{
	send_pfx_sol(&entry->daddr, &entry->saddr, entry->ifindex);

	if (++entry->retries <= MAX_RTR_SOLICITATIONS)
		mipv6_slist_push_first(&pfx_list, jiffies +
				       RTR_SOLICITATION_INTERVAL * HZ);
	else
		kfree(mipv6_slist_del_first(&pfx_list));
}

/**
 * pfx_timer_handler - general timer handler
 * @dummy: dummy
 *
 * calls set_ha_pfx_list and set_mn_pfx_list to do the thing when
 * a timer goes off
 */
static void pfx_timer_handler(unsigned long dummy)
{
	unsigned long flags, tmp;
	struct pfx_list_entry *entry;

	DEBUG_FUNC();

	write_lock_irqsave(&pfx_list_lock, flags);
	if (!(entry = mipv6_slist_get_first(&pfx_list)))
		goto out;

	if (mipv6_is_ha)
		set_ha_pfx_list(entry);
	else
		set_mn_pfx_list(entry);

	if ((tmp = mipv6_slist_get_first_key(&pfx_list)))
		mod_timer(&pfx_timer, tmp);

 out:
	write_unlock_irqrestore(&pfx_list_lock, flags);
}

int mipv6_initialize_pfx_icmpv6(void)
{
	INIT_LIST_HEAD(&pfx_list);

	init_timer(&pfx_timer);
	pfx_timer.function = pfx_timer_handler;

	return 0;
}

void mipv6_shutdown_pfx_icmpv6(void)
{
	unsigned long flags;
	struct prefix_info *tmp;

	if (timer_pending(&pfx_timer))
		del_timer(&pfx_timer);

	write_lock_irqsave(&pfx_list_lock, flags);
	while ((tmp = mipv6_slist_del_first(&pfx_list)))
		kfree(tmp);
	write_unlock_irqrestore(&pfx_list_lock, flags);
}
