#include <pthread.h>
#include <semaphore.h>
#include <fcntl.h>
#include <unistd.h>
#include <math.h>
#include <stdio.h>

#include "linux/paapi_syscall.h"
#include "paapi_dvs.h"

struct paapi_time_info timetracebuffer[TIME_TRACE_BUFFER_LENGTH];
unsigned time_trace_buffer_index = 0;

#define MAX_THREADS 32

/* variables used to compute the static factors and for the schedulability
   analysis */
static unsigned num_threads_created = 0;

static pthread_t *threads[MAX_THREADS];

void 
paapi_dvs_create_thread_type(struct thread_type_t *th_type_info,
			     struct thread_type_handle_t *type_handle)
{
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);
  TAKE_TIMESTAMP(t1);
  paapi_create_type(th_type_info, type_handle);
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_CREATE_TYPE, TOTAL_TIME(t1,t2));
}

void 
paapi_dvs_change_thread_type(struct thread_type_t *th_type_info)
{
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);
  TAKE_TIMESTAMP(t1);
  paapi_control_taskset(PAAPI_CTRL_CHANGE_TYPE, th_type_info);
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_CHANGE_TYPE, TOTAL_TIME(t1,t2));
}

void 
paapi_dvs_create_instance(struct thread_type_handle_t * type_handle,
			  struct thread_instance_handle_t *instance_handle,
			  void * (*code)(void *), void *data)
{
  pthread_t *thread = (pthread_t *)malloc(sizeof(pthread_t));
  struct sched_param schedp;
  pthread_attr_t attr;
  struct paapi_thread_data_t th_data;
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);

  TAKE_TIMESTAMP(t1);
  paapi_create_instance(type_handle, instance_handle, 0); 

  memcpy(&(th_data.inst_handle), instance_handle, 
	 sizeof(struct thread_instance_handle_t));

  th_data.data = data;

  /* create a light weight process passing the instance index table in as
     a parameter besides the actual process data */
  pthread_attr_init(&attr);
  pthread_attr_setschedpolicy(&attr, SCHED_FIFO);
  schedp.sched_priority = MIN_PRIORITY;
  pthread_attr_setschedparam(&attr, &schedp);
  pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
  pthread_create( thread, &attr, code, (void*)&th_data);
  threads[num_threads_created] = thread;
  num_threads_created++;
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_CREATE_INSTANCE, TOTAL_TIME(t1,t2));
}

void
paapi_dvs_init()
{
  struct sched_param schedp;

  schedp.sched_priority = 1;
  if ( sched_setscheduler(0, SCHED_FIFO, &schedp) ) {
    printf("Fail to set scheduler policy!\n");
  }
}

void
paapi_dvs_select_policy(int policy)
{
  paapi_control_taskset(PAAPI_CTRL_SET_POLICY, &policy);
}

void 
paapi_dvs_app_start(void)
{
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);

  TAKE_TIMESTAMP(t1);
  paapi_app_status(PAAPI_DVS_APP_START);
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_STATUS_START, TOTAL_TIME(t1,t2));
}

void 
paapi_dvs_app_start_period(void) 
{
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);

  TAKE_TIMESTAMP(t1);
  paapi_app_status(PAAPI_DVS_APP_START_PERIOD);
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_STATUS_START_PERIOD, TOTAL_TIME(t1,t2));
}

void
paapi_dvs_app_done(void)
{
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);

  TAKE_TIMESTAMP(t1);
  paapi_app_status(PAAPI_DVS_APP_DONE);
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_STATUS_DONE, TOTAL_TIME(t1,t2));
}

void
paapi_dvs_app_associate(struct paapi_thread_data_t *th_data)
{
  paapi_associate(th_data->inst_handle.instance_info_index);
}

void
paapi_dvs_app_leave()
{
  int retval = 0;
  DECLARE_TIMESTAMP(t1);
  DECLARE_TIMESTAMP(t2);

  TAKE_TIMESTAMP(t1);
  paapi_destroy_task();
  TAKE_TIMESTAMP(t2);
  LOG_TIME_TRACE(APP_TIME_PAAPI_LEAVE, TOTAL_TIME(t1,t2));
  pthread_exit(&retval);
}

void
paapi_dvs_app_sleep_until_next_period()
{
  paapi_sleep_until_next_period();
}

void
paapi_dvs_start_taskset(void)
{
  unsigned i = 0;
  int error, retval;
  paapi_control_taskset(PAAPI_CTRL_START_TASKSET, NULL);
  for (i = 0; i < num_threads_created; i++) {
    if ( error = pthread_join(*(threads[i]), (void **)&retval) ) {
      switch(error) {
        case ESRCH:
          printf("No thread found\n");
        break;
        case EINVAL:
          printf("Thread has been detached or another thread is waiting\n");
        break;
        case EDEADLK:
          printf("Argument refers to calling thread\n");
        break;
      }
    }
  }
  paapi_control_taskset(PAAPI_PRINT_LOG, NULL);
  PRINT_TIME_TRACE;
}

void
paapi_dvs_get_wcet(unsigned *wcet)
{
  paapi_control_taskset(PAAPI_CTRL_GET_TASK_INFO, wcet);
}

void log_timetrace(char type, int time)
{
  timetracebuffer[time_trace_buffer_index].type = type;
  timetracebuffer[time_trace_buffer_index].time = time;
  time_trace_buffer_index =
    (time_trace_buffer_index+1) % TIME_TRACE_BUFFER_LENGTH;
}
                                                                                
void print_time_trace()
{
  int i;
  printf("applevelstart\n");
  for ( i = 0; i < time_trace_buffer_index; i++ ) {
    printf("%d:%d\n", timetracebuffer[i].type, timetracebuffer[i].time);
  }
  printf("applevelend\n");
  time_trace_buffer_index = 0;
}

