/**
 * @file libDynamoMiddleware.h
 * @brief Desktop based Middleware Libraries.  
 *
 * Dynamo Middleware libraries, provide a easy to use interface for querying system 
 * information, utilizing shared memory, and communication framework
 *
 * Copyright (C) 2004
 * Dynamo Middleware Group
 *
 * version 0.5
 *
 * Authors: Michael Philpott, Shivajit Mohapatra 
 * 
 */

#ifndef _DYNMIDDLEWARE_H_
#define _DYNMIDDLEWARE_H_

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h> 
#include <string.h> 
#include <stdlib.h>
#include <sys/uio.h>
#include <memory.h>
#include <pthread.h> 
#include <unistd.h>

//
// Structures

/**
 * @struct dynamo_backlight
 * @brief Contains Backlight information.
 */
struct dynamo_backlight 
{
	int level;			///< Level from 0 - 255
	int status;			///< If the backlight is currently on or off 1|0
};

/**
 * @struct dynamo_battery
 * @brief Contains Battery information.
 */
struct dynamo_battery 
{
	int ac_status;		///< If the device is plugged
	int level;			///< The power level of the battery 1 - 100
	int min_remaining;	///< minutes until there is no power
};

/**
 * @struct dynamo_cpu
 * @brief Contains CPU Information
 */
struct dynamo_cpu 
{
	char model[40];		///< The model name of the CPU
	float load;			///< load value of the cpu 0 - 1.0 average of CPU over the past min
	int total_process;	///< total processes active on the device
	int active_process;	///< currently active processes on the server
};

/**
 * @struct dynamo_memory
 * @brief Contains Memory Information.
 */
struct dynamo_memory 
{
	int total_mem;		///< Total memory available to the device
	int free_mem;		///< Free memory on the device
	int total_swap;		///< Total swap available on the device
	int free_swap;		///< Free swap on the device
};

/**
 * @struct dynamo_network
 * @brief Contains Network Information
 */
struct dynamo_network 
{
	int connected;		///< If the device is connected to the network
	int wireless;		///< If the device is connected to a wireless network
	int link;			///< The link strength of the wireless signal
	int signal;			///< The signal strength of the wireless network
};

//
// Shared Memory Operations

/**
 * @brief creates a chunk of shared memory
 * @param key the Key to the shared memory
 * @param size the size that the shaed memory should be
 * @return The ID to the shared memory
 */
int createSharedMemory(int key, int size);

/**
 * @brief reads from the shared memory into a structuer
 * @param shmid the ID of the shared memory
 * @param *data the structure that will be read
 * @param size the size of the item that will be read
 */
int readSharedMemory(int shmid, void* data, int size);

/**
 * @brief writes data to the shared memory
 * @param shmid the ID of the shared memory to write to
 * @param *data the information to write inot the shared memory
 * @param size the size of the item that will be written
 */
int writeSharedMemory(int shmid, void* data, int size);

/**
 * @brief releases the memory with id of shmid
 * @param shmid the id of the shared memory that is being freed
 */
int freeSharedMemory(int shmid);

//
// Semaphore Operations

/**
 * @brief translate semaphore key to an ID
 * @param key the Key that the semaphore should be created with 
 * @return the ID to the shared memory
 */
int createSemaphore(int key);

/**
 * @brief free specified Semaphore
 * @param sem_id the ID of the semaphore to free
 */
int freeSemaphore(int sem_id);

/**
 * @brief acquire semaphore 
 * @param sid the ID of the semaphore to obtain
 */
void P(int sid);

/**
 * @brief release semaphore
 * @param sid the ID of the semaphore to release
 */
void V(int sid);

/**
 * @brief Sends a Datagram to a remote computer
 * @param *targetIP character array of target computers IP Address
 * @param localPort local port of the UPD socket
 * @param targetPort the remote hosts port
 * @param *buf the information to be sent
 * @param bufsiz the size of the item to be sent
 */
int SendDatagram(char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

/**
 * @brief creates a UPD socket
 * @param *localIP character array of the local IP address
 * @param localPort a port number
 */
int CreateUDPSocket(char *localIP, int localPort);

/**
 * @brief Recives a datagram a remote computer
 * @param sockfd pointer to the UDP socket
 * @param *buf address of the structure to fill
 * @param size size of the structure to fill
 */
int UDP_Recv(int sockfd, void *buf, int size);

/**
 * @brief Sends a Datagram to a remote computer
 * @param sockfd pointer to the upd socket
 * @param *targetIP character array of target computers IP Address
 * @param targetPort the remote hosts port
 * @param localPort local port of the UPD socket
 * @param *buf the information to be sent
 * @param bufsiz the size of the item to be sent
 */
int UDP_Send(int sockfd, char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

#endif



