#include <linux/power/paapi_sched.h>

/* table of task types */
struct thread_type_info_t *type_table[MAX_TASKS_TYPES];
unsigned num_types = 0;

/* table of task instances */
struct thread_instance_info_t *instance_table[MAX_TASKS_INSTANCES];
unsigned num_instances = 0;

unsigned dvs_policy = 0;
unsigned admitted = 0;

struct trace_info tracebuffer[TRACE_BUFFER_LENGTH];
unsigned trace_buffer_index = 0;

void log_trace(char type, unsigned task, unsigned time, 
  unsigned param0, unsigned param1)
{
  tracebuffer[trace_buffer_index].type = type;
  tracebuffer[trace_buffer_index].task = task;
  tracebuffer[trace_buffer_index].time = time;
  tracebuffer[trace_buffer_index].exec_time = 0;
  tracebuffer[trace_buffer_index].total_time = 0;
  tracebuffer[trace_buffer_index].deadline = 0;
  tracebuffer[trace_buffer_index].next_arrival = 0;
  switch(type) {
    case SCHED_PREEMPT:
      tracebuffer[trace_buffer_index].exec_time = param0;
      tracebuffer[trace_buffer_index].total_time = param1;
      break;
    case SCHED_PREEMPT_DONE:
      tracebuffer[trace_buffer_index].deadline = param0;
      tracebuffer[trace_buffer_index].next_arrival = param1;
      break;
    case SCHED_PREEMPT_DEAD:
      tracebuffer[trace_buffer_index].deadline = param0;
      break;
    case CHANGE_FREQ_START:
      tracebuffer[trace_buffer_index].speed = param0;
      break;
    case CHANGE_FREQ_DONE:
      tracebuffer[trace_buffer_index].speed = param0;
      break;
    case STATIC_FACTOR:
      tracebuffer[trace_buffer_index].speed = param0;
      tracebuffer[trace_buffer_index].factor = param1;
      break;
    case DYNAMIC_FACTOR:
      tracebuffer[trace_buffer_index].speed = param0;
      tracebuffer[trace_buffer_index].factor = param1;
      break;
    case CHANGE_PARAMETERS:
      tracebuffer[trace_buffer_index].speed = param0;
      break;
  }
  trace_buffer_index = (trace_buffer_index+1) % TRACE_BUFFER_LENGTH;
}

void print_trace() 
{
  int i; 
  printk("kerneltracestart\n");
  for ( i = 0; i < trace_buffer_index; i++ ) {
    printk("%d: task_%d ", tracebuffer[i].time, tracebuffer[i].task);
    switch(tracebuffer[i].type) {
    case APP_SLEEP:
      printk("APP_SLEEP\n");
      break;
    case SCHED_START:
      printk("SCHED_START\n");
      break;
    case SCHED_PREEMPT:
      printk("SCHED_PREEMPT exec_time: %d, total_time: %d\n", 
             tracebuffer[i].exec_time, tracebuffer[i].total_time);
      break;
    case SCHED_PREEMPT_DONE:
      printk("SCHED_PREEMPT DONE deadline: %d, next_arrival: %d\n", 
             tracebuffer[i].deadline, tracebuffer[i].next_arrival);
      break;
    case SCHED_PREEMPT_DEAD:
      printk("SCHED PREEMPT missed deadline: %d\n", tracebuffer[i].deadline);
      break;
    case LEAVING:
      printk("LEAVING\n");
      break;
    case WAKE_UP:
      printk("WAKE_UP\n");
      break;
    case WAKE_UP_READJUST:
      printk("WAKE_UP_READJUST\n");
      break;
    case WAKE_UP_DEAD:
      printk("WAKE_UP deadline was missed %d, waking up\n", 
        tracebuffer[i].deadline);
      break;
    case CHANGE_FREQ_START:
      printk("CHANGE_FREQ_START: %d\n", tracebuffer[i].speed); 
      break;
    case CHANGE_FREQ_DONE:
      printk("CHANGE_FREQ_DONE: %d\n", tracebuffer[i].speed); 
      break;
    case STATIC_FACTOR:
      printk("STATIC_FACTOR factor: %d (%d)\n", tracebuffer[i].speed,
                                                tracebuffer[i].factor);
      break;
    case DYNAMIC_FACTOR:
      printk("DYNAMIC_FACTOR factor: %d (%d)\n", tracebuffer[i].speed,
                                                 tracebuffer[i].factor);
      break;
    case CHANGE_PARAMETERS:
      printk("CHANGE_PARAMETERS: %d\n", tracebuffer[i].speed);
      break;
    }
  }
  printk("kerneltraceend\n");
  trace_buffer_index = 0;
}

// Function to predict execution time of a task
// There are basically two cases to be taken care of:
//   . the task is going to start a new execution
//   . the task is being resumed from a previous preemption
// At the end the execution time is derived and the dynamic scaling
// can be calculated
//
// Input: information about the task instance
// Output: OS predicted time, OS predicted time timestamp and 
//         dynamic scaling down factor fields are filled with info

void 
predict_execution_time(struct thread_instance_info_t *th_info)
{
  int start_slot_number, i;
  unsigned num_executions = 0, weighted_sum = 0;
  unsigned time_prediction;
  unsigned a, b;

  // the task has been preempted and one has to predict the remaining
  // execution time
  if ( th_info->preempted && !th_info->done) {
    a =  ( th_info->total_time_so_far * th_info->type_info->info.wcet );
    b = ( ( 100 * NUMBER_OF_WCET_SLOTS )  / th_info->sscale_factor );
    start_slot_number = ( DIV_CEIL(a, b) ) - 1;

    if ( start_slot_number >= NUMBER_OF_WCET_SLOTS )
      start_slot_number = NUMBER_OF_WCET_SLOTS - 1;
    else if ( start_slot_number < 0 )
      start_slot_number = 0;
  }
  // that's a new execution of this task and hence the whole task
  // execution time has to be predicted
  else
    start_slot_number = 0;

  // printk("slot number: %d\n", start_slot_number);
  // computed the expected value of the times probability distribution
  for ( i = start_slot_number; i < NUMBER_OF_WCET_SLOTS; i++ ) {
    weighted_sum += 
      th_info->type_info->thread_type_rt->slotted_wcet[i] * (i + 1); 
    num_executions += 
      th_info->type_info->thread_type_rt->slotted_wcet[i];
  }

  //printk("weighted sum: %d\n", weighted_sum);
  //printk("num_execution: %d\n", num_executions);
  //printk("wcet: %d\n", th_info->type_info->wcet);
  if ( num_executions == 0 ) {
    time_prediction = 
      ( th_info->type_info->info.wcet * 100 ) / th_info->sscale_factor;
  }
  else {
    time_prediction = (weighted_sum / num_executions) *
      (((th_info->type_info->info.wcet * 100) / th_info->sscale_factor) / 
       NUMBER_OF_WCET_SLOTS);
  }
  printk("%d: task_%d time prediction: %d\n", PAAPI_CURRENT_TIME, 
     th_info->instance_number, time_prediction);
  th_info->os_time_remaining = time_prediction;
  th_info->timestamp_os_tr = PAAPI_CURRENT_TIME;
}

// Function to compute the dynamic slow down factor of task
// Based on this the next time the task is executed the processor
// operating frequency is reduced according to this slow down factor
void 
compute_dynamic_factor(struct thread_instance_info_t *th_info)
{
  unsigned dscale_factor;
  unsigned scaled_wcet;
                              
  //printk("predicted time remaining: %d\n", th_info->os_time_remaining);
  //printk("wcet: %d\n", 100 * th_info->type_info->info.wcet);
  //printk("sfactor: %d\n", th_info->sscale_factor);
  //printk("time so far: %d\n", th_info->total_time_so_far);

  scaled_wcet = ( 100 * th_info->type_info->info.wcet ) / 
    th_info->sscale_factor;

  //printk("statically scaled wcet: %d\n", scaled_wcet);

  if ( th_info->total_time_so_far < th_info->os_time_remaining &&
       th_info->total_time_so_far < scaled_wcet ) {
    unsigned predicted_remain, wcet_remain;

    predicted_remain = ( th_info->os_time_remaining - 
			 th_info->total_time_so_far );
    wcet_remain = scaled_wcet - th_info->total_time_so_far;
    //printk("pred remain: %d, wcet_remain: %d\n", predicted_remain, 
    //	   wcet_remain);
    dscale_factor =  ( predicted_remain * 100 ) / wcet_remain;
  }
  else {
    dscale_factor = 100;
  }
  printk("%d: dynamic factor: %d(/100)\n", PAAPI_CURRENT_TIME, dscale_factor);
  th_info->dscale_factor = dscale_factor;
}

#define MAX_THRESHOLD 2 // maximum number of deadlines missed tolerated
#define MIN_THRESHOLD 0 // minimum number of deadlines missed tolerated
#define DEADLINE_SEARCH_WINDOW (3)
#define MIN_ADAPTIVE_FACTOR 50
#define DECREASE_AMOUNT 5
#define INCREASE_AMOUNT 10

// Based on the number of deadlines missed in the past DEADLINE_SEARCH_WINDOW
// execution return an adaptive factor that should be applied to the time 
// prediction in order to make it less/more agressive 
void 
compute_adaptive_factor(struct thread_instance_info_t *th_info)
{
  // DEADLINE_SEARCH_WINDOW executions
  unsigned adaptive_factor = th_info->adaptive_factor;

  if ( th_info->type_info->thread_type_rt->num_last_prev_executions == 
       DEADLINE_SEARCH_WINDOW ) {
    if ( th_info->type_info->thread_type_rt->num_last_missed_deadlines >= 
	 MAX_THRESHOLD ) {
      adaptive_factor += INCREASE_AMOUNT;
    }
    if ( th_info->type_info->thread_type_rt->num_last_missed_deadlines <= 
	 MIN_THRESHOLD) {
      if ( adaptive_factor > MIN_ADAPTIVE_FACTOR) {
	adaptive_factor -= DECREASE_AMOUNT;
      }
    }
    th_info->type_info->thread_type_rt->num_last_prev_executions = 0;
    th_info->type_info->thread_type_rt->num_last_missed_deadlines = 0;
  }
  th_info->adaptive_factor = adaptive_factor;
  printk("%d: adaptive factor: %d\n", PAAPI_CURRENT_TIME, adaptive_factor);
}


/*--------------------------------------------------------------------*/
/* Debugging functions */

/* print the fields of the type handle */
void print_type_handle(
  struct thread_type_handle_t type_handle
  )
{ 
#ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
  printk("Index in the type table: %d\n", type_handle.type_info_index);
#endif
}

/* print the type table for debug purposes */
void print_type_table(void
  )
{
  int i;

  for ( i = 0; i < MAX_TASKS_TYPES; i++ ) {
    if ( type_table[i] != (void *)0 ) {
    #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      printk("Table index: %d\n", i);
      printk("  wcet: %d\n", type_table[i]->wcet);
      printk("  period: %d\n", type_table[i]->period);
      printk("  deadline: %d\n", type_table[i]->deadline);
      printk("  hardness: %d\n", type_table[i]->hard);
    #endif
    }
  }
}

/* print the fields of the instance handle */
void print_instance_handle(
  struct thread_instance_handle_t instance_handle
  )
{
#ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
  printk("Index in the type table: %d\n", 
    instance_handle.type_info_index);
  printk("Index in the instance table: %d\n", 
         instance_handle.instance_info_index);
#endif
}

/* print the fields of the instance table */
void print_instance_table(void
  )
{
  int i;

  for ( i = 0; i < MAX_TASKS_INSTANCES; i++ ) {
    if ( instance_table[i] != (void *)0 ) {
    #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      printk("Table index: %d\n", i);
      printk("  arrival time: %d\n", instance_table[i]->arrival_time);
      printk("  total time so far: %d\n", 
        instance_table[i]->total_time_so_far); 
      printk("  time remaining: %d\n", 
        instance_table[i]->time_remaining); 
      printk("  timestamp time remaining: %d\n", 
              instance_table[i]->timestamp_tr);
      printk("  timestamp os time remining: %d\n", 
              instance_table[i]->os_time_remaining);
      printk("  done flag: %d\n", instance_table[i]->done);
      printk("  preempted flag: %d\n", instance_table[i]->preempted);
      printk("  rescheduling time: %d\n", 
        instance_table[i]->last_resched);
    #endif
    }
  }
}
