/*
 *      Mobile-node functionality
 *
 *      Authors:
 *      Sami Kivisaari          <skivisaa@cc.hut.fi>
 *
 *      $Id: mn.c,v 1.113 2002/08/15 20:18:00 ville Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/autoconf.h>
#include <linux/sched.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/init.h>
#include <linux/skbuff.h>
#include <linux/rtnetlink.h>
#include <linux/if_arp.h>
#include <linux/ipsec.h>
#include <linux/notifier.h>
#include <linux/list.h>
#include <linux/route.h>

#include <asm/uaccess.h>

#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/ndisc.h>
#include <net/ipv6_tunnel.h>
#include <net/mipv6.h>
#include <net/ip6_route.h>

#include "util.h"
#include "mdetect.h"
#include "bul.h"
#include "sendopts.h"
#include "debug.h"
#include "mn.h"
#include "dhaad.h"
#include "multiaccess_ctl.h"
#include "prefix.h"
#include "tunnel.h"
#include "stats.h"

static LIST_HEAD(mn_info_list);

struct ifr_holder {
	struct in6_ifreq ifr;
	int old_ifi;
};

/* Whether all parts are initialized, from mipv6.c */
extern int mipv6_is_initialized;

/* Determines whether manually configured home addresses are preferred as 
 * source addresses over dynamically configured ones
 */
int mipv6_use_preconfigured_hoaddr = 1; 

/* Determines whether home addresses, which are at home are preferred as 
 * source addresses over other home addresses
 */
int mipv6_use_topol_corr_hoaddr = 0;

/* Lock for list of MN infos */
static rwlock_t mn_info_lock = RW_LOCK_UNLOCKED;

/*  Defined in ndisc.c of IPv6 module */
extern void ndisc_send_na(
	struct net_device *dev, struct neighbour *neigh,
	struct in6_addr *daddr, struct in6_addr *solicited_addr,
	int router, int solicited, int override, int inc_opt);
int init_home_registration(struct mn_info *, struct in6_addr *);

/**
 * mipv6_mninfo_readlock - acquire mn_info_lock (read)
 * 
 * Acquires write lock @mn_info_lock.  Lock must be held when reading
 * from a mn_info entry.
 **/
void mipv6_mninfo_readlock(void)
{
	read_lock_bh(&mn_info_lock);
}

/**
 * mipv6_mninfo_readunlock - release mn_info_lock (read)
 * 
 * Releases write lock @mn_info_lock.
 **/
void mipv6_mninfo_readunlock(void)
{
	read_unlock_bh(&mn_info_lock);
}

/**
 * mipv6_mninfo_writelock - acquire mn_info_lock (write)
 * 
 * Acquires write lock @mn_info_lock.  Lock must be held when writing
 * to a mn_info entry.
 **/
void mipv6_mninfo_writelock(void)
{
	write_lock_bh(&mn_info_lock);
}

/**
 * mipv6_mninfo_writeunlock - release mn_info_lock (write)
 * 
 * Releases write lock mn_info_lock.
 **/
void mipv6_mninfo_writeunlock(void)
{
	write_unlock_bh(&mn_info_lock);
}

/**
 * mipv6_mn_is_home_addr - Determines if addr is node's home address
 * @addr: IPv6 address
 *
 * Returns 1 if addr is node's home address.  Otherwise returns zero.
 **/
int mipv6_mn_is_home_addr(struct in6_addr *addr)
{
	int ret = 0;

	if (addr == NULL) {
		DEBUG((DBG_CRITICAL, "mipv6_mn_is_home_addr: Null argument"));
		return -1;
	}
	read_lock_bh(&mn_info_lock);
	if (mipv6_mn_get_info(addr))
		ret = 1;
	read_unlock_bh(&mn_info_lock);

	return (ret);
}

/** 
 * mipv6_mn_is_at_home - determine if node is home for a home address
 * @home_addr : home address of MN
 *
 * Returns 1 if home address in question is in the home network, 0
 * otherwise.  Caller MUST NOT not hold @mn_info_lock.
 **/ 
int mipv6_mn_is_at_home(struct in6_addr *home_addr)
{
	struct mn_info *minfo;
	int ret = 0;
	read_lock_bh(&mn_info_lock);
	if ((minfo = mipv6_mn_get_info(home_addr)) != NULL) {	
		ret = (minfo->is_at_home == MN_AT_HOME);
	}
	read_unlock_bh(&mn_info_lock);
	return ret;
}	

/**
 * mipv6_mn_get_homeaddr - Get node's home address
 * @home_addr: buffer to store home address
 *
 * Stores Mobile Node's home address in the given space.  Returns
 * prefix length of home address.  Negative return value means failure
 * to retrieve home address.  Caller MUST NOT hold @mn_info_lock.
 **/
int mipv6_mn_get_homeaddr(struct in6_addr *home_addr)
{
	struct mn_info *minfo;
	struct list_head *p;
	int plen = 0;

	if (!home_addr) return -1;

	read_lock_bh(&mn_info_lock);

	if (list_empty(&mn_info_list)) {
		read_unlock_bh(&mn_info_lock);
		return -1;
	}

	p = mn_info_list.next;

	if (p != NULL) {
		minfo = list_entry(p, struct mn_info, list);
		ipv6_addr_copy(home_addr, &minfo->home_addr);
		plen = minfo->home_plen;
	}
	read_unlock_bh(&mn_info_lock);

	return plen;
}

void mipv6_get_saddr_hook(struct inet6_ifaddr *ifp,
			  struct in6_addr *homeaddr)
{
	int found = 0, reiter = 0;
	struct list_head *lh;
	struct mn_info *minfo = NULL;
	struct in6_addr coa;

	read_lock_bh(&mn_info_lock);
restart:
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if ((ipv6_addr_scope(homeaddr) != ipv6_addr_scope(&minfo->home_addr)) 
		    || ipv6_chk_addr(&minfo->home_addr, NULL) == 0)
			continue; 
		if (!((minfo->is_at_home == MN_AT_HOME) ||
		      minfo->has_home_reg))
			continue;
		if (mipv6_use_topol_corr_hoaddr && 
		    minfo->is_at_home == MN_AT_HOME) {
			ipv6_addr_copy(homeaddr, &minfo->home_addr);
			found = 1;
			break;
		} 
		if (mipv6_use_preconfigured_hoaddr && minfo->man_conf) {
			ipv6_addr_copy(homeaddr, &minfo->home_addr);
			found = 1;
			break;
		} 
		if (!(mipv6_use_preconfigured_hoaddr || 
		      mipv6_use_topol_corr_hoaddr) || reiter) {
			ipv6_addr_copy(homeaddr, &minfo->home_addr);
			found = 1;
			break;
		}
	}
	if (!found && !reiter) {
		reiter = 1;
		goto restart;
	}

	if (!found && minfo && mipv6_get_care_of_address(&minfo->home_addr, &coa) == 0)
		ipv6_addr_copy(homeaddr, &coa); 
	read_unlock_bh(&mn_info_lock);

	DEBUG((DBG_INFO, "Source address selection:  %x:%x:%x:%x:%x:%x:%x:%x", 
	       NIPV6ADDR(homeaddr)));
	return;
}

/**
 * mipv6_mninfo_get_by_id - Lookup mn_info with id
 * @id: DHAAD identifier
 *
 * Searches for a mn_info entry with @dhaad_id set to @id.  You MUST
 * hold @mn_info_lock (write) when calling this function.  Returns
 * pointer to mn_info entry or %NULL on failure.
 **/
struct mn_info *mipv6_mninfo_get_by_id(unsigned short id)
{
	struct list_head *lh;
	struct mn_info *minfo;

	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (minfo->dhaad_id == id) {
			return minfo;
		}
	}
	return NULL;
}

/** 
 * mipv6_mninfo_get_by_index - gets mn_info at index  
 * @info: pointer to memory where mn_info will be copied 
 * @index: index of entry 
 *
 * Returns 0 on success and -1 on failure.  Caller MUST NOT hold
 * @mn_info_lock.
 **/
int mipv6_mninfo_get_by_index(int index, struct mn_info *info)
{
	struct list_head *lh;
	struct mn_info *minfo; 
	int i = 0;

	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		if (index == i++) {
			minfo = list_entry(lh, struct mn_info, list);
			memcpy(info, minfo, sizeof(struct mn_info));
			read_unlock_bh(&mn_info_lock);
			return 0;
		}
	}
	read_unlock_bh(&mn_info_lock);
	return -1;
}

/** 
 * mipv6_mn_set_hashomereg - Set has_home_reg field in a mn_info
 * @haddr: home address whose home registration state is changed
 * @has_reg: value to set
 *
 * Changes state of home info and also adjusts home address state
 * accordingly.  Returns 0 on success, a negative value otherwise.
 * Caller MUST NOT hold @mn_info_lock.
 **/
int mipv6_mn_set_hashomereg(struct in6_addr *haddr, int has_reg)
{
	struct mn_info *minfo;

	write_lock_bh(&mn_info_lock);
	if ((minfo = mipv6_mn_get_info(haddr)) != NULL) {
		minfo->has_home_reg = has_reg;
		write_unlock_bh(&mn_info_lock);
		return 0;
	}
	DEBUG((DBG_ERROR, "set_has_homereg: No mn_info for home addr"));
	write_unlock_bh(&mn_info_lock);
	return -1;
}

/**
 * mipv6_mn_hashomereg - Check if home address is registered
 * @haddr: Home Address to check
 *
 * Checks if @haddr has been successfully registered.  If address is
 * not found, returns negative.  If found, returns 1 for registered or
 * 0 for non-registered address.  Caller MUST NOT hold @mn_info_lock.
 **/
int mipv6_mn_hashomereg(struct in6_addr *haddr)
{
	struct mn_info *minfo;
	int has_reg;
	read_lock_bh(&mn_info_lock);
	if ((minfo = mipv6_mn_get_info(haddr)) != NULL) {
		has_reg = minfo->has_home_reg;
		read_unlock_bh(&mn_info_lock);
		return has_reg;
	}
	DEBUG((DBG_ERROR, "set_has_homereg: No mn_info for home addr"));
	read_unlock_bh(&mn_info_lock);
	return -1;
}

/** 
 * mipv6_mn_get_info - Returns mn_info for a home agent address
 * @haddr: home agent address of MN
 *
 * Returns mn_info on success %NULL otherwise.  Caller MUST hold
 * @mn_info_lock (read or write).
 **/
struct mn_info *mipv6_mn_get_info_by_ha(struct in6_addr *ha_addr)
{
	struct list_head *lh;
	struct mn_info *minfo;

	DEBUG_FUNC();

	if (!ha_addr)
		return NULL;

	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!ipv6_addr_cmp(&minfo->ha, ha_addr)) {
			return minfo;
		}
	}
	return NULL;
}

/**
 * mipv6_get_home_dev_by_ha - Returns home device
 * @haddr: home agent for the home device
 *
 * Returns the if_index of the device, where the first home address for ha 
 * is currently stored. 
 **/
int mipv6_get_home_dev_by_ha(struct in6_addr *home_agent)
{
	struct mn_info *minfo;
	int if_index = 0;

	read_lock_bh(&mn_info_lock);
	minfo = mipv6_mn_get_info_by_ha(home_agent);
	if (minfo)
		if_index = minfo->ifindex;
	read_unlock_bh(&mn_info_lock);

	return if_index;
}
/** 
 * mipv6_mn_get_info - Returns mn_info for a home address
 * @haddr: home address of MN
 *
 * Returns mn_info on success %NULL otherwise.  Caller MUST hold
 * @mn_info_lock (read or write).
 **/
struct mn_info *mipv6_mn_get_info(struct in6_addr *haddr)
{
	struct list_head *lh;
	struct mn_info *minfo;

	DEBUG_FUNC();

	if (!haddr)
		return NULL;

	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!ipv6_addr_cmp(&minfo->home_addr, haddr)) {
			return minfo;
		}
	}
	return NULL;
}

/** 
 * mipv6_mn_add_info - Adds a new home info for MN
 * @home_addr:  Home address of MN, must be set
 * @plen: prefix length of the home address, must be set
 * @isathome : home address at home
 * @lifetime: lifetime of the home address, 0 is infinite
 * @ha: home agent for the home address
 * @ha_plen: prefix length of home agent's address, can be zero 
 * @ha_lifetime: Lifetime of the home address, 0 is infinite
 *
 * The function adds a new home info entry for MN, allowing it to
 * register the home address with the home agent.  Starts home
 * registration process.  If @ha is %ADDRANY, DHAAD is performed to
 * find a home agent.  Returns 0 on success, a negative value
 * otherwise.  Caller MUST NOT hold @mn_info_lock or
 * @addrconf_hash_lock.
 **/
void mipv6_mn_add_info(struct in6_addr *home_addr, int plen, int isathome,
		       unsigned long lifetime, struct in6_addr *ha, 
		       int ha_plen, unsigned long ha_lifetime, int man_conf)
{
	struct mn_info *minfo = NULL, minfo_init;
	struct inet6_ifaddr *ifp;
	struct in6_addr coa;

	DEBUG_FUNC();

	write_lock_bh(&mn_info_lock);
	if ((minfo = mipv6_mn_get_info(home_addr)) == NULL) {
	        minfo = kmalloc(sizeof(struct mn_info), GFP_ATOMIC);
		if (!minfo) {
			write_unlock_bh(&mn_info_lock);
			return;
		}
		memset(minfo, 0, sizeof(struct mn_info));
		list_add(&minfo->list, &mn_info_list);
	}
	
	ipv6_addr_copy(&minfo->home_addr, home_addr);

	if (ha)
		ipv6_addr_copy(&minfo->ha, ha);
	minfo->home_plen = plen;       
	ifp = ipv6_get_ifaddr(home_addr, NULL);

	if (ifp != NULL) {
		if(!lifetime)
			lifetime = ifp->valid_lft;
		minfo->ifindex = ifp->idev->dev->ifindex;
		in6_ifa_put(ifp);
	} else {
		lifetime = 0;
		minfo->ifindex = 0;
	}
	minfo->home_addr_expires = jiffies + lifetime * HZ;
	/* manual configuration flag cannot be unset by dynamic updates 
	 *  from prefix advertisements
	 */
	if (!minfo->man_conf) minfo->man_conf = man_conf; 
	minfo->is_at_home = isathome;

	memcpy(&minfo_init, minfo, sizeof(minfo_init));
	write_unlock_bh(&mn_info_lock);

	mipv6_get_care_of_address(&minfo_init.home_addr, &coa); 
	init_home_registration(&minfo_init, &coa);
}

/**
 * mipv6_mn_del_info - Delete home info for MN 
 * @home_addr : Home address or prefix 
 * @del_dyn_only : Delete only dynamically created home entries 
 *
 *
 * Deletes every mn_info entry that matches the first plen bits of
 * @home_addr.  Returns number of deleted entries on success and a
 * negative value otherwise.  Caller MUST NOT hold @mn_info_lock.
 **/
int mipv6_mn_del_info(struct in6_addr *home_addr, int del_dyn_only)
{
	struct list_head *lh, *next;
	struct mn_info *minfo;
	int ret = -1;
	if (!home_addr)
		return -1;

	write_lock_bh(&mn_info_lock);

	list_for_each_safe(lh, next, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!mipv6_prefix_compare(&minfo->home_addr, home_addr, 
					  minfo->home_plen) 
		    && ((!minfo->man_conf && del_dyn_only) || !del_dyn_only)) {
			list_del(&minfo->list);
			kfree(minfo);
			ret++;
		}
	}
	write_unlock_bh(&mn_info_lock);
	return ret;
}

static void mn_move_homeaddr_task(void *arg)
{
	mm_segment_t oldfs;
	int err = 0, new_if = 0, new_plen = 0;
	struct ifr_holder *ifrh = (struct ifr_holder *) arg;

	DEBUG((DBG_INFO, "mipv6 move home address task"));


	/* need to inform that we are doing this inside kernel */
	oldfs = get_fs(); set_fs(KERNEL_DS);
	if (ifrh->old_ifi) {
		new_if = ifrh->ifr.ifr6_ifindex;
		ifrh->ifr.ifr6_ifindex = ifrh->old_ifi;
		new_plen = ifrh->ifr.ifr6_prefixlen;
		ifrh->ifr.ifr6_prefixlen = 128;
		err = addrconf_del_ifaddr(&ifrh->ifr); 
		ifrh->ifr.ifr6_prefixlen = new_plen;
		ifrh->ifr.ifr6_ifindex = new_if;
	}
	if(!err) err = addrconf_add_ifaddr(&ifrh->ifr);
	set_fs(oldfs);

	if (err < 0)
		DEBUG((DBG_WARNING, "adding of home address to a new interface failed %d", err));
	else
		DEBUG((DBG_WARNING, "adding of home address to a new interface OK"));
}

/**
 * mipv6_mn_get_bulifetime - Get lifetime for a binding update
 * @home_addr: home address for BU 
 * @coa: care-of address for BU
 * @flags: flags used for BU 
 *
 * Returns maximum lifetime for BUs determined by the lifetime of
 * care-of address and the lifetime of home address.
 **/
__u32 mipv6_mn_get_bulifetime(struct in6_addr *home_addr, struct in6_addr *coa,
			      __u8 flags)
{
	
	__u32 lifetime; 
	
	struct inet6_ifaddr * ifp_coa, *ifp_hoa;

	ifp_hoa = ipv6_get_ifaddr(home_addr, NULL);
	if(!ifp_hoa) {
		DEBUG((DBG_ERROR, "home address missing"));
		return 0;
	}
	ifp_coa = ipv6_get_ifaddr(coa, NULL);
	if (!ifp_coa) {
		in6_ifa_put(ifp_hoa);
		DEBUG((DBG_ERROR, "care-of address missing"));
		return 0;
	}
	if (flags & MIPV6_BU_F_HOME)
		lifetime = HA_BU_DEF_LIFETIME;
	else
		lifetime = CN_BU_DEF_LIFETIME;

	if (!(ifp_hoa->flags & IFA_F_PERMANENT)){
		if (ifp_hoa->valid_lft)
			lifetime = min_t(__u32, lifetime, ifp_hoa->valid_lft);
		else
			DEBUG((DBG_ERROR, "Zero lifetime for home address"));
	}
	if (!(ifp_coa->flags & IFA_F_PERMANENT)) {
		if(ifp_coa->valid_lft)
			lifetime = min_t(__u32, lifetime, ifp_coa->valid_lft);
		else
			DEBUG((DBG_ERROR, 
			       "Zero lifetime for care-of address"));
	}
	in6_ifa_put(ifp_hoa);
	in6_ifa_put(ifp_coa);
	DEBUG((DBG_INFO, "Lifetime for binding is %ld", lifetime));
	return lifetime;
}

static int 
mipv6_mn_tnl_rcv_send_bu_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	struct ipv6hdr *inner = (struct ipv6hdr *)skb->h.raw;
	struct ipv6hdr *outer = skb->nh.ipv6h; 
	struct mn_info *minfo = NULL;
	__u32 lifetime;

	DEBUG_FUNC();

	if (!is_mipv6_tnl(t))
		return IPV6_TNL_ACCEPT;

	read_lock(&mn_info_lock);
	minfo = mipv6_mn_get_info(&inner->daddr);

	if (!minfo) {
		DEBUG((DBG_WARNING, "MN info missing"));
		read_unlock(&mn_info_lock);
		return IPV6_TNL_ACCEPT;
	}
	DEBUG((DBG_INFO, "MIPV6 MN: Received a tunneled IPv6 packet"
	       " to %x:%x:%x:%x:%x:%x:%x:%x,"
	       " from %x:%x:%x:%x:%x:%x:%x:%x with\n tunnel header"
	       "daddr: %x:%x:%x:%x:%x:%x:%x:%x,"
	       "saddr: %x:%x:%x:%x:%x:%x:%x:%x", 
	       NIPV6ADDR(&inner->daddr), NIPV6ADDR(&inner->saddr),
	       NIPV6ADDR(&outer->daddr), NIPV6ADDR(&outer->saddr)
		));
	/* We don't send bus in response to all tunneled packets */

        if (!ipv6_addr_cmp(&minfo->ha, &inner->saddr)) {
                DEBUG((DBG_ERROR, "HA BUG: Received a tunneled packet "
		       "originally sent by home agent, not sending BU"));
		read_unlock(&mn_info_lock);
		return IPV6_TNL_ACCEPT;
        }
	if (ipv6_addr_cmp(&minfo->ha, &outer->saddr)) {
		DEBUG((DBG_WARNING, "MIPV6 MN: Received a tunneled IPv6 packet"
		       " that was not tunneled by HA %x:%x:%x:%x:%x:%x:%x:%x,"
		       " but by %x:%x:%x:%x:%x:%x:%x:%x", 
		       NIPV6ADDR(&minfo->ha), NIPV6ADDR(&outer->saddr)));
		read_unlock(&mn_info_lock);
		return IPV6_TNL_ACCEPT;
        }
	read_unlock(&mn_info_lock);

	DEBUG((DBG_INFO, "Sending BU to correspondent node"));

	if (inner->nexthdr != IPPROTO_DSTOPTS) {
		DEBUG((DBG_INFO, "Received tunneled packet without dst_opts"));
		lifetime = mipv6_mn_get_bulifetime(&inner->daddr,
						   &outer->daddr, 0); 
		if(lifetime)
		/* max wait 1000 ms  before sending an empty packet with BU */
			mipv6_send_upd_option(&inner->daddr,&inner->saddr,
					      CN_BU_DELAY, 
					      INITIAL_BINDACK_TIMEOUT,
					      MAX_BINDACK_TIMEOUT , 1, 
#ifdef CN_REQ_ACK
					      MIPV6_BU_F_ACK, /* ack */
#else
					      0, /* no ack */
#endif
					      0, lifetime, NULL);
	}
	/* (Mis)use ipsec tunnel flag  */
	DEBUG((DBG_INFO, "setting rcv_tunnel flag in skb"));
	skb->security = skb->security | RCV_TUNNEL;
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_mn_tnl_rcv_send_bu_ops = {
	{NULL, NULL}, 
	IPV6_TNL_PRE_DECAP,
	IPV6_TNL_PRI_FIRST,
	mipv6_mn_tnl_rcv_send_bu_hook
};

static int
mipv6_mn_tnl_xmit_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_encapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_mn_tnl_xmit_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_ENCAP,
	IPV6_TNL_PRI_LAST,
	mipv6_mn_tnl_xmit_stats_hook
};

static int
mipv6_mn_tnl_rcv_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();	
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_decapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_mn_tnl_rcv_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_DECAP,
	IPV6_TNL_PRI_LAST,
	mipv6_mn_tnl_rcv_stats_hook
};

void mipv6_check_tunneled_packet(struct sk_buff *skb)
{
	DEBUG_FUNC();
	/* If tunnel flag was set */
	if (skb->security & RCV_TUNNEL) {
		struct in6_addr coa; 
		__u32 lifetime;
		mipv6_get_care_of_address(&skb->nh.ipv6h->daddr, &coa);
		lifetime = mipv6_mn_get_bulifetime(&skb->nh.ipv6h->daddr,
 							 &coa, 0); 

		DEBUG((DBG_WARNING, "packet to address %x:%x:%x:%x:%x:%x:%x:%x"
		       "was tunneled. NOT Sending BU to CN" 
		       "%x:%x:%x:%x:%x:%x:%x:%x", 
		       NIPV6ADDR(&skb->nh.ipv6h->daddr),
		       NIPV6ADDR(&skb->nh.ipv6h->saddr))); 
		/* This should work also with home address option */
		mipv6_send_upd_option(
			&skb->nh.ipv6h->daddr,
			&skb->nh.ipv6h->saddr,
			CN_BU_DELAY, INITIAL_BINDACK_TIMEOUT,
			MAX_BINDACK_TIMEOUT, 1, 
#ifdef CN_REQ_ACK
			MIPV6_BU_F_ACK, /* ack */
#else
			0, /* no ack */
#endif
			64, lifetime, NULL);
	}
}
int create_homeaddr_task(struct mn_info *minfo, int plen_new, int newif, int oldif)
{
	struct ifr_holder *ifrh = NULL;
	struct tq_struct *t;

	ifrh = kmalloc(sizeof(*ifrh), GFP_ATOMIC);
	t = kmalloc(sizeof(struct tq_struct), GFP_ATOMIC);
	if (ifrh == NULL || t == NULL) {
		if (ifrh)
			kfree(ifrh);
		else if (t)
			kfree(t);
		DEBUG((DBG_ERROR, "Out of memory"));
		return -1;
	}
	/* must queue task to avoid deadlock with rtnl */
	ifrh->ifr.ifr6_ifindex = newif;
	ifrh->ifr.ifr6_prefixlen = plen_new;
	ipv6_addr_copy(&ifrh->ifr.ifr6_addr, &minfo->home_addr);
	ifrh->old_ifi = oldif;
	INIT_TQUEUE(t, mn_move_homeaddr_task, ifrh);
	queue_task(t, &tq_timer);
	minfo->ifindex = newif;
	return 0;
}
/**
 * mn_handoff - called for every bul entry to send BU to CN
 * @rawentry: bul entry
 * @args: handoff event
 * @sortkey:
 *
 * Since MN can have many home addresses and home networks, every BUL
 * entry needs to be checked
 **/
static int mn_handoff(void *rawentry, void *args, unsigned long *sortkey)
{
	__u8 athome;
	struct mipv6_bul_entry *entry = (struct mipv6_bul_entry *)rawentry;
	struct handoff *ho = (struct handoff *)args;
	int pfixlen = 64; /* Get the prefixlength instead of fixed 64 bits */
	DEBUG_FUNC();

	if (mipv6_prefix_compare(&ho->rtr_new.raddr, 
				 &entry->home_addr, pfixlen)) {
		athome = MN_RETURNING_HOME;
		entry->lifetime = 0;
	} else {
		athome = MN_NOT_AT_HOME;
		entry->lifetime = 
			mipv6_mn_get_bulifetime(&entry->home_addr, 
						&ho->rtr_new.CoA,
						entry->flags);	
	} 

	if (entry->flags & MIPV6_BU_F_HOME) {
		/* Update mn_info for home address 
		 */
		struct mn_info *minfo = NULL;
		
		read_lock_bh(&mn_info_lock);
		minfo = mipv6_mn_get_info(&entry->home_addr);
		if (minfo) { 
			minfo->is_at_home = athome;
			
			if (athome != MN_NOT_AT_HOME) {
				/* Check if home address has been moved around 
				 */
				if (minfo->ifindex != ho->rtr_new.ifindex) {
					DEBUG((DBG_INFO, "Moving home address back to the home interface"));
					create_homeaddr_task(minfo, 64, ho->rtr_new.ifindex, minfo->ifindex);
				}
			}
			
		}
		read_unlock_bh(&mn_info_lock);
		//Update here is_at_home flag for mn_info
		DEBUG((DBG_INFO, "Sending home de ? %d registration for "
		       "home address: %x:%x:%x:%x:%x:%x:%x:%x\n" 
		       "to home agent %x:%x:%x:%x:%x:%x:%x:%x, "
		       "with lifetime %ld, prefixlength %d", 
		       (athome != MN_NOT_AT_HOME),  
		       NIPV6ADDR(&entry->home_addr), 
		       NIPV6ADDR(&entry->cn_addr), entry->lifetime, entry->prefix));
		mipv6_send_upd_option(
			&entry->home_addr, &entry->cn_addr, HA_BU_DELAY, 
			INITIAL_BINDACK_TIMEOUT, MAX_BINDACK_TIMEOUT, 1,
			entry->flags, entry->prefix, entry->lifetime, NULL);
	} else {
		DEBUG((DBG_INFO, "Sending BU for home address: %x:%x:%x:%x:%x:%x:%x:%x \n" 
		       "to CN: %x:%x:%x:%x:%x:%x:%x:%x, "
		       "with lifetime %ld",   NIPV6ADDR(&entry->home_addr), 
		       NIPV6ADDR(&entry->cn_addr), entry->lifetime));
		mipv6_send_upd_option(
			&entry->home_addr, &entry->cn_addr, CN_BU_DELAY, 
			INITIAL_BINDACK_TIMEOUT, MAX_BINDACK_TIMEOUT, 1,
			entry->flags, entry->prefix, entry->lifetime, NULL);
	}

	return ITERATOR_CONT;
}

/**
 * init_home_registration - start Home Registration process
 * @hinfo: mn_info entry for the home address
 * @coa: care-of address
 *
 * Checks whether we have a Home Agent address for this home address.
 * If not starts Dynamic Home Agent Address Discovery.  Otherwise
 * tries to register with home agent if not already registered.
 **/
int init_home_registration(struct mn_info *hinfo, struct in6_addr *coa)
{
	__u32 lifetime;

	DEBUG_FUNC();

	if (mipv6_prefix_compare(&hinfo->home_addr, coa, hinfo->home_plen)) { 
		DEBUG((DBG_INFO, "Adding home address, MN at home"));
		return 0;
	}

	if (ipv6_addr_any(&hinfo->ha)) {
		DEBUG((DBG_INFO, "Home Agent address not set, initiating DHAAD"));
		mipv6_mn_dhaad_send_req(&hinfo->home_addr, hinfo->home_plen, &hinfo->dhaad_id);
	} else {
		int mipv6_dad = 0;
		struct mipv6_bul_entry *bul;

		if (hinfo->man_conf)
			mipv6_pfx_add_ha(&hinfo->ha, coa, hinfo->ifindex);

		bul = mipv6_bul_get(&hinfo->ha, &hinfo->home_addr);
		if (bul) {
			if (!ipv6_addr_cmp(&bul->home_addr, &hinfo->home_addr)){
				DEBUG((DBG_INFO, "BU already sent to HA"));
				mipv6_bul_put(bul);
				return 0;
			}
			mipv6_bul_put(bul);
		}
		lifetime = mipv6_mn_get_bulifetime(
			&hinfo->home_addr, coa, 
			MIPV6_BU_F_HOME | MIPV6_BU_F_ACK);
		DEBUG((DBG_INFO, "Sending initial home registration for "
		       "home address: %x:%x:%x:%x:%x:%x:%x:%x\n" 
		       "to home agent %x:%x:%x:%x:%x:%x:%x:%x, "
		       "with lifetime %ld, prefixlength %d",   
		       NIPV6ADDR(&hinfo->home_addr), 
		       NIPV6ADDR(&hinfo->ha), lifetime, 0));
#ifndef MIPV6_NO_PROXY_DAD
		mipv6_dad = MIPV6_BU_F_DAD;
#endif
		mipv6_send_upd_option(
			&hinfo->home_addr, &hinfo->ha, 
			HA_BU_DELAY, 
			INITIAL_BINDACK_DAD_TIMEOUT, 
			MAX_BINDACK_TIMEOUT, 1,
			MIPV6_BU_F_HOME | MIPV6_BU_F_ACK | mipv6_dad, 
			0, lifetime, NULL);
	}
	return 0;
}

/**
 * mipv6_mobile_node_moved - Send BUs to all HAs and CNs
 * @ho: handoff structure contains the new and previous routers
 *
 * Event for handoff.  Sends BUs everyone on Binding Update List.
 **/
int mipv6_mobile_node_moved(struct handoff *ho)
{
	struct list_head *lh;
	struct mn_info *minfo = NULL;
	int bu_to_prev_router = 1;
	int dummy;

	DEBUG_FUNC();

	if (!mipv6_is_initialized)
		return 0;

	ma_ctl_upd_iface(ho->rtr_new.ifindex, 
			 MA_IFACE_CURRENT | MA_IFACE_HAS_ROUTER, &dummy);

	/* First send BUs to all nodes which are on BU list */
	bul_iterate(mn_handoff, (void *)ho);

	/* Then go through the home infos to see if there's a new one
	 * which has not been registered
	 */
	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);

		if (mipv6_prefix_compare(&minfo->home_addr,
					 &ho->rtr_new.raddr,
					 minfo->home_plen)) {
			minfo->is_at_home = MN_AT_HOME;
			minfo->has_home_reg = 0;
			mipv6_pfx_cancel_send(&ho->rtr_new.raddr,
					      minfo->ifindex);
		} else {
			minfo->is_at_home = MN_NOT_AT_HOME;
			if (!minfo->has_home_reg) {
				init_home_registration(
					minfo, &ho->rtr_new.CoA);
			}
		}
	}
	
	read_unlock_bh(&mn_info_lock);		     
	
	/* Add current care-of address to mn_info list, if current router acts 
	   as a HA. 
	   		*/ 
	 
	if (!mipv6_reverse_tunnel && ho->rtr_new.flags & ND_RA_FLAG_HA && 
	    ho->rtr_new.glob_addr && bu_to_prev_router) 
		mipv6_mn_add_info(&ho->rtr_new.CoA, ho->rtr_new.pfix_len, 
				  MN_AT_HOME, 0, &ho->rtr_new.raddr, 
				  ho->rtr_new.pfix_len, ROUTER_BU_DEF_LIFETIME,
				  0);
				  

	return 0;		
}

/**
 * mipv6_mn_send_home_na - send NA when returning home
 * @haddr: home address to advertise
 *
 * After returning home, MN must advertise all its valid addresses in
 * home link to all nodes.
 **/
void mipv6_mn_send_home_na(struct in6_addr *haddr)
{
	struct net_device *dev = NULL;
	struct in6_addr mc_allnodes;
	struct mn_info *hinfo = NULL;
	struct in6_addr addr;
 
	read_lock_bh(&mn_info_lock);
	hinfo = mipv6_mn_get_info(haddr);
	if (!hinfo) {
		read_unlock_bh(&mn_info_lock);
		return;
	}
	hinfo->is_at_home = 1;
	dev = dev_get_by_index(hinfo->ifindex);
	read_unlock_bh(&mn_info_lock);
	if (dev == NULL) {
		DEBUG((DBG_ERROR, "Send home_na: device not found."));
		return;
	}
	
	ipv6_addr_all_nodes(&mc_allnodes);
	if (ipv6_get_lladdr(dev, &addr) == 0)
		ndisc_send_na(dev, NULL, &mc_allnodes, &addr, 0, 0, 1, 1);
	ndisc_send_na(dev, NULL, &mc_allnodes, haddr, 0, 0, 1, 1);
	dev_put(dev);
}

static int mn_dev_event(struct notifier_block *nb, unsigned long event, void *ptr)
{
	struct net_device *dev = ptr;
	struct list_head *lh;
	struct mn_info *minfo;
	int newif = 0, err =0;

	/* here are probably the events we need to worry about */
	switch (event) {
	case NETDEV_UP:
		DEBUG((DBG_DATADUMP, "New netdevice %s registered.", dev->name));
		if((dev->type != ARPHRD_LOOPBACK) &&
		   (dev->type != ARPHRD_IPV6_IPV6_TUNNEL))
			ma_ctl_add_iface(dev->ifindex);
		break;
	case NETDEV_GOING_DOWN:
		DEBUG((DBG_DATADUMP, "Netdevice %s disappeared.", dev->name));
		ma_ctl_upd_iface(dev->ifindex, MA_IFACE_NOT_PRESENT, &newif);
		newif =  ma_ctl_get_preferred_if();
		if (newif <= 0  || (newif == loopback_dev.ifindex)) {
			DEBUG((DBG_WARNING, "Couldn't find a valid interface to add home address into"));
			err = -1;
		}
		else
			DEBUG((DBG_INFO, "Netdevice %s was in use.  Switch to %d.",
			       dev->name, newif));
		/* 
		 * Go through mn_info list and move all home addresses on the
		 * netdev going down to a new device. This will make it 
                 * practically impossible for the home address to return home,
		 * but allow MN to retain its connections using the address.
		 */

		write_lock_bh(&mn_info_lock);
		list_for_each(lh, &mn_info_list) {
			minfo = list_entry(lh, struct mn_info, list);
			if (minfo->ifindex == dev->ifindex) {
				if (err)
					minfo->ifindex = 0;
				else if (minfo->man_conf && 
					 create_homeaddr_task(minfo, 128, 
							      newif, 0) < 0) {
					write_unlock_bh(&mn_info_lock);
					return NOTIFY_DONE;
				}
			}
				
		}
		
		write_unlock_bh(&mn_info_lock);
	}
	return NOTIFY_DONE;
}

struct notifier_block mipv6_mn_dev_notifier = {
	mn_dev_event,
	NULL,
	0 /* check if using zero is ok */
};

static void deprecate_addr(struct mn_info *minfo)
{
	/*
	 * Lookup address from IPv6 address list and set deprecated flag
	 */
	
}

int mipv6_mn_foreign_rt_clear(struct router *rtr, struct rt6_info *rt) 
{
	struct list_head *lh;
	struct mn_info *minfo; 
	int ret = 1;
	
	DEBUG_FUNC();

	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (mipv6_prefix_compare(&rtr->raddr, 
					 &minfo->home_addr, 
					 minfo->home_plen)) {

			/* Keep direct routes to home prefix */
			if (!(rt->rt6i_flags & RTF_GATEWAY) &&
			    mipv6_prefix_compare(&rt->rt6i_dst.addr, 
						 &minfo->home_addr, 
						 minfo->home_plen)) {
				DEBUG((DBG_INFO, 
				       "Kept direct route to home prefix"));
				ret = 0;
				break;
			} 
			/* Keep gateway routes through router */
			if ((rt->rt6i_flags & RTF_GATEWAY) &&
			    !ipv6_addr_cmp(&rt->rt6i_gateway,
					   &rtr->ll_addr)) {
				DEBUG((DBG_INFO, "Kept routes thru gateway"));
				ret = 0;
				break;
			}
		} else {

			/* Don't delete real route to ha */
			if ((rt->rt6i_flags & RTF_HOST) && 
			    (rt->rt6i_flags & RTF_GATEWAY) &&
			    !ipv6_addr_cmp(&rt->rt6i_dst.addr, &minfo->ha) &&
			    !ipv6_addr_cmp(&rt->rt6i_gateway, 
					   &rtr->ll_addr)) {
				DEBUG((DBG_INFO, "Kept route to ha"));
				ret = 0;
				break;
			}
			
			/* Router might use its global address 
			   for NUD so keep the route to it */
			if (rtr->glob_addr && (rt->rt6i_flags & RTF_HOST) && 
			    !(rt->rt6i_flags & RTF_GATEWAY) &&
			    !ipv6_addr_cmp(&rt->rt6i_dst.addr, &rtr->raddr)) {
				DEBUG((DBG_INFO, 
				       "Kept route to global "
				       "address of router"));
				ret = 0;
				break;
			}
		}
	}
	read_unlock_bh(&mn_info_lock);
	/* Remove others */

	return  ret;
}

static __inline__ int dev_is_mipv6_tnl(struct net_device *dev)
{
	struct ipv6_tnl *t = (struct ipv6_tnl *)dev->priv;
	return (dev->type == ARPHRD_IPV6_IPV6_TUNNEL && is_mipv6_tnl(t));
}

int mipv6_mn_reverse_tnl_rt_keep(struct rt6_info *rt)
{	
	DEBUG_FUNC();
	return dev_is_mipv6_tnl(rt->rt6i_dev);
}

static __inline__ int add_rt_to_ha(struct in6_addr *ha_addr, 
				   struct in6_addr *coa, 
				   struct in6_addr *gw_addr,
				   int ifindex) 
{
	struct in6_rtmsg rtmsg;
	int err;
	
	memset(&rtmsg, 0, sizeof(rtmsg));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, ha_addr);
	rtmsg.rtmsg_dst_len = 128;
	ipv6_addr_copy(&rtmsg.rtmsg_gateway, gw_addr);
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_HOST | RTF_GATEWAY;
	rtmsg.rtmsg_ifindex = ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((err = ip6_route_add(&rtmsg)) == -EEXIST) {
		DEBUG((DBG_INFO,"add_rt_to_ha: route exists"));
		return 0;
	}
	return err;
}


static __inline__ int add_glbl_rtr_rt(struct in6_addr *rtr_addr, 
				      struct in6_addr *coa, 
				      int ifindex) 
{
	struct in6_rtmsg rtmsg;
	int err;
	
	DEBUG((DBG_INFO, "adding route to global address of router "
	       "%x:%x:%x:%x:%x:%x:%x:%x", NIPV6ADDR(rtr_addr)));
	
	memset(&rtmsg, 0, sizeof(rtmsg));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, rtr_addr);
	rtmsg.rtmsg_dst_len = 128;
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_HOST;
	rtmsg.rtmsg_ifindex = ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((err = ip6_route_add(&rtmsg)) == -EEXIST) {
		DEBUG((DBG_INFO,"add_glbl_rtr_rt: route exists"));
		return 0;
	}
	return err;
}

int mipv6_mn_add_support_routes(struct router *rtr)
{
	struct list_head *lh;
	struct mn_info *minfo = NULL; 
	int err = 0;
	DEBUG_FUNC();
	
	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!mipv6_prefix_compare(&rtr->raddr, 
					  &minfo->home_addr, 
					  minfo->home_plen)) {
			err = add_rt_to_ha(&minfo->ha, &rtr->CoA, 
					   &rtr->ll_addr, rtr->ifindex);
			if (err) {
				DEBUG((DBG_CRITICAL, 
				       "add route to ha failed"));
				return err;
			}
			if (rtr->glob_addr) {
				err = add_glbl_rtr_rt(&rtr->raddr, &rtr->CoA, 
						      rtr->ifindex); 
				if (err) {
					DEBUG((DBG_WARNING, 
					       "add route to router failed"));
				}
			}
		}
	}
	read_unlock_bh(&mn_info_lock);
	return err;
}


int mipv6_mn_approve_rt(struct  in6_addr *addr)
{
	struct list_head *lh;
	struct mn_info *minfo; 
	int ret = 0;
	
	DEBUG_FUNC();

	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (minfo->man_conf &&
		    mipv6_prefix_compare(&minfo->home_addr, addr, 
					 minfo->home_plen)) {
			ret = 1;
			break;
		}
	}
	read_unlock_bh(&mn_info_lock);

	return ret;
}

int mipv6_mn_may_solicit_ha(struct in6_addr *target)
{
	struct list_head *lh;
	struct mn_info *minfo; 
	int ret = 0;

	DEBUG_FUNC();

	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!ipv6_addr_cmp(&minfo->ha, target) && 
		    (ret = (minfo->is_at_home == MN_RETURNING_HOME))) {
			break;
		}
	}
	read_unlock_bh(&mn_info_lock);
	return ret;
}

int mipv6_mn_may_advertise(struct in6_addr *target, struct in6_addr *saddr)
{
	struct list_head *lh;
	struct mn_info *minfo = NULL; 
	int ret = 1;

	DEBUG_FUNC();

	read_lock_bh(&mn_info_lock);
	list_for_each(lh, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!ipv6_addr_cmp(&minfo->home_addr, target)) {
			break;
		}
	}
	/* Only answer to a NS to the  home address from the HA when it is 
	   registered home */
	if (minfo != NULL && mipv6_bul_get(&minfo->ha, target) != NULL) {
		ret = !ipv6_addr_cmp(saddr, &minfo->ha);
	}
	read_unlock_bh(&mn_info_lock);

	return ret;
}

int __init mipv6_initialize_mn(void)
{
	struct net_device *dev;

	DEBUG_FUNC();

	ma_ctl_init();
	for (dev = dev_base; dev; dev = dev->next) {
		if ((dev->flags & IFF_UP) && 
		    (dev->type != ARPHRD_LOOPBACK) &&
		    (dev->type != ARPHRD_IPV6_IPV6_TUNNEL)){
			ma_ctl_add_iface(dev->ifindex);
		}
	} 
	DEBUG((DBG_INFO, "Multiaccess support initialized"));

	register_netdevice_notifier(&mipv6_mn_dev_notifier);

	if (!mipv6_reverse_tunnel) {
		ipv6_ipv6_tnl_register_hook(&mipv6_mn_tnl_rcv_send_bu_ops);
	} 
	ipv6_ipv6_tnl_register_hook(&mipv6_mn_tnl_xmit_stats_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_mn_tnl_rcv_stats_ops);

	return 0;
}

void mipv6_shutdown_mn(void)
{
	struct list_head *lh, *tmp;
	struct mn_info *minfo;
	DEBUG_FUNC();

	ipv6_ipv6_tnl_unregister_hook(&mipv6_mn_tnl_rcv_stats_ops);
	ipv6_ipv6_tnl_unregister_hook(&mipv6_mn_tnl_xmit_stats_ops);

	if (!mipv6_reverse_tunnel) {
		ipv6_ipv6_tnl_unregister_hook(
			&mipv6_mn_tnl_rcv_send_bu_ops);
	}
	ma_ctl_clean();

	unregister_netdevice_notifier(&mipv6_mn_dev_notifier);
	write_lock_bh(&mn_info_lock);

	list_for_each_safe(lh, tmp, &mn_info_list) {
		minfo = list_entry(lh, struct mn_info, list);
		if (!minfo->is_at_home) 
			deprecate_addr(minfo);
		list_del_init(&minfo->list);
		kfree(minfo);
	}
	write_unlock_bh(&mn_info_lock);
}
