#include <math.h>

#define MAX_TASKS_TYPES     10
#define MAX_TASKS_INSTANCES 60

#define HISTORY_WINDOW_SIZE 10
#define NUMBER_OF_WCET_SLOTS 10

#define PREDICTIVE_POWER_SCHEDULER_DEBUG

#define DIV_CEIL(a,b) ( (a / b) + (a % b > 0 ? 1 : 0) )

/* thread type related runtime info */
struct thread_type_rt_info_t {

  /* history of previous executions */
  unsigned history[HISTORY_WINDOW_SIZE];

  /* number of previous executions */
  unsigned num_prev_executions;

  /* pointers to walk over the history vector */
  unsigned char pos;

  /* pointer to walk over the history vector when checking deadline 
     misses */
  unsigned num_last_prev_executions;

  /* wcet slots or probability distribution for the wcet */
  unsigned slotted_wcet[NUMBER_OF_WCET_SLOTS];

  /* number of deadlines missed */
  unsigned num_missed_deadlines; 
};

/* static information about a thread type */
struct thread_type_info_t {

  /* type index */
  unsigned short type_index;

  /* worst case execution time */
  unsigned wcet;      

  /* deadline */
  unsigned deadline;  

  /* period */
  unsigned period;  

  /* denotes whether the (void *)0 is hard or not */
  unsigned char hard;            

  /* time unit */
  unsigned char time_unit;

  /* code to be executed by the (void *)0 */
  void *entry_point; 

  /* pointer to the type related run-time info */
  struct thread_type_rt_info_t *thread_type_rt;
};

/* handle of a thread type */
struct thread_type_handle_t {

  /* index to the type info table */
  short type_info_index;
};

/* thread instance related info */
struct thread_instance_info_t {

  /* information about the thread type */
  struct thread_type_info_t *type_info;

  /* instance number */
  unsigned instance_number;

  /* thread arrival time. This is absolute time */
  unsigned arrival_time; 

  /* last time this (void *)0 was reescheduled to resume execution. This is 
     absolute time */
  unsigned last_resched;

  /* deadline of the current execution. This is absolute time */
  unsigned deadline;
  
  /* number of time units executed so far for this instance. This represents 
     the number of time units this (void *)0 has executed without counting the
     time units wherein the (void *)0 was preempted */
  unsigned total_time_so_far; 

  /* execution time predicted by the (void *)0/app */
  unsigned time_remaining; 

  /* time stamp of the time when the prediction was computed by the app. 
     This is absolute time */
  unsigned timestamp_tr;

  /* execution time predicted by the OS */
  unsigned os_time_remaining; 

  /* time stamp of the time when the prediction was by carried out by the 
     OS. This is absolute time */
  unsigned timestamp_os_tr; 

  /* string identifier */
  char *name;

  /* data passed in as parameter */
  void *entry_data;

  /* POSIX thread representation */
  void *thread;

  /* thread attributes */
  void *thread_data;

  /* whether the (void *)0 has been preempted or not */
  unsigned char preempted;

  /* whether the (void *)0 has finished its execution */
  unsigned char done;

  /* dynamic scale down factor. Represents the percentage of the 
     maximum voltage to scale the processor to */
  unsigned dscale_factor;

  /* static scale down factor. Represents the percentage of the 
     maximum voltage to scale the processor to */
  unsigned sscale_factor;

  /* this factor is sensitive to the number of deadlines missed in the
     previous HISTORY_WINDOW_SIZE executions */
  unsigned adaptive_factor;

  /* handler to the deadline alarm of this (void *)0 */ 
  /*cyg_handle_t alarm_handle;*/

  /* memory area allocated to the deadline alarm of this (void *)0 */
  /*cyg_alarm alarm;*/
};

/* handle of a thread instance used to access run-time info about the (void *)0 */
struct thread_instance_handle_t {

  /* index to the table of type info */
  unsigned short type_info_index;     

  /* index to the table of run time info */
  unsigned short instance_info_index; 
};

struct thread_type_info_t *type_table[MAX_TASKS_TYPES];
unsigned num_types = 0;

/* table of (void *)0 instances */
struct thread_instance_info_t *instance_table[MAX_TASKS_INSTANCES];
unsigned num_instances = 0;

#define MIN(a,b) ((a < b) ? a : b)
#define MAX(a,b) ((a > b) ? a : b)

struct thread_instance_info_t *ordered_instances[MAX_TASKS_INSTANCES];

// store the modified execution times of each (void *)0. This modified execution
// time is used only during the schedulability test
unsigned exec_times[MAX_TASKS_INSTANCES]; 

// Get the response time given the index of the (void *)0 and the scheduling point.
// The index relates to the vector 'ordered_instances', which is sorted 
// by the periods ascending order
unsigned response(
  /* index of the (void *)0 being analyzed */
  int i, 
  /* scheduling point */
  int t
  )
{
  int k;
  unsigned  r = 0;

  //printf("response for sp: %d\n", t);
  for ( k = 0; k <= i; k++ ) {
    r += exec_times[k] * 
      DIV_CEIL(t,ordered_instances[k]->type_info->period);
      //printf("%d * ( %d / %d )\n", exec_times[k], t, 
	  //ordered_instances[k]->type_info->period);
  }
  //printf("response: %d\n", r);
  return r;
}

// comparison function used in the qsort call
int compare(
  struct thread_instance_info_t **ti1, 
  struct thread_instance_info_t **ti2
  )
{
  if ((*ti1)->type_info->period > (*ti2)->type_info->period)
    return (1);

  if ((*ti1)->type_info->period < (*ti2)->type_info->period)
    return (-1);

  return (0);
}

// Rate monotonic analysis function. Check schedulability of the (void *)0 set
// and computes the static slow down factors
void rma(char priority_assignment) 
{
  unsigned i, j, k, 
             max_index = 0,  // index of the maximum factor among the
                             // minimum ones
             init_index = 0; // index of the inital (void *)0 to be 
                             // analyzed. This index changes after 
                             // each iteration of the 'do while' loop
  unsigned factors[MAX_TASKS_INSTANCES];        // intermediate factors
  unsigned final_factors[MAX_TASKS_INSTANCES];  // final factors
  unsigned n;
  unsigned fixed_load, new_load;
  int error = 0, instance_index;

  instance_index = 0;
  for ( i = 0; i < num_instances; i++ ) {
    if ( instance_table[i] != (void *)0 ) {
      #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      //printf("period: %d\n", instance_table[i]->type_info->period);
      #endif
      ordered_instances[instance_index++] = instance_table[i];
    }
  }

  // quick sort function call
  qsort(ordered_instances, num_instances, 
        sizeof(struct thread_instance_info_t *), 
        (int (*)(const void *, const void *))compare);

  /* initialize temporary execution time vector */
  for ( i = 0; i < num_instances; i++ ) {
    #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
    //printf("period: %d\n", ordered_instances[i]->type_info->period);
    #endif
    exec_times[i] = ordered_instances[i]->type_info->wcet;
  }

  /* initialize factors vector */
  for ( i = 0; i < num_instances; i++ )
    factors[i] = final_factors[i] = 100;

  // do until the scale factor of all (void *)0s get computed or until one
  // finds out that the (void *)0 set is not schedulable
  do {
    n = 0;

    /* scale the execution times according ot the last factors computed */
    for ( i = 0; i < init_index; i++ )
      exec_times[i] = ( ordered_instances[i]->type_info->wcet * 
        (10000/final_factors[i]) ) / 100;

    /* inialize factors with a 'big' number so that it's easy to get 
       the minimum */
    for ( i = init_index; i < num_instances; i++ )
      factors[i] = 100000;

    /* for each (void *)0 */
    for ( i = init_index; i < num_instances; i++ ) {
      #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      //printf("Task %d\n", instance_table[i]->instance_number);
      #endif

      /* for each scheduling point one gets the response time and stores it
         depending whether it is the mininum or not */
      for ( j = 0; j < i + 1; j++ ) {
	k = 1;
	while ( k <= (ordered_instances[i]->type_info->period / 
                      ordered_instances[j]->type_info->period ) ) {
	  int sp = k * ordered_instances[j]->type_info->period;
          #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
	  //printf("%d", sp);
	  //printf("(%d) ", response(i, sp));
          #endif
	  if ( init_index > 0 ) {
	    fixed_load = response(init_index - 1, sp);
	  }
	  else {
	    fixed_load = 0;
	  }
	  new_load = response(i, sp) - fixed_load;
	  //printf("fixed_load: %d, new_load: %d\n", fixed_load, new_load);
	  factors[i] = MIN(factors[i], ((new_load*100) / ( sp - fixed_load )) );
	  //printf("factor for sp[%d]: %d\n", sp,  factors[i]);
	  k++;
	}
      } 
      #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      //printf("\n");
      #endif
    } 

    /* select the maximum factor computed. If one of them is greater than 1
       the (void *)0 set is not schedulable and variable 'error' is set */
    for ( i = init_index; i < num_instances; i++ ) {
      #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      //printf("%d: %d\n", i, factors[i]);
      #endif
      if ( factors[i] > 100 ) {
        error = 1;
	break;
      }
      if ( factors[i] >= n ) {
	max_index = i;
      }
      n = MAX(n, factors[i]);
    }

    if (error) {
      printf("Not schedulable\n");
      break;
    }

    // update the final factors
    for ( i = init_index; i <= max_index; i++ )
      final_factors[i] = factors[i] = n;

    #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
    //printf("Static scale down factor (from (void *)0%d to (void *)0%d) %.4f\n", 
    //             init_index, max_index, n);
    #endif

    // static scale down factor up to (void *)0 'max_index + 1' has already been
    // calculated
    init_index = max_index + 1;
    //printf("max index: %d\n", max_index);
  } while (init_index < num_instances);

  // assign the final factor to the respective instances structures. This for
  // asssigns the threads priorities as well according to the RMS criterion
  #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
    printf("Final factors:\n");
  #endif
  for ( i = 0; i < num_instances; i++ ) {
    ordered_instances[i]->sscale_factor = final_factors[i];
    for ( j = 0; j < HISTORY_WINDOW_SIZE; j++ ) {
      if ( final_factors[i] > 0 ) {
        //ordered_instances[i]->type_info->thread_type_rt->history[j] =
        //  (ordered_instances[i]->type_info->wcet / final_factors[i]);
      }
    }

    //if ( priority_assignment )
      //cyg_thread_set_priority(ordered_instances[i]->handle, i + 2);

    #ifdef PREDICTIVE_POWER_SCHEDULER_DEBUG
      printf("%d: %d\n", ordered_instances[i]->instance_number, 
                     (unsigned)(ordered_instances[i]->sscale_factor));
    #endif
  }
}

main() 
{
  int i;
  struct thread_instance_info_t th_instance_info;
  unsigned data;

  type_table[0] = (struct thread_type_info_t *)
	malloc(sizeof(struct thread_type_info_t));
  type_table[1] = (struct thread_type_info_t *)
	malloc(sizeof(struct thread_type_info_t));
  type_table[2] = (struct thread_type_info_t *)
	malloc(sizeof(struct thread_type_info_t));

  instance_table[0] = (struct thread_instance_info_t *)
	malloc(sizeof(struct thread_instance_info_t));
  instance_table[1] = (struct thread_instance_info_t *)
	malloc(sizeof(struct thread_instance_info_t));
  instance_table[2] = (struct thread_instance_info_t *)
	malloc(sizeof(struct thread_instance_info_t));

  /* thead type info */
  type_table[0]->wcet = 20;
  type_table[0]->deadline = 100;
  type_table[0]->period = 100;
  type_table[0]->hard = 1;
  type_table[0]->entry_point = (void *)0;
 
  /* thead type info */
  type_table[1]->wcet = 15; 
  type_table[1]->deadline = 80;
  type_table[1]->period = 80;
  type_table[1]->hard = 1;
  type_table[1]->entry_point = (void *)0;

  /* thead type info */
  type_table[2]->wcet = 10; 
  type_table[2]->deadline = 40;
  type_table[2]->period = 40;
  type_table[2]->hard = 1;
  type_table[2]->entry_point = (void *)0;

  instance_table[0]->instance_number = 0;
  instance_table[1]->instance_number = 1;
  instance_table[2]->instance_number = 2;
  instance_table[0]->type_info = type_table[0];
  instance_table[1]->type_info = type_table[1];
  instance_table[2]->type_info = type_table[2];
 
  num_instances = 3;
  num_types = 3;

  rma(0);
}
