/************************************************************************
* Copyright (C) 2004
* Dynamo Middleware Group
*
* version 0.5
*
* Authors: Michael Philpott, Shivajit Mohapatra 
*
* This library (soon to be) is to be used to retreive the iPaqs backlight, 
* battery, cpu, memory and network information.  It retrieves the network,
* memory and cpu utilization from the kernels /proc directory.  The backlight
* and battery information are obtained though the system devices using a
* system specific datastructure.
* 
************************************************************************/

#ifndef _DYNMIDDLEWARE_
#define _DYNMIDDLEWARE_

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h> 
#include <string.h> 
#include <stdlib.h>
#include <sys/uio.h>
#include <memory.h>
#include <pthread.h> 
#include <unistd.h>

/**
 * Structures
 */

/**
 * @brief Contains Backlight information.
 */
struct dynamo_backlight 
{
	int level;			///< Level from 0 - 255
	int status;			///< If the backlight is currently on or off 1|0
};

/**
 * @brief Contains Battery information.
 */
struct dynamo_battery 
{
	int ac_status;		///< If the device is plugged
	int level;			///< The power level of the battery 1 - 100
	int min_remaining;	///< minutes until there is no power
};

/**
 * @brief Contains CPU Information
 */
struct dynamo_cpu 
{
	char model[40];		///< The model name of the CPU
	float load;			///< load value of the cpu 0 - 1.0 average of CPU over the past min
	int total_process;	///< total processes active on the device
	int active_process;	///< currently active processes on the server
};

/**
 * @brief Contains Memory Information.
 */
struct dynamo_memory 
{
	int total_mem;		///< Total memory available to the device
	int free_mem;		///< Free memory on the device
	int total_swap;		///< Total swap available on the device
	int free_swap;		///< Free swap on the device
};

/**
 * @brief Contains Network Information
 */
struct dynamo_network 
{
	int connected;		///< If the device is connected to the network
	int wireless;		///< If the device is connected to a wireless network
	int link;			///< The link strength of the wireless signal
	int signal;			///< The signal strength of the wireless network
};

//
// Shared Memory Operations

/**
 * @brief creates a hunk of shared memory of size with
 * with key of key and returns and id to that memory
 */
int createSharedMemory(int key, int size);

/**
 * @brief reads the shared memory at shmid and reads a
 * chunk of memory [size] into the structure data
 */
int readSharedMemory(int shmid, void* data, int size);

/**
 * @brief writes data of size(size) into the memory with id shmid
 */
int writeSharedMemory(int shmid, void* data, int size);

/**
 * @brief releases the memory with id of shmid
 */
int freeSharedMemory(int shmid);

//
// Semaphore Operations

/**
 * @brief translate semaphore key to an ID 
 */
int createSemaphore(int key);

/**
 * @brief free Semaphore
 */
int freeSemaphore(int sem_id);

/**
 * @brief acquire semaphore 
 */
void P(int sid);

/**
 * @brief release semaphore 
 */
void V(int sid);

/**
 * @brief sends a datagram of buf to targetIP:targetPort from localPort
 */
int SendDatagram(char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

/**
 * @brief creates a socket at localIP: localPort and returns a pointer to that
 * socet
 */
int CreateUDPSocket(char *localIP, int localPort);

/**
 * @brief Recives a datagram from socket sockfd of size and populateds buf
 */
int UDP_Recv(int sockfd, void *buf, int size);

/**
 * @brief Sends a datagram (buf) of size (bufsize) from socket sockfd to 
 * targetIP:targetPort from localPort
 */
int UDP_Send(int sockfd, char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

#endif



