/************************************************************************
* Copyright (C) 2004
* Dynamo Middleware Group
*
* version 0.5
*
* Authors: Michael Philpott, Shivajit Mohapatra 
*
* This library (soon to be) is to be used to retreive the iPaqs backlight, 
* battery, cpu, memory and network information.  It retrieves the network,
* memory and cpu utilization from the kernels /proc directory.  The backlight
* and battery information are obtained though the system devices using a
* system specific datastructure.
* 
************************************************************************/

#ifndef _DYNMIDDLEWARE_
#define _DYNMIDDLEWARE_

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h> 
#include <string.h> 
#include <stdlib.h>
#include <sys/uio.h>
#include <memory.h>
#include <pthread.h> 
#include <unistd.h>

/**
 * Structures
 */

/**
 * @brief Contains Backlight information.
 */
struct dynamo_backlight 
{
	int level;			///< Level from 0 - 255
	int status;			///< If the backlight is currently on or off 1|0
};

/**
 * @brief Contains Battery information.
 */
struct dynamo_battery 
{
	int ac_status;		///< If the device is plugged
	int level;			///< The power level of the battery 1 - 100
	int min_remaining;	///< minutes until there is no power
};

/**
 * @brief Contains CPU Information
 */
struct dynamo_cpu 
{
	char model[40];		///< The model name of the CPU
	float load;			///< load value of the cpu 0 - 1.0 average of CPU over the past min
	int total_process;	///< total processes active on the device
	int active_process;	///< currently active processes on the server
};

/**
 * @brief Contains Memory Information.
 */
struct dynamo_memory 
{
	int total_mem;		///< Total memory available to the device
	int free_mem;		///< Free memory on the device
	int total_swap;		///< Total swap available on the device
	int free_swap;		///< Free swap on the device
};

/**
 * @brief Contains Network Information
 */
struct dynamo_network 
{
	int connected;		///< If the device is connected to the network
	int wireless;		///< If the device is connected to a wireless network
	int link;			///< The link strength of the wireless signal
	int signal;			///< The signal strength of the wireless network
};

/**
 * @brief Gets the backlight level
 * Returns the backlight level 0 - 255
 * it returns -1 if there was an error reading the touchscreen
 */
extern int get_backlight_level();

/**
 * @brief Sets the backlight to lvl
 * Allows the caller to set the backlight intensity
 */
extern int set_backlight_level(int lvl);

/**
 * @brief Gets the backlights status [on|off]
 * Returns a number 0 or 1 representing if the backlight is on or off
 * it returns -1 if there was an error reading the touchscreen
 */
extern int get_backlight_status();

/**
 * @brief Sets the backlight [on 1|off 0]
 * Allows the caller to set the status of the backlight 1 for on 0 for off
 */
extern int set_backlight_status(int status);

/**
 * @brief Gets if the Device is plugged in
 * Returns the systems AC status (is the system pluged in)
 * returns 0 or 1
 */
extern int get_battery_ac_status();

/**
 * @brief Gets devices battery level
 * Returns the systems power level in percentage
 * 0(empty) - 100(full)
 */
extern int get_battery_power_level();

/**
 * @brief Gets the amount of power in minutes remaining
 * Returns the amount of minutes that the system will have power for
 * Note: need to implament to return -1 if its pluged in
 */
extern int get_battery_min_remaining();

/**
 * @brief Gets the CPU model string
 * Returns a char* of the processors model information
 * this will have to be freed by the caller
 */ 
extern char* get_cpu_model();

/**
 * @brief returns the average load of the device
 * Returns the average load of the cpu over the past minute
 */
extern float get_cpu_load();

/**
 * @brief Gets the amount of total processes on the device
 * Returns the amount of processes that have been started (active or passive)
 */
extern int get_cpu_total_process();

/**
 * @brief Gets the current number of active processes on the device
 * Returns the amount of active tasks on the device
 */
extern int get_cpu_running_process();

/**
 * @brief Gets the total amount of memory
 * Returns the full amount of memory (ie your system has 512 megs of ram)
 */
extern int get_mem_total();

/**
 * @brief Gets the amount of free memory
 * Returns the amount of free memory available on your system
 */
extern int get_mem_free();

/**
 * @brief Gets the total amount of swap
 * Returns the total swap amount avaiable to your system
 */
extern int get_mem_total_swap();

/**
 * @brief Gets the amount of free swap
 * Returns the free swap amount of the system
 */
extern int get_mem_free_swap();

/**
 * @brief Gets the current network connection status
 * Returns 1 if the device is connected to the network or returns a 0 if 
 * there is no network present
 */  
extern int get_net_is_connected();

/**
 * @brief Gets the current wireless network connection status
 * Returns 1 if the device is connected to a wireless network or returns a 0 if 
 * there is no wireless network present
 *
 * need to test when the card is in but not connected
 */ 
extern int get_net_is_wireless();

/**
 * @brief Gets the link quality of the wireless network
 * Returns the link quality of the wireless network if nothing is present returns
 * a -1
 */  
extern int get_net_link_quality();

/**
 * @brief Gets the signal strength of the wireless network
 * Returns the signal quality of the wireless network if nothing is present returns
 * a -1
 */  
extern int get_net_sig_strength();

/**
 * @brief Collects the devices backlight information
 * @see get_backlight_level()
 * @see get_backlight_status()
 */
extern int get_backlight_info(struct dynamo_backlight* backlight);

/**
 * @brief Collects the devices battery information
 * @see get_battery_ac_status()
 * @see get_battery_power_level()
 * @see get_battery_min_remaining()
 */
extern int get_battery_info(struct dynamo_battery* battery);

/**
 * @brief Collects the devices cpu information
 * @see get_cpu_model()
 * @see get_cpu_load()
 * @see get_cpu_total_process()
 * @see get_cpu_running_process()
 */
extern int get_cpu_info(struct dynamo_cpu* cpu);

/**
 * @brief Collects the devices memory information
 * @see get_mem_total()
 * @see get_mem_free()
 * @see get_mem_total_swap()
 * @see get_mem_free_swap()
 */
extern int get_memory_info(struct dynamo_memory* memory);

/** 
 * @brief Collects the devices network information
 * @see get_net_is_connected()
 * @see get_net_is_wireless()
 * @see get_net_link_quality()
 * @see get_net_sig_strength()
 */
extern int get_network_info(struct dynamo_network* network);

//
// Shared Memory Operations

/**
* @brief creates a hunk of shared memory of size with
 * with key of key and returns and id to that memory
 */
int createSharedMemory(int key, int size);

/**
* @brief reads the shared memory at shmid and reads a
 * chunk of memory [size] into the structure data
 */
int readSharedMemory(int shmid, void* data, int size);

/**
* @brief writes data of size(size) into the memory with id shmid
 */
int writeSharedMemory(int shmid, void* data, int size);

/**
* @brief releases the memory with id of shmid
 */
int freeSharedMemory(int shmid);

//
// Semaphore Operations

/**
* @brief translate semaphore key to an ID 
 */
int createSemaphore(int key);

/**
* @brief free Semaphore
 */
int freeSemaphore(int sem_id);

/**
* @brief acquire semaphore 
 */
void P(int sid);

/**
* @brief release semaphore 
 */
void V(int sid);

/**
* @brief sends a datagram of buf to targetIP:targetPort from localPort
 */
int SendDatagram(char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

/**
* @brief creates a socket at localIP: localPort and returns a pointer to that
 * socet
 */
int CreateUDPSocket(char *localIP, int localPort);

/**
* @brief Recives a datagram from socket sockfd of size and populateds buf
 */
int UDP_Recv(int sockfd, void *buf, int size);

/**
* @brief Sends a datagram (buf) of size (bufsize) from socket sockfd to 
 * targetIP:targetPort from localPort
 */
int UDP_Send(int sockfd, char *targetIP, int targetPort, int localPort, void *buf, int bufsiz);

#endif

