#!/usr/local/bin/python
#
# Exif information decoder
# Written by Thierry Bousch <bousch@topo.math.u-psud.fr>
# Public Domain
#
# Heavily modified for use in webimg, D. Eppstein, Jun 2001
#
# $Id: exifdump.py,v 1.13 1999/08/21 14:46:36 bousch Exp $
#
# Since I don't have a copy of the Exif standard, I got most of the
# information from the TIFF/EP draft International Standard:
#
#   ISO/DIS 12234-2
#   Photography - Electronic still picture cameras - Removable Memory
#   Part 2: Image data format - TIFF/EP
#
# You can (and should) get a copy of this document from PIMA's web site;
# their URL is:  http://www.pima.net/it10a.htm
# You'll also find there some documentation on DCF (Digital Camera Format)
# which is based on Exif.
#
# Another must-read is the TIFF 6.0 specification, which can be
# obtained from Adobe's FTP site, at
# ftp://ftp.adobe.com/pub/adobe/devrelations/devtechnotes/pdffiles/tiff6.pdf
#
# Many thanks to Doraneko <doraneko@unioncg.or.jp> for filling the
# holes in the Exif tag list.

import sys, string, time
import strptime

def s2n_motorola(str):
    x = 0
    for c in str:
        x = (x << 8) | ord(c)
    return x

def s2n_intel(str):
    x = 0
    y = 0
    for c in str:
        x = x | (ord(c) << y)
        y = y + 8
    return x

class Fraction:

    def __init__(self, num, den):
        self.num = num
        self.den = den

    def __repr__(self):
        # String representation
        return '%d/%d' % (self.num, self.den)

class TIFF_file:

    def __init__(self, data):
        self.data = data
        self.endian = data[0]

    def s2n(self, offset, length, signed=0):
        slice = self.data[offset:offset+length]
        if self.endian == 'I':
            val = s2n_intel(slice)
        else:
            val = s2n_motorola(slice)
        # Sign extension ?
        if signed:
            msb = 1 << (8*length - 1)
            if val & msb:
                val = val - (msb << 1)
        return val

    def first_IFD(self):
        return self.s2n(4, 4)

    def next_IFD(self, ifd):
        entries = self.s2n(ifd, 2)
        return self.s2n(ifd + 2 + 12 * entries, 4)

    def list_IFDs(self):
        i = self.first_IFD()
        a = []
        while i:
            a.append(i)
            i = self.next_IFD(i)
        return a

    def dump_IFD(self, ifd):
        entries = self.s2n(ifd, 2)
        a = []
        for i in range(entries):
            entry = ifd + 2 + 12*i
            tag = self.s2n(entry, 2)
            type = self.s2n(entry+2, 2)
            if not 1 <= type <= 10:
                continue # not handled
            typelen = [ 1, 1, 2, 4, 8, 1, 1, 2, 4, 8 ] [type-1]
            count = self.s2n(entry+4, 4)
            offset = entry+8
            if count*typelen > 4:
                offset = self.s2n(offset, 4)
            if type == 2:
                # Special case: nul-terminated ASCII string
                values = self.data[offset:offset+count-1]
            else:
                values = []
                signed = (type == 6 or type >= 8)
                for j in range(count):
                    if type % 5:
                        # Not a fraction
                        value_j = self.s2n(offset, typelen, signed)
                    else:
                        # The type is either 5 or 10
                        value_j = Fraction(self.s2n(offset,   4, signed),
                                           self.s2n(offset+4, 4, signed))
                    values.append(value_j)
                    offset = offset + typelen
            # Now "values" is either a string or an array
            a.append((tag,type,values))
        return a

def proc_date(datestring,safe_value):
    try:
        date = time.mktime(strptime.strptime(datestring,\
            "%Y:%m:%d %H:%M:%S"))
        return date
    except ValueError:
        print "Unable to parse date: " + datestring
        return safe_value

def process_IFD(fields, info):
    for (tag,type,values) in fields:
        if tag==0x10F: # Make
            if type == 2:   # ASCII
                info.cammfg = string.capitalize(\
                    string.lower(\
                        string.split(values)[0]))
        elif tag==0x110: # Model
            if type == 2:   # ASCII
                info.cammod = values
                if info.cammod == 'C700UZ':
                    info.c700uz = 1
        elif tag==0x132: # DateTime
            if type == 2 and not info.datetimeorig:
                info.cdate = proc_date(values,info.cdate)
        elif tag==0x9003: # DateTimeOrig
            if type == 2:
                info.cdate = proc_date(values,info.cdate)
                info.datetimeorig = 1
        elif tag==0x829A: # ExposureTime
            if (type == 5 or type == 10) and info.c700uz:
                info.exptime = "1/" + \
                    repr(values[0].den/10) + "s"
            else:
                inf.exptime = repr(values[0]) + "s"
        elif tag==0x829D: # FNumber
            if (type == 5 or type == 10) and info.c700uz:
                info.aperture = "F%d.%d" %\
                    (values[0].num/10,values[0].num%10)
            else:
                info.aperture = "F" + repr(values[0])
        elif tag==0x8827: # ISOSpeedRatings
            info.iso = "ISO " + repr(values[0])
        elif tag==0x920A or tag==0x920D: # Zoom
            if (type == 5 or type == 10) and info.c700uz:
                info.zoom = \
                    repr((values[0].num*377+290)/580) +\
                    "mm"
            else:
                info.zoom = repr(values[0]) + "mm"
            pass

def append_tech(str,x,sep=", "):
    if len(str) == 0: return x
    if len(x) == 0: return str
    return str + sep + x

def process_file(file, info):
    info.datetimeorig = 0
    data = file.read(12)
    if data[0:4] <> '\377\330\377\341' or data[6:10] <> 'Exif':
        return

    info.c700uz = 0
    info.cammfg = ""
    info.cammod = ""
    info.exptime = ""
    info.aperture = ""
    info.iso = ""
    info.zoom = ""

    length = ord(data[4])*256 + ord(data[5])
    data = file.read(length-8)
    T = TIFF_file(data)
    L = T.list_IFDs()
    for i in range(len(L)):
        IFD = T.dump_IFD(L[i])
        process_IFD(IFD,info)
        exif_off = 0
        for tag,type,values in IFD:
            if tag == 0x8769:
                exif_off = values[0]
        if exif_off:
            IFD = T.dump_IFD(exif_off)
            process_IFD(IFD,info)
            intr_off = 0
            for tag,type,values in IFD:
                if tag == 0xA005:
                    intr_off = values[0]
            if intr_off:
                IFD = T.dump_IFD(intr_off)
                process_IFD(IFD, info)

    model = append_tech(info.cammfg,info.cammod,sep=" ")
    camparam = append_tech(info.exptime, info.aperture, " @ ")
    camset = append_tech(camparam, info.iso, ", ")
    info.tech = append_tech(model, camset, ", ")
    info.tech = append_tech(info.tech, info.zoom, ", ")
    if len(info.tech) == 0: info.tech = "None available"
    return
