/*
 * Decompiled with CFR 0.152.
 */
package libcore.io;

import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import libcore.io.ErrnoException;
import libcore.io.IoUtils;
import libcore.io.Libcore;
import libcore.io.OsConstants;
import libcore.io.Streams;

public final class DiskLruCache
implements Closeable {
    static final String JOURNAL_FILE = "journal";
    static final String JOURNAL_FILE_TMP = "journal.tmp";
    static final String MAGIC = "libcore.io.DiskLruCache";
    static final String VERSION_1 = "1";
    private static final String CLEAN = "CLEAN";
    private static final String DIRTY = "DIRTY";
    private static final String REMOVE = "REMOVE";
    private static final String READ = "READ";
    private final File directory;
    private final File journalFile;
    private final File journalFileTmp;
    private final int appVersion;
    private final long maxSize;
    private final int valueCount;
    private long size = 0L;
    private Writer journalWriter;
    private final LinkedHashMap<String, Entry> lruEntries = new LinkedHashMap(0, 0.75f, true);
    private int redundantOpCount;
    private final ExecutorService executorService = new ThreadPoolExecutor(0, 1, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    private final Callable<Void> cleanupCallable = new Callable<Void>(){

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Void call() throws Exception {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (DiskLruCache.this.journalWriter == null) {
                    return null;
                }
                DiskLruCache.this.trimToSize();
                if (DiskLruCache.this.journalRebuildRequired()) {
                    DiskLruCache.this.rebuildJournal();
                    DiskLruCache.this.redundantOpCount = 0;
                }
            }
            return null;
        }
    };

    private DiskLruCache(File directory, int appVersion, int valueCount, long maxSize) {
        this.directory = directory;
        this.appVersion = appVersion;
        this.journalFile = new File(directory, JOURNAL_FILE);
        this.journalFileTmp = new File(directory, JOURNAL_FILE_TMP);
        this.valueCount = valueCount;
        this.maxSize = maxSize;
    }

    public static DiskLruCache open(File directory, int appVersion, int valueCount, long maxSize) throws IOException {
        if (maxSize <= 0L) {
            throw new IllegalArgumentException("maxSize <= 0");
        }
        if (valueCount <= 0) {
            throw new IllegalArgumentException("valueCount <= 0");
        }
        DiskLruCache cache = new DiskLruCache(directory, appVersion, valueCount, maxSize);
        if (cache.journalFile.exists()) {
            try {
                cache.readJournal();
                cache.processJournal();
                cache.journalWriter = new BufferedWriter(new FileWriter(cache.journalFile, true));
                return cache;
            }
            catch (IOException journalIsCorrupt) {
                System.logW("DiskLruCache " + directory + " is corrupt: " + journalIsCorrupt.getMessage() + ", removing");
                cache.delete();
            }
        }
        directory.mkdirs();
        cache = new DiskLruCache(directory, appVersion, valueCount, maxSize);
        cache.rebuildJournal();
        return cache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readJournal() throws IOException {
        BufferedInputStream in = new BufferedInputStream(new FileInputStream(this.journalFile));
        try {
            String magic = Streams.readAsciiLine(in);
            String version = Streams.readAsciiLine(in);
            String appVersionString = Streams.readAsciiLine(in);
            String valueCountString = Streams.readAsciiLine(in);
            String blank = Streams.readAsciiLine(in);
            if (!(MAGIC.equals(magic) && VERSION_1.equals(version) && Integer.toString(this.appVersion).equals(appVersionString) && Integer.toString(this.valueCount).equals(valueCountString) && "".equals(blank))) {
                throw new IOException("unexpected journal header: [" + magic + ", " + version + ", " + valueCountString + ", " + blank + "]");
            }
            try {
                while (true) {
                    this.readJournalLine(Streams.readAsciiLine(in));
                }
            }
            catch (EOFException endOfJournal) {
                IoUtils.closeQuietly(in);
            }
        }
        catch (Throwable throwable) {
            IoUtils.closeQuietly(in);
            throw throwable;
        }
    }

    private void readJournalLine(String line) throws IOException {
        String[] parts = line.split(" ");
        if (parts.length < 2) {
            throw new IOException("unexpected journal line: " + line);
        }
        String key = parts[1];
        if (parts[0].equals(REMOVE) && parts.length == 2) {
            this.lruEntries.remove(key);
            return;
        }
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        }
        if (parts[0].equals(CLEAN) && parts.length == 2 + this.valueCount) {
            entry.readable = true;
            entry.currentEditor = null;
            entry.setLengths(Arrays.copyOfRange(parts, 2, parts.length));
        } else if (parts[0].equals(DIRTY) && parts.length == 2) {
            entry.currentEditor = new Editor(entry);
        } else if (!parts[0].equals(READ) || parts.length != 2) {
            throw new IOException("unexpected journal line: " + line);
        }
    }

    private void processJournal() throws IOException {
        DiskLruCache.deleteIfExists(this.journalFileTmp);
        Iterator i = this.lruEntries.values().iterator();
        while (i.hasNext()) {
            int t;
            Entry entry = (Entry)i.next();
            if (entry.currentEditor == null) {
                for (t = 0; t < this.valueCount; ++t) {
                    this.size += entry.lengths[t];
                }
                continue;
            }
            entry.currentEditor = null;
            for (t = 0; t < this.valueCount; ++t) {
                DiskLruCache.deleteIfExists(entry.getCleanFile(t));
                DiskLruCache.deleteIfExists(entry.getDirtyFile(t));
            }
            i.remove();
        }
    }

    private synchronized void rebuildJournal() throws IOException {
        if (this.journalWriter != null) {
            this.journalWriter.close();
        }
        BufferedWriter writer = new BufferedWriter(new FileWriter(this.journalFileTmp));
        writer.write(MAGIC);
        writer.write("\n");
        writer.write(VERSION_1);
        writer.write("\n");
        writer.write(Integer.toString(this.appVersion));
        writer.write("\n");
        writer.write(Integer.toString(this.valueCount));
        writer.write("\n");
        writer.write("\n");
        for (Entry entry : this.lruEntries.values()) {
            if (entry.currentEditor != null) {
                writer.write("DIRTY " + entry.key + '\n');
                continue;
            }
            writer.write("CLEAN " + entry.key + entry.getLengths() + '\n');
        }
        ((Writer)writer).close();
        this.journalFileTmp.renameTo(this.journalFile);
        this.journalWriter = new BufferedWriter(new FileWriter(this.journalFile, true));
    }

    private static void deleteIfExists(File file) throws IOException {
        block2: {
            try {
                Libcore.os.remove(file.getPath());
            }
            catch (ErrnoException errnoException) {
                if (errnoException.errno == OsConstants.ENOENT) break block2;
                throw errnoException.rethrowAsIOException();
            }
        }
    }

    public synchronized Snapshot get(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            return null;
        }
        if (!entry.readable) {
            return null;
        }
        InputStream[] ins = new InputStream[this.valueCount];
        try {
            for (int i = 0; i < this.valueCount; ++i) {
                ins[i] = new FileInputStream(entry.getCleanFile(i));
            }
        }
        catch (FileNotFoundException e) {
            return null;
        }
        ++this.redundantOpCount;
        this.journalWriter.append("READ " + key + '\n');
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return new Snapshot(ins);
    }

    public synchronized Editor edit(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        } else if (entry.currentEditor != null) {
            return null;
        }
        Editor editor = new Editor(entry);
        entry.currentEditor = editor;
        this.journalWriter.write("DIRTY " + key + '\n');
        this.journalWriter.flush();
        return editor;
    }

    public File getDirectory() {
        return this.directory;
    }

    public long maxSize() {
        return this.maxSize;
    }

    public synchronized long size() {
        return this.size;
    }

    private synchronized void completeEdit(Editor editor, boolean success) throws IOException {
        int i;
        Entry entry = editor.entry;
        if (entry.currentEditor != editor) {
            throw new IllegalStateException();
        }
        if (success && !entry.readable) {
            for (i = 0; i < this.valueCount; ++i) {
                if (entry.getDirtyFile(i).exists()) continue;
                editor.abort();
                throw new IllegalStateException("edit didn't create file " + i);
            }
        }
        for (i = 0; i < this.valueCount; ++i) {
            File dirty = entry.getDirtyFile(i);
            if (success) {
                long newLength;
                if (!dirty.exists()) continue;
                File clean = entry.getCleanFile(i);
                dirty.renameTo(clean);
                long oldLength = entry.lengths[i];
                ((Entry)entry).lengths[i] = newLength = clean.length();
                this.size = this.size - oldLength + newLength;
                continue;
            }
            DiskLruCache.deleteIfExists(dirty);
        }
        ++this.redundantOpCount;
        entry.currentEditor = null;
        if (entry.readable | success) {
            entry.readable = true;
            this.journalWriter.write("CLEAN " + entry.key + entry.getLengths() + '\n');
        } else {
            this.lruEntries.remove(entry.key);
            this.journalWriter.write("REMOVE " + entry.key + '\n');
        }
        if (this.size > this.maxSize || this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
    }

    private boolean journalRebuildRequired() {
        int REDUNDANT_OP_COMPACT_THRESHOLD = 2000;
        return this.redundantOpCount >= 2000 && this.redundantOpCount >= this.lruEntries.size();
    }

    public synchronized boolean remove(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null || entry.currentEditor != null) {
            return false;
        }
        for (int i = 0; i < this.valueCount; ++i) {
            File file = entry.getCleanFile(i);
            if (!file.delete()) {
                throw new IOException("failed to delete " + file);
            }
            this.size -= entry.lengths[i];
            ((Entry)entry).lengths[i] = 0L;
        }
        ++this.redundantOpCount;
        this.journalWriter.append("REMOVE " + key + '\n');
        this.lruEntries.remove(key);
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return true;
    }

    public boolean isClosed() {
        return this.journalWriter == null;
    }

    private void checkNotClosed() {
        if (this.journalWriter == null) {
            throw new IllegalStateException("cache is closed");
        }
    }

    public synchronized void flush() throws IOException {
        this.checkNotClosed();
        this.trimToSize();
        this.journalWriter.flush();
    }

    public synchronized void close() throws IOException {
        if (this.journalWriter == null) {
            return;
        }
        for (Entry entry : new ArrayList(this.lruEntries.values())) {
            if (entry.currentEditor == null) continue;
            entry.currentEditor.abort();
        }
        this.trimToSize();
        this.journalWriter.close();
        this.journalWriter = null;
    }

    private void trimToSize() throws IOException {
        while (this.size > this.maxSize) {
            Map.Entry<String, Entry> toEvict = this.lruEntries.eldest();
            this.remove(toEvict.getKey());
        }
    }

    public void delete() throws IOException {
        this.close();
        IoUtils.deleteContents(this.directory);
    }

    private void validateKey(String key) {
        if (key.contains(" ") || key.contains("\n") || key.contains("\r")) {
            throw new IllegalArgumentException("keys must not contain spaces or newlines: \"" + key + "\"");
        }
    }

    private static String inputStreamToString(InputStream in) throws IOException {
        return Streams.readFully(new InputStreamReader(in, Charsets.UTF_8));
    }

    private final class Entry {
        private final String key;
        private final long[] lengths;
        private boolean readable;
        private Editor currentEditor;

        private Entry(String key) {
            this.key = key;
            this.lengths = new long[DiskLruCache.this.valueCount];
        }

        public String getLengths() throws IOException {
            StringBuilder result = new StringBuilder();
            for (long size : this.lengths) {
                result.append(' ').append(size);
            }
            return result.toString();
        }

        private void setLengths(String[] strings) throws IOException {
            if (strings.length != DiskLruCache.this.valueCount) {
                throw this.invalidLengths(strings);
            }
            try {
                for (int i = 0; i < strings.length; ++i) {
                    this.lengths[i] = Long.parseLong(strings[i]);
                }
            }
            catch (NumberFormatException e) {
                throw this.invalidLengths(strings);
            }
        }

        private IOException invalidLengths(String[] strings) throws IOException {
            throw new IOException("unexpected journal line: " + Arrays.toString(strings));
        }

        public File getCleanFile(int i) {
            return new File(DiskLruCache.this.directory, this.key + "." + i);
        }

        public File getDirtyFile(int i) {
            return new File(DiskLruCache.this.directory, this.key + "." + i + ".tmp");
        }
    }

    public final class Editor {
        private final Entry entry;
        private boolean hasErrors;

        private Editor(Entry entry) {
            this.entry = entry;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public InputStream newInputStream(int index) throws IOException {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                if (!this.entry.readable) {
                    return null;
                }
                return new FileInputStream(this.entry.getCleanFile(index));
            }
        }

        public String getString(int index) throws IOException {
            InputStream in = this.newInputStream(index);
            return in != null ? DiskLruCache.inputStreamToString(in) : null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public OutputStream newOutputStream(int index) throws IOException {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                return new FaultHidingOutputStream(new FileOutputStream(this.entry.getDirtyFile(index)));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void set(int index, String value) throws IOException {
            OutputStreamWriter writer = null;
            try {
                writer = new OutputStreamWriter(this.newOutputStream(index), Charsets.UTF_8);
                writer.write(value);
            }
            catch (Throwable throwable) {
                IoUtils.closeQuietly(writer);
                throw throwable;
            }
            IoUtils.closeQuietly(writer);
        }

        public void commit() throws IOException {
            if (this.hasErrors) {
                DiskLruCache.this.completeEdit(this, false);
                DiskLruCache.this.remove(this.entry.key);
            } else {
                DiskLruCache.this.completeEdit(this, true);
            }
        }

        public void abort() throws IOException {
            DiskLruCache.this.completeEdit(this, false);
        }

        private class FaultHidingOutputStream
        extends FilterOutputStream {
            private FaultHidingOutputStream(OutputStream out) {
                super(out);
            }

            public void write(int oneByte) {
                try {
                    this.out.write(oneByte);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            public void write(byte[] buffer, int offset, int length) {
                try {
                    this.out.write(buffer, offset, length);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            public void close() {
                try {
                    this.out.close();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            public void flush() {
                try {
                    this.out.flush();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }
        }
    }

    public static final class Snapshot
    implements Closeable {
        private final InputStream[] ins;

        private Snapshot(InputStream[] ins) {
            this.ins = ins;
        }

        public InputStream getInputStream(int index) {
            return this.ins[index];
        }

        public String getString(int index) throws IOException {
            return DiskLruCache.inputStreamToString(this.getInputStream(index));
        }

        public void close() {
            for (InputStream in : this.ins) {
                IoUtils.closeQuietly(in);
            }
        }
    }
}

