# ScheduleMailingsForNewCustomer.sql
#
# ICS 184 / EECS 116 Summer 2006
# Code Example
#
# This stored procedure is an example of using cursors to work your way
# through the result of a query, one row at a time.  They're roughly
# equivalent to iterators from Java, with much less pleasant syntax.
# The intent of this procedure is to schedule "New customer" mailings
# to be sent to a newly-added customer.  Note that, in order for this
# procedure to apply to a new customer, it has to be called separately
# from inserting the customer.  (Another solution to this problem, which
# ties the work directly to the insertion of the new customer, will be
# presented in a future lecture and as a commented code example.)
#
# I've sprinkled comments throughout the procedure to clarify some issues
# along the way.
#
# This procedure can be called with a CALL statement in SQL, an example of
# which is:
#
#     CALL schedule_mailings_for_new_customer(101);
#
# The DELIMITER statements are necessary only so that MySQL is not confused,
# by the presence of semicolons, into thinking that the procedure definition
# has ended prematurely.


DELIMITER //

CREATE PROCEDURE schedule_mailings_for_new_customer(cid INTEGER)
BEGIN
    DECLARE mid INTEGER;
    DECLARE mstate CHAR(2);

	-- The DEFAULT clause gives a variable an initial value.
    DECLARE done INTEGER DEFAULT 0;

    DECLARE cstate CHAR(2);

	-- Using a cursor requires a few steps:
	--   (1) Declaring the cursor.  (That's being done here.)
	--   (2) Opening the cursor.
	--   (3) Fetching rows from the cursor, one at a time, until there
	--       are no more rows.
	--   (4) Closing the cursor.

    DECLARE m CURSOR FOR SELECT mailing_id, mailing_state
                         FROM mailing
                         WHERE mailing_type = 'NEW';


	-- This is similar to a catch block from Java.  It says "When an
	-- error with code '02000' occurs, set the done variable to 1, then
	-- continue the procedure as if nothing was wrong."  The code
	-- '02000' refers to the error that occurs when you try to fetch
	-- rows after the cursor has already returned all of them.

	DECLARE CONTINUE HANDLER FOR SQLSTATE '02000' SET done = 1;

    SELECT customer_state INTO cstate
    FROM customer
    WHERE customer_id = cid;

    OPEN m;
    
    REPEAT
    	-- Each step of the way, we fetch the rows from the query in the
    	-- cursor's declaration into variables; we can then use the
    	-- variables normally.
        FETCH m INTO mid, mstate;

        IF NOT done THEN
            IF mstate = cstate THEN

                INSERT INTO scheduled_mailing
                VALUES (cid, mid, CURDATE());

            END IF;
        END IF;
    UNTIL done END REPEAT;

	-- We can't forget to close the cursor!
    CLOSE m;
END;//

DELIMITER ;
