// SmartIntPtr.hpp
//
// ICS 45C Spring 2022
// Code Example
//
// A "smart pointer" that implements unique ownership of a dynamically-
// allocated integer.  By "unique ownership", I mean that it is assumed
// that the destruction of the smart pointer should also cause the
// automatic destruction of the dynamically-allocated integer.
//
// Note that this is a somewhat simplified version of a smart pointer
// built into the C++ Standard Library called std::unique_ptr.

#ifndef SMARTINTPTR_HPP
#define SMARTINTPTR_HPP



class SmartIntPtr
{
public:
    // When we construct a SmartIntPtr, we give it a pointer to a
    // dynamically-allocated integer.  Doing this, we've said "Please
    // take unique ownership of this integer."
    SmartIntPtr(int* p)
        : p{p}
    {
    }


    // When a SmartIntPtr dies, it destroys the integer that it manages.
    ~SmartIntPtr()
    {
        delete p;
    }


    // This is how we make copy-construction and copy-assignment of
    // SmartIntPtrs illegal.  The syntax declares the functions to
    // be "deleted" (so that we don't get the versions that are
    // generated by the compiler automatically).  Of course, since
    // they're deleted, we don't have to supply implementations.
    SmartIntPtr(const SmartIntPtr& p) = delete;
    SmartIntPtr& operator=(const SmartIntPtr& p) = delete;


    // get() returns the pointer stored within this SmartIntPtr.  You
    // might need to be able to get that pointer if you wanted to pass
    // it as a parameter expecting a pointer that wasn't "smart".
    // (This would imply a sort of "loan" of the integer, as opposed
    // to something that implies ownership.)
    int* get() const
    {
        return p;
    }


    // This is how we overload the (unary) * operator, so we can
    // dereference one of our SmartIntPtrs.  Conceptually, it's a lot
    // like overloading the = operator, except that the return type is
    // different: Here, we're returning a reference to the int object
    // that the smart pointer refers to, which would allow someone to
    // do something like this with a SmartIntPtr called ptr:
    //
    //     *ptr = 5;
    int& operator*() const
    {
        return *p;
    }


private:
    int* p;
};



#endif

